"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const errors_1 = require("../../lib/errors");
const base_1 = require("./base");
class SSHUseCommand extends base_1.SSHBaseCommand {
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                name: 'use',
                type: 'global',
                summary: 'Set your active Ionic SSH key',
                description: `
This command modifies the SSH configuration file (${chalk_1.default.bold('~/.ssh/config')}) to set an active private key for the ${chalk_1.default.bold('git.ionicjs.com')} host. Read more about SSH configuration by running the ${chalk_1.default.green('man ssh_config')} command or by visiting online man pages[^ssh-config-docs].

Before making changes, ${chalk_1.default.green('ionic ssh use')} will print a diff and ask for permission to write the file.
      `,
                footnotes: [
                    {
                        id: 'ssh-config-docs',
                        url: 'https://linux.die.net/man/5/ssh_config',
                    },
                ],
                inputs: [
                    {
                        name: 'key-path',
                        summary: 'Location of private key file to use',
                        validators: [cli_framework_1.validators.required],
                    },
                ],
            };
        });
    }
    run(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { ERROR_SSH_INVALID_PRIVKEY, ERROR_SSH_MISSING_PRIVKEY, validatePrivateKey } = yield Promise.resolve().then(() => require('../../lib/ssh'));
            const { ensureHostAndKeyPath, getConfigPath } = yield Promise.resolve().then(() => require('../../lib/ssh-config'));
            const keyPath = format_1.expandPath(inputs[0]);
            try {
                yield validatePrivateKey(keyPath);
            }
            catch (e) {
                if (e === ERROR_SSH_MISSING_PRIVKEY) {
                    throw new errors_1.FatalException(`${chalk_1.default.bold(format_1.prettyPath(keyPath))} does not appear to exist. Please specify a valid SSH private key.\n` +
                        `If you are having issues, try using ${chalk_1.default.green('ionic ssh setup')}.`);
                }
                else if (e === ERROR_SSH_INVALID_PRIVKEY) {
                    throw new errors_1.FatalException(`${chalk_1.default.bold(format_1.prettyPath(keyPath))} does not appear to be a valid SSH private key. (Missing '-----BEGIN RSA PRIVATE KEY-----' header.)\n` +
                        `If you are having issues, try using ${chalk_1.default.green('ionic ssh setup')}.`);
                }
                else {
                    throw e;
                }
            }
            const { SSHConfig } = yield Promise.resolve().then(() => require('../../lib/ssh-config'));
            const sshConfigPath = getConfigPath();
            const text1 = yield utils_fs_1.fileToString(sshConfigPath);
            const conf = SSHConfig.parse(text1);
            ensureHostAndKeyPath(conf, { host: this.env.config.getGitHost(), port: this.env.config.getGitPort() }, keyPath);
            const text2 = SSHConfig.stringify(conf);
            if (text1 === text2) {
                this.env.log.msg(`${chalk_1.default.bold(format_1.prettyPath(keyPath))} is already your active SSH key.`);
                return;
            }
            else {
                const { diffPatch } = yield Promise.resolve().then(() => require('../../lib/diff'));
                const diff = yield diffPatch(sshConfigPath, text1, text2);
                this.env.log.rawmsg(diff);
                const confirm = yield this.env.prompt({
                    type: 'confirm',
                    name: 'confirm',
                    message: `May we make the above change(s) to '${format_1.prettyPath(sshConfigPath)}'?`,
                });
                if (!confirm) {
                    // TODO: link to docs about manual git setup
                    throw new errors_1.FatalException();
                }
            }
            yield utils_fs_1.writeFile(sshConfigPath, text2, { encoding: 'utf8', mode: 0o600 });
            this.env.log.ok(`Your active Ionic SSH key has been set to ${chalk_1.default.bold(keyPath)}!`);
        });
    }
}
exports.SSHUseCommand = SSHUseCommand;
