"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const string_1 = require("@ionic/cli-framework/utils/string");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const Debug = require("debug");
const lodash = require("lodash");
const path = require("path");
const constants_1 = require("../constants");
const command_1 = require("../lib/command");
const errors_1 = require("../lib/errors");
const executor_1 = require("../lib/executor");
const project_1 = require("../lib/project");
const shell_1 = require("../lib/shell");
const emoji_1 = require("../lib/utils/emoji");
const debug = Debug('ionic:commands:start');
class StartCommand extends command_1.Command {
    constructor() {
        super(...arguments);
        this.canRemoveExisting = false;
    }
    getMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const starterTemplates = yield this.getStarterTemplates();
            return {
                name: 'start',
                type: 'global',
                summary: 'Create a new project',
                description: `
This command creates a working Ionic app. It installs dependencies for you and sets up your project.

Running ${chalk_1.default.green('ionic start')} without any arguments will prompt you for information about your new project.

The first argument is your app's ${chalk_1.default.green('name')}. Don't worry--you can always change this later. The ${chalk_1.default.green('--project-id')} is generated from ${chalk_1.default.green('name')} unless explicitly specified.

The second argument is the ${chalk_1.default.green('template')} from which to generate your app. You can list all templates with the ${chalk_1.default.green('--list')} option. You can also specify a git repository URL for ${chalk_1.default.green('template')}, in which case the existing project will be cloned.
      `,
                exampleCommands: [
                    '',
                    '--list',
                    'myApp',
                    'myApp blank',
                    'myApp tabs --cordova',
                    'myApp tabs --capacitor',
                    'myApp super --type=angular',
                    'myApp blank --type=ionic1',
                    'cordovaApp tabs --cordova',
                    '"My App" blank',
                    '"Conference App" https://github.com/ionic-team/ionic-conference-app',
                ],
                inputs: [
                    {
                        name: 'name',
                        summary: `The name of your new project (e.g. ${chalk_1.default.green('myApp')}, ${chalk_1.default.green('"My App"')})`,
                        validators: [cli_framework_1.validators.required],
                    },
                    {
                        name: 'template',
                        summary: `The starter template to use (e.g. ${['blank', 'tabs'].map(t => chalk_1.default.green(t)).join(', ')}; use ${chalk_1.default.green('--list')} to see all)`,
                        validators: [cli_framework_1.validators.required],
                    },
                ],
                options: [
                    {
                        name: 'list',
                        summary: 'List available starter templates',
                        type: Boolean,
                        aliases: ['l'],
                    },
                    {
                        name: 'type',
                        summary: `Type of project to start (e.g. ${lodash.uniq(starterTemplates.map(t => t.type)).map(type => chalk_1.default.green(type)).join(', ')})`,
                        type: String,
                    },
                    {
                        name: 'cordova',
                        summary: 'Include Cordova integration',
                        type: Boolean,
                    },
                    {
                        name: 'capacitor',
                        summary: 'Include Capacitor integration',
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Experimental],
                    },
                    {
                        name: 'deps',
                        summary: 'Do not install npm/yarn dependencies',
                        type: Boolean,
                        default: true,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'git',
                        summary: 'Do not initialize a git repo',
                        type: Boolean,
                        default: true,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'link',
                        summary: 'Do not ask to connect the app to Ionic Appflow',
                        type: Boolean,
                        default: true,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'id',
                        summary: 'Specify an app ID from the Ionic Appflow to link',
                    },
                    {
                        name: 'pro-id',
                        summary: `Use the ${chalk_1.default.green('--id')} option`,
                        groups: [cli_framework_1.OptionGroup.Deprecated],
                        spec: { value: 'id' },
                    },
                    {
                        name: 'project-id',
                        summary: 'Specify a slug for your app (used for the directory name and npm/yarn package name)',
                        groups: [cli_framework_1.OptionGroup.Advanced],
                        spec: { value: 'slug' },
                    },
                    {
                        name: 'package-id',
                        summary: 'Specify the bundle ID/application ID for your app (reverse-DNS notation)',
                        groups: [cli_framework_1.OptionGroup.Advanced],
                        spec: { value: 'id' },
                    },
                    {
                        name: 'tag',
                        summary: `Specify a tag to use for the starters (e.g. ${['latest', 'testing', 'next'].map(t => chalk_1.default.green(t)).join(', ')})`,
                        default: 'latest',
                        groups: [cli_framework_1.OptionGroup.Hidden],
                    },
                ],
            };
        });
    }
    preRun(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { promptToLogin } = yield Promise.resolve().then(() => require('../lib/session'));
            const starterTemplates = yield this.getStarterTemplates();
            const cloned = string_1.isValidURL(inputs[1]);
            // If the action is list then lets just end here.
            if (options['list']) {
                const headers = ['name', 'project type', 'description'];
                this.env.log.rawmsg(format_1.columnar(starterTemplates.map(({ name, type, description }) => [chalk_1.default.green(name), chalk_1.default.bold(type), description || '']), { headers }));
                throw new errors_1.FatalException('', 0);
            }
            if (options['skip-deps']) {
                this.env.log.warn(`The ${chalk_1.default.green('--skip-deps')} option has been deprecated. Please use ${chalk_1.default.green('--no-deps')}.`);
                options['deps'] = false;
            }
            if (options['skip-link']) {
                this.env.log.warn(`The ${chalk_1.default.green('--skip-link')} option has been deprecated. Please use ${chalk_1.default.green('--no-link')}.`);
                options['link'] = false;
            }
            if (options['pro-id']) {
                this.env.log.warn(`The ${chalk_1.default.green('--pro-id')} option has been deprecated. Please use ${chalk_1.default.green('--id')}.`);
                options['id'] = options['pro-id'];
            }
            if (options['id']) {
                if (!options['link']) {
                    this.env.log.warn(`The ${chalk_1.default.green('--no-link')} option has no effect with ${chalk_1.default.green('--id')}. App must be linked.`);
                }
                options['link'] = true;
                if (!options['git']) {
                    this.env.log.warn(`The ${chalk_1.default.green('--no-git')} option has no effect with ${chalk_1.default.green('--id')}. Git must be used.`);
                }
                options['git'] = true;
            }
            if (cloned) {
                if (!options['git']) {
                    this.env.log.warn(`The ${chalk_1.default.green('--no-git')} option has no effect when cloning apps. Git must be used.`);
                }
                options['git'] = true;
            }
            if (this.project && this.project.details.context === 'app') {
                const confirm = yield this.env.prompt({
                    type: 'confirm',
                    name: 'confirm',
                    message: 'You are already in an Ionic project directory. Do you really want to start another project here?',
                    default: false,
                });
                if (!confirm) {
                    this.env.log.info('Not starting project within existing project.');
                    throw new errors_1.FatalException();
                }
            }
            const projectType = options['type'] ? String(options['type']) : 'angular';
            const appflowId = options['id'] ? String(options['id']) : undefined;
            yield this.validateProjectType(projectType);
            if (options['v1'] || options['v2']) {
                throw new errors_1.FatalException(`The ${chalk_1.default.green('--v1')} and ${chalk_1.default.green('--v2')} flags have been removed.\n` +
                    `Use the ${chalk_1.default.green('--type')} option. (see ${chalk_1.default.green('ionic start --help')})`);
            }
            if (options['app-name']) {
                this.env.log.warn(`The ${chalk_1.default.green('--app-name')} option has been removed. Use the ${chalk_1.default.green('name')} argument with double quotes: e.g. ${chalk_1.default.green('ionic start "My App"')}`);
            }
            if (options['display-name']) {
                this.env.log.warn(`The ${chalk_1.default.green('--display-name')} option has been removed. Use the ${chalk_1.default.green('name')} argument with double quotes: e.g. ${chalk_1.default.green('ionic start "My App"')}`);
            }
            if (options['bundle-id']) {
                this.env.log.warn(`The ${chalk_1.default.green('--bundle-id')} option has been deprecated. Please use ${chalk_1.default.green('--package-id')}.`);
                options['package-id'] = options['bundle-id'];
            }
            if (appflowId) {
                if (!this.env.session.isLoggedIn()) {
                    yield promptToLogin(this.env);
                }
            }
            if (!inputs[0]) {
                if (appflowId) {
                    const { AppClient } = yield Promise.resolve().then(() => require('../lib/app'));
                    const token = this.env.session.getUserToken();
                    const appClient = new AppClient(token, this.env);
                    const tasks = this.createTaskChain();
                    tasks.next(`Looking up app ${chalk_1.default.green(appflowId)}`);
                    const app = yield appClient.load(appflowId);
                    // TODO: can ask to clone via repo_url
                    tasks.end();
                    this.env.log.info(`Using ${chalk_1.default.bold(app.name)} for ${chalk_1.default.green('name')} and ${chalk_1.default.bold(app.slug)} for ${chalk_1.default.green('--project-id')}.`);
                    inputs[0] = app.name;
                    options['project-id'] = app.slug;
                }
                else {
                    if (this.env.flags.interactive) {
                        this.env.log.nl();
                        this.env.log.msg(`${chalk_1.default.bold(`Every great app needs a name! ${emoji_1.emoji('😍', '')}`)}\n` +
                            `Please enter the full name of your app. You can change this at any time. To bypass this prompt next time, supply ${chalk_1.default.green('name')}, the first argument to ${chalk_1.default.green('ionic start')}.\n\n`);
                    }
                    const name = yield this.env.prompt({
                        type: 'input',
                        name: 'name',
                        message: 'Project name:',
                        validate: v => cli_framework_1.validators.required(v),
                    });
                    inputs[0] = name;
                }
            }
            let projectId = options['project-id'] ? String(options['project-id']) : undefined;
            if (projectId) {
                yield this.validateProjectId(projectId);
            }
            else {
                projectId = options['project-id'] = project_1.isValidProjectId(inputs[0]) ? inputs[0] : string_1.slugify(inputs[0]);
            }
            const projectDir = path.resolve(projectId);
            const packageId = options['package-id'] ? String(options['package-id']) : undefined;
            if (projectId) {
                yield this.checkForExisting(projectDir);
            }
            if (cloned) {
                this.schema = {
                    cloned,
                    url: inputs[1],
                    projectId,
                    projectDir,
                };
            }
            else {
                if (!inputs[1]) {
                    if (this.env.flags.interactive) {
                        this.env.log.nl();
                        this.env.log.msg(`${chalk_1.default.bold(`Let's pick the perfect starter template! ${emoji_1.emoji('💪', '')}`)}\n` +
                            `Starter templates are ready-to-go Ionic apps that come packed with everything you need to build your app. To bypass this prompt next time, supply ${chalk_1.default.green('template')}, the second argument to ${chalk_1.default.green('ionic start')}.\n\n`);
                    }
                    const template = yield this.env.prompt({
                        type: 'list',
                        name: 'template',
                        message: 'Starter template:',
                        choices: () => {
                            const starterTemplateList = starterTemplates.filter(st => st.type === projectType);
                            const cols = format_1.columnar(starterTemplateList.map(({ name, description }) => [chalk_1.default.green(name), description || '']), {}).split('\n');
                            if (starterTemplateList.length === 0) {
                                throw new errors_1.FatalException(`No starter templates found for project type: ${chalk_1.default.green(projectType)}.`);
                            }
                            return starterTemplateList.map((starterTemplate, i) => {
                                return {
                                    name: cols[i],
                                    short: starterTemplate.name,
                                    value: starterTemplate.name,
                                };
                            });
                        },
                    });
                    inputs[1] = template;
                }
                this.schema = {
                    cloned,
                    name: inputs[0],
                    type: projectType,
                    template: inputs[1],
                    projectId,
                    projectDir,
                    packageId,
                    appflowId,
                };
            }
        });
    }
    run(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { pkgManagerArgs } = yield Promise.resolve().then(() => require('../lib/utils/npm'));
            const { getTopLevel, isGitInstalled } = yield Promise.resolve().then(() => require('../lib/git'));
            const { getIonicDevAppText, getIonicProText } = yield Promise.resolve().then(() => require('../lib/start'));
            if (!this.schema) {
                throw new errors_1.FatalException(`Invalid information: cannot start app.`);
            }
            const { projectId, projectDir, packageId, appflowId } = this.schema;
            const tag = options['tag'] ? String(options['tag']) : 'latest';
            let linkConfirmed = typeof appflowId === 'string';
            const gitDesired = options['git'] ? true : false;
            const gitInstalled = yield isGitInstalled(this.env);
            const gitTopLevel = yield getTopLevel(this.env);
            let gitIntegration = gitDesired && gitInstalled && !gitTopLevel ? true : false;
            if (!gitInstalled) {
                const installationDocs = `See installation docs for git: ${chalk_1.default.bold('https://git-scm.com/book/en/v2/Getting-Started-Installing-Git')}`;
                if (appflowId) {
                    throw new errors_1.FatalException(`Git CLI not found on your PATH.\n` +
                        `Git must be installed to connect this app to Ionic. ${installationDocs}`);
                }
                if (this.schema.cloned) {
                    throw new errors_1.FatalException(`Git CLI not found on your PATH.\n` +
                        `Git must be installed to clone apps with ${chalk_1.default.green('ionic start')}. ${installationDocs}`);
                }
            }
            if (gitTopLevel && !this.schema.cloned) {
                this.env.log.info(`Existing git project found (${chalk_1.default.bold(gitTopLevel)}). Git operations are disabled.`);
            }
            const tasks = this.createTaskChain();
            tasks.next(`Preparing directory ${chalk_1.default.green(format_1.prettyPath(projectDir))}`);
            if (this.canRemoveExisting) {
                yield utils_fs_1.remove(projectDir);
            }
            yield utils_fs_1.mkdir(projectDir);
            tasks.end();
            if (this.schema.cloned) {
                yield this.env.shell.run('git', ['clone', this.schema.url, projectDir, '--progress'], { stdio: 'inherit' });
            }
            else {
                const starterTemplate = yield this.findStarterTemplate(this.schema.template, this.schema.type, tag);
                yield this.downloadStarterTemplate(projectDir, starterTemplate);
            }
            let project;
            if (this.project && this.project.details.context === 'multiapp' && !this.schema.cloned) {
                // We're in a multi-app setup, so the new config file isn't wanted.
                yield utils_fs_1.unlink(path.resolve(projectDir, 'ionic.config.json'));
                project = yield project_1.createProjectFromDetails({ context: 'multiapp', configPath: path.resolve(this.project.rootDirectory, constants_1.PROJECT_FILE), id: projectId, type: this.schema.type, errors: [] }, this.env);
                project.config.set('type', this.schema.type);
                project.config.set('root', path.relative(this.project.rootDirectory, projectDir));
            }
            else {
                project = yield project_1.createProjectFromDirectory(projectDir, { _: [] }, this.env, { logErrors: false });
            }
            // start is weird, once the project directory is created, it becomes a
            // "project" command and so we replace the `Project` instance that was
            // autogenerated when the CLI booted up. This has worked thus far?
            this.namespace.root.project = project;
            if (!this.project) {
                throw new errors_1.FatalException('Error while loading project.');
            }
            this.env.shell.alterPath = p => shell_1.prependNodeModulesBinToPath(projectDir, p);
            if (!this.schema.cloned) {
                if (typeof options['cordova'] === 'undefined' && !options['capacitor']) {
                    const confirm = yield this.env.prompt({
                        type: 'confirm',
                        name: 'confirm',
                        message: 'Integrate your new app with Cordova to target native iOS and Android?',
                        default: false,
                    });
                    if (confirm) {
                        options['cordova'] = true;
                    }
                }
                if (options['cordova']) {
                    yield executor_1.runCommand(runinfo, ['integrations', 'enable', 'cordova', '--quiet']);
                }
                if (options['capacitor']) {
                    yield executor_1.runCommand(runinfo, ['integrations', 'enable', 'capacitor', '--quiet', '--', this.schema.name, packageId ? packageId : 'io.ionic.starter']);
                }
                yield this.project.personalize({ name: this.schema.name, projectId, packageId });
                this.env.log.nl();
            }
            const shellOptions = { cwd: projectDir, stdio: 'inherit' };
            if (options['deps']) {
                this.env.log.msg('Installing dependencies may take several minutes.');
                this.env.log.rawmsg(yield getIonicDevAppText());
                const [installer, ...installerArgs] = yield pkgManagerArgs(this.env.config.get('npmClient'), { command: 'install' });
                yield this.env.shell.run(installer, installerArgs, shellOptions);
            }
            if (!this.schema.cloned) {
                if (gitIntegration) {
                    try {
                        yield this.env.shell.run('git', ['init'], shellOptions); // TODO: use initializeRepo()?
                    }
                    catch (e) {
                        this.env.log.warn('Error encountered during repo initialization. Disabling further git operations.');
                        gitIntegration = false;
                    }
                }
                if (options['link'] && !linkConfirmed) {
                    this.env.log.rawmsg(yield getIonicProText());
                    const confirm = yield this.env.prompt({
                        type: 'confirm',
                        name: 'confirm',
                        message: 'Install the free Ionic Appflow SDK and connect your app?',
                        fallback: false,
                        default: true,
                    });
                    if (confirm) {
                        linkConfirmed = true;
                    }
                }
                if (linkConfirmed) {
                    const [installer, ...installerArgs] = yield pkgManagerArgs(this.env.config.get('npmClient'), { command: 'install', pkg: '@ionic/pro' });
                    yield this.env.shell.run(installer, installerArgs, shellOptions);
                    const cmdArgs = ['link'];
                    if (appflowId) {
                        cmdArgs.push(appflowId);
                    }
                    cmdArgs.push('--name', this.schema.name);
                    yield executor_1.runCommand(runinfo, cmdArgs);
                }
                const manifestPath = path.resolve(projectDir, 'ionic.starter.json');
                const manifest = yield this.loadManifest(manifestPath);
                if (manifest) {
                    yield utils_fs_1.unlink(manifestPath);
                }
                if (gitIntegration) {
                    try {
                        yield this.env.shell.run('git', ['add', '-A'], shellOptions);
                        yield this.env.shell.run('git', ['commit', '-m', 'Initial commit', '--no-gpg-sign'], shellOptions);
                    }
                    catch (e) {
                        this.env.log.warn('Error encountered during commit. Disabling further git operations.');
                        gitIntegration = false;
                    }
                }
                if (manifest) {
                    yield this.performManifestOps(manifest);
                }
            }
            this.env.log.nl();
            yield this.showNextSteps(projectDir, this.schema.cloned, linkConfirmed);
        });
    }
    getStarterTemplates() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { STARTER_TEMPLATES } = yield Promise.resolve().then(() => require('../lib/start'));
            return STARTER_TEMPLATES;
        });
    }
    getStarterProjectTypes() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const starterTemplates = yield this.getStarterTemplates();
            return lodash.uniq(starterTemplates.map(t => t.type));
        });
    }
    checkForExisting(projectDir) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const projectExists = yield utils_fs_1.pathExists(projectDir);
            if (projectExists) {
                const confirm = yield this.env.prompt({
                    type: 'confirm',
                    name: 'confirm',
                    message: `${chalk_1.default.green(format_1.prettyPath(projectDir))} exists. ${chalk_1.default.red('Overwrite?')}`,
                    default: false,
                });
                if (!confirm) {
                    this.env.log.msg(`Not erasing existing project in ${chalk_1.default.green(format_1.prettyPath(projectDir))}.`);
                    throw new errors_1.FatalException();
                }
                this.canRemoveExisting = confirm;
            }
        });
    }
    findStarterTemplate(template, type, tag) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { STARTER_BASE_URL, getStarterList } = yield Promise.resolve().then(() => require('../lib/start'));
            const starterTemplates = yield this.getStarterTemplates();
            const starterTemplate = starterTemplates.find(t => t.type === type && t.name === template);
            if (starterTemplate) {
                return Object.assign({}, starterTemplate, { archive: `${STARTER_BASE_URL}/${tag === 'latest' ? '' : `${tag}/`}${starterTemplate.id}.tar.gz` });
            }
            const tasks = this.createTaskChain();
            tasks.next('Looking up starter');
            const starterList = yield getStarterList(this.env.config, tag);
            const starter = starterList.starters.find(t => t.type === type && t.name === template);
            if (starter) {
                tasks.end();
                return Object.assign({}, starter, { archive: `${STARTER_BASE_URL}/${tag === 'latest' ? '' : `${tag}/`}${starter.id}.tar.gz` });
            }
            else {
                throw new errors_1.FatalException(`Unable to find starter template for ${chalk_1.default.green(template)}\n` +
                    `If this is not a typo, please make sure it is a valid starter template within the starters repo: ${chalk_1.default.bold('https://github.com/ionic-team/starters')}`);
            }
        });
    }
    validateProjectType(type) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const projectTypes = yield this.getStarterProjectTypes();
            if (!projectTypes.includes(type)) {
                throw new errors_1.FatalException(`${chalk_1.default.green(type)} is not a valid project type.\n` +
                    `Please choose a different ${chalk_1.default.green('--type')}. Use ${chalk_1.default.green('ionic start --list')} to list all available starter templates.`);
            }
        });
    }
    validateProjectId(projectId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!project_1.isValidProjectId(projectId)) {
                throw new errors_1.FatalException(`${chalk_1.default.green(projectId)} is not a valid package or directory name.\n` +
                    `Please choose a different ${chalk_1.default.green('--project-id')}. Alphanumeric characters are always safe.`);
            }
        });
    }
    loadManifest(manifestPath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { readStarterManifest } = yield Promise.resolve().then(() => require('../lib/start'));
            try {
                return yield readStarterManifest(manifestPath);
            }
            catch (e) {
                debug(`Error with manifest file ${chalk_1.default.bold(format_1.prettyPath(manifestPath))}: ${e}`);
            }
        });
    }
    performManifestOps(manifest) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (manifest.welcome) {
                this.env.log.nl();
                this.env.log.msg(`${chalk_1.default.bold('Starter Welcome')}:`);
                this.env.log.msg(manifest.welcome);
            }
        });
    }
    downloadStarterTemplate(projectDir, starterTemplate) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { createRequest, download } = yield Promise.resolve().then(() => require('../lib/utils/http'));
            const { tar } = yield Promise.resolve().then(() => require('../lib/utils/archive'));
            const tasks = this.createTaskChain();
            const task = tasks.next(`Downloading and extracting ${chalk_1.default.green(starterTemplate.name.toString())} starter`);
            debug('Tar extraction created for %s', projectDir);
            const ws = tar.extract({ cwd: projectDir });
            const { req } = yield createRequest('GET', starterTemplate.archive, this.env.config.getHTTPConfig());
            yield download(req, ws, { progress: (loaded, total) => task.progress(loaded, total) });
            tasks.end();
        });
    }
    showNextSteps(projectDir, cloned, linkConfirmed) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const steps = [
                `Go to your ${cloned ? 'cloned' : 'newly created'} project: ${chalk_1.default.green(`cd ${format_1.prettyPath(projectDir)}`)}`,
                `Run ${chalk_1.default.green('ionic serve')} within the app directory to see your app`,
                `Build features and components: ${chalk_1.default.bold('https://ion.link/scaffolding-docs')}`,
                `Get Ionic DevApp for easy device testing: ${chalk_1.default.bold('https://ion.link/devapp')}`,
            ];
            if (linkConfirmed) {
                steps.push(`Push your code to Ionic Appflow to perform real-time updates, and more: ${chalk_1.default.green('git push ionic master')}`);
            }
            this.env.log.info(`${chalk_1.default.bold('Next Steps')}:\n${steps.map(s => `- ${s}`).join('\n')}`);
        });
    }
}
exports.StartCommand = StartCommand;
