"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const node_1 = require("@ionic/cli-framework/utils/node");
const process_1 = require("@ionic/cli-framework/utils/process");
const chalk_1 = require("chalk");
const Debug = require("debug");
const path = require("path");
const commands_1 = require("./commands");
const guards_1 = require("./guards");
const lib_1 = require("./lib");
const executor_1 = require("./lib/executor");
const init_1 = require("./lib/init");
tslib_1.__exportStar(require("./constants"), exports);
tslib_1.__exportStar(require("./guards"), exports);
const debug = Debug('ionic');
const PACKAGE_ROOT_PATH = __dirname;
const PACKAGE_JSON_PATH = path.resolve(PACKAGE_ROOT_PATH, 'package.json');
let _pkg;
let _executor;
function loadPackageJson() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (!_pkg) {
            _pkg = yield node_1.readPackageJsonFile(PACKAGE_JSON_PATH);
        }
        return _pkg;
    });
}
function generateContext() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const pkg = yield loadPackageJson();
        if (!pkg.bin || !pkg.bin.ionic) {
            throw new Error(`Missing "${chalk_1.default.bold('bin.ionic')}" in Ionic CLI package.json`);
        }
        if (!pkg.main) {
            throw new Error(`Missing "${chalk_1.default.bold('main')}" in Ionic CLI package.json`);
        }
        return {
            binPath: path.resolve(PACKAGE_ROOT_PATH, pkg.bin.ionic),
            libPath: PACKAGE_ROOT_PATH,
            execPath: process.cwd(),
            version: pkg.version,
        };
    });
}
exports.generateContext = generateContext;
function loadExecutor(ctx, pargv) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (!_executor) {
            const deps = yield lib_1.generateIonicEnvironment(ctx, pargv);
            const namespace = new commands_1.IonicNamespace(deps);
            _executor = new executor_1.Executor({ namespace });
        }
        return _executor;
    });
}
exports.loadExecutor = loadExecutor;
function run(pargv) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        let err;
        let executor;
        try {
            executor = yield loadExecutor(yield generateContext(), pargv);
        }
        catch (e) {
            process.stderr.write(`${e.message ? e.message : (e.stack ? e.stack : e)}\n`);
            process.exitCode = 1;
            return;
        }
        const ienv = executor.namespace.env;
        if (pargv[0] !== '_') {
            try {
                debug('Context: %o', ienv.ctx);
                ienv.config.set('version', ienv.ctx.version);
                const token = process.env['IONIC_TOKEN'];
                const email = process.env['IONIC_EMAIL'];
                const password = process.env['IONIC_PASSWORD'];
                if (token) {
                    const wasLoggedIn = ienv.session.isLoggedIn();
                    debug(`${chalk_1.default.bold('IONIC_TOKEN')} environment variable detected`);
                    if (ienv.config.get('tokens.user') !== token) {
                        debug(`${chalk_1.default.bold('IONIC_TOKEN')} mismatch with current session--attempting login`);
                        yield ienv.session.tokenLogin(token);
                        if (wasLoggedIn) {
                            ienv.log.info(`You have been logged out--using ${chalk_1.default.bold('IONIC_TOKEN')} environment variable`);
                        }
                    }
                }
                else if (email && password) {
                    debug(`${chalk_1.default.bold('IONIC_EMAIL')} / ${chalk_1.default.bold('IONIC_PASSWORD')} environment variables detected`);
                    if (ienv.config.get('user.email') !== email) {
                        debug(`${chalk_1.default.bold('IONIC_EMAIL')} mismatch with current session--attempting login`);
                        try {
                            yield ienv.session.login(email, password);
                        }
                        catch (e) {
                            ienv.log.error(`Error occurred during automatic login via ${chalk_1.default.bold('IONIC_EMAIL')} / ${chalk_1.default.bold('IONIC_PASSWORD')} environment variables.`);
                            throw e;
                        }
                    }
                }
                const parsedArgs = cli_framework_1.stripOptions(pargv, { includeSeparated: false });
                const foundCommand = init_1.mapLegacyCommand(parsedArgs[0]);
                // If an legacy command is being executed inform the user that there is a
                // new command available
                if (foundCommand) {
                    ienv.log.msg(`The ${chalk_1.default.green(parsedArgs[0])} command has been renamed. To find out more, run:\n\n` +
                        `    ${chalk_1.default.green(`ionic ${foundCommand} --help`)}\n\n`);
                }
                else {
                    yield executor.execute(pargv, process.env);
                }
                if (ienv.flags.interactive) {
                    const updateNotifier = yield Promise.resolve().then(() => require('update-notifier'));
                    updateNotifier({ pkg: yield loadPackageJson() }).notify({ isGlobal: true });
                }
            }
            catch (e) {
                err = e;
            }
        }
        if (err) {
            process.exitCode = 1;
            if (err instanceof cli_framework_1.InputValidationError) {
                for (const e of err.errors) {
                    ienv.log.error(e.message);
                }
                ienv.log.msg(`Use the ${chalk_1.default.green('--help')} flag for more details.`);
            }
            else if (guards_1.isSuperAgentError(err)) {
                const { formatSuperAgentError } = yield Promise.resolve().then(() => require('./lib/http'));
                ienv.log.rawmsg(formatSuperAgentError(err));
            }
            else if (err.code && err.code === 'ENOTFOUND' || err.code === 'ECONNREFUSED') {
                ienv.log.error(`Network connectivity error occurred, are you offline?\n` +
                    `If you are behind a firewall and need to configure proxy settings, see: ${chalk_1.default.bold('https://ion.link/cli-proxy-docs')}\n\n` +
                    chalk_1.default.red(String(err.stack ? err.stack : err)));
            }
            else if (guards_1.isExitCodeException(err)) {
                if (err.message) {
                    if (err.exitCode > 0) {
                        ienv.log.error(err.message);
                    }
                    else {
                        ienv.log.msg(err.message);
                    }
                }
                yield process_1.processExit(err.exitCode);
            }
            else if (err instanceof cli_framework_1.BaseError) {
                ienv.log.error(err.message);
            }
            else {
                ienv.log.msg(chalk_1.default.red(String(err.stack ? err.stack : err)));
                if (err.stack) {
                    debug(chalk_1.default.red(String(err.stack)));
                }
            }
        }
    });
}
exports.run = run;
function receive(msg) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (!_executor) {
            throw new Error('Executor not initialized.');
        }
        const { env, project } = _executor.namespace;
        if (msg.type === 'telemetry') {
            const { sendCommand } = yield Promise.resolve().then(() => require('./lib/telemetry'));
            yield sendCommand({
                getInfo: env.getInfo,
                client: env.client,
                config: env.config,
                ctx: env.ctx,
                project,
                session: env.session,
            }, msg.data.command, msg.data.args);
        }
    });
}
exports.receive = receive;
