"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const chalk_1 = require("chalk");
const guards_1 = require("../guards");
const http_1 = require("./http");
function formatName(app) {
    if (app.org) {
        return `${chalk_1.default.dim(`${app.org.name} / `)}${app.name}`;
    }
    return app.name;
}
exports.formatName = formatName;
class AppClient extends http_1.ResourceClient {
    constructor(token, e) {
        super();
        this.token = token;
        this.e = e;
    }
    load(id) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.e.client.make('GET', `/apps/${id}`);
            this.applyAuthentication(req, this.token);
            const res = yield this.e.client.do(req);
            if (!guards_1.isAppResponse(res)) {
                throw http_1.createFatalAPIFormat(req, res);
            }
            return res.data;
        });
    }
    create(details) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.e.client.make('POST', '/apps');
            this.applyAuthentication(req, this.token);
            req.send(details);
            const res = yield this.e.client.do(req);
            if (!guards_1.isAppResponse(res)) {
                throw http_1.createFatalAPIFormat(req, res);
            }
            return res.data;
        });
    }
    paginate(args = {}) {
        return this.e.client.paginate(Object.assign({ reqgen: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const { req } = yield this.e.client.make('GET', '/apps');
                this.applyAuthentication(req, this.token);
                return { req };
            }), guard: guards_1.isAppsResponse }, args));
    }
    createAssociation(id, association) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.e.client.make('POST', `/apps/${id}/repository`);
            req
                .set('Authorization', `Bearer ${this.token}`)
                .send({
                repository_id: association.repoId,
                type: association.type,
                branches: association.branches,
            });
            const res = yield this.e.client.do(req);
            if (!guards_1.isAppAssociationResponse(res)) {
                throw http_1.createFatalAPIFormat(req, res);
            }
            return res.data;
        });
    }
    deleteAssociation(id) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.e.client.make('DELETE', `/apps/${id}/repository`);
            req
                .set('Authorization', `Bearer ${this.token}`)
                .send({});
            yield req;
        });
    }
}
exports.AppClient = AppClient;
