"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const terminal_1 = require("@ionic/cli-framework/utils/terminal");
const chalk_1 = require("chalk");
const LogUpdate = require("log-update");
const guards_1 = require("../guards");
const logger_1 = require("./utils/logger");
class Command extends cli_framework_1.BaseCommand {
    constructor(namespace) {
        super(namespace);
        this.namespace = namespace;
        this.taskChains = [];
    }
    get env() {
        return this.namespace.root.env;
    }
    get project() {
        return this.namespace.root.project;
    }
    createTaskChain() {
        let output;
        const formatter = logger_1.createFormatter();
        if (this.env.flags.interactive) {
            output = new cli_framework_1.LogUpdateOutputStrategy({ LogUpdate });
            this.env.log.handlers = new Set([new cli_framework_1.StreamHandler({ stream: output.stream, formatter })]);
        }
        else {
            this.env.log.handlers = logger_1.createDefaultLoggerHandlers();
            output = new cli_framework_1.StreamOutputStrategy({ stream: this.env.log.createWriteStream(cli_framework_1.LOGGER_LEVELS.INFO, false) });
        }
        const chain = output.createTaskChain();
        this.taskChains.push(chain);
        chain.on('end', () => {
            this.env.log.handlers = logger_1.createDefaultLoggerHandlers();
        });
        return chain;
    }
    execute(inputs, options, runinfo) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (guards_1.isCommandPreRun(this)) {
                yield this.preRun(inputs, options, runinfo);
            }
            try {
                yield this.validate(inputs);
            }
            catch (e) {
                if (!this.env.flags.interactive) {
                    this.env.log.warn(`Command ran non-interactively due to ${chalk_1.default.green('--no-interactive')} flag, CI being detected, non-TTY, or a config setting.`);
                }
                throw e;
            }
            const runPromise = this.run(inputs, options, runinfo);
            const telemetryPromise = (() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (this.env.config.get('telemetry') !== false && !terminal_1.TERMINAL_INFO.ci) {
                    const { Telemetry } = yield Promise.resolve().then(() => require('./telemetry'));
                    let cmdInputs = [];
                    const metadata = yield this.getMetadata();
                    if (metadata.name === 'login' || metadata.name === 'logout') {
                        yield runPromise;
                    }
                    else if (metadata.name === 'help') {
                        cmdInputs = inputs;
                    }
                    else {
                        cmdInputs = yield this.getCleanInputsForTelemetry(inputs, options);
                    }
                    const cmd = this;
                    const path = yield cli_framework_1.generateCommandPath(cmd);
                    const telemetry = new Telemetry({ client: this.env.client, config: this.env.config, getInfo: this.env.getInfo, ctx: this.env.ctx, project: this.project, session: this.env.session });
                    yield telemetry.sendCommand(path.map(([p]) => p).join(' '), cmdInputs);
                }
            }))();
            yield Promise.all([runPromise, telemetryPromise]);
        });
    }
    getCleanInputsForTelemetry(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const initialOptions = { _: [] };
            const metadata = yield this.getMetadata();
            const filteredInputs = inputs.map((input, i) => metadata.inputs && (metadata.inputs[i] && metadata.inputs[i].private) ? '*****' : input);
            const filteredOptions = Object.keys(options)
                .filter(optionName => {
                if (optionName === '_') {
                    return false;
                }
                const metadataOption = metadata.options && metadata.options.find(o => {
                    return o.name === optionName || (typeof o.aliases !== 'undefined' && o.aliases.includes(optionName));
                });
                if (metadataOption && metadataOption.aliases && metadataOption.aliases.includes(optionName)) {
                    return false; // exclude aliases
                }
                if (!metadataOption) {
                    return true; // include unknown options
                }
                if (metadataOption.private) {
                    return false; // exclude private options
                }
                if (typeof metadataOption.default !== 'undefined' && metadataOption.default === options[optionName]) {
                    return false; // exclude options that match their default value (means it wasn't supplied by user)
                }
                return true;
            })
                .reduce((allOptions, optionName) => {
                allOptions[optionName] = options[optionName];
                return allOptions;
            }, initialOptions);
            const optionInputs = cli_framework_1.unparseArgs(filteredOptions, { useDoubleQuotes: true });
            return filteredInputs.concat(optionInputs);
        });
    }
}
exports.Command = Command;
