"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const path = require("path");
const app_1 = require("../../app");
const git_1 = require("../../git");
const config_1 = require("../../integrations/cordova/config");
const project_1 = require("../../integrations/cordova/project");
const npm_1 = require("../../utils/npm");
const base_1 = require("./base");
tslib_1.__exportStar(require("./base"), exports);
tslib_1.__exportStar(require("./utils"), exports);
class NpmInstalledLocally extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'npm-installed-locally';
        this.treatable = true;
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (`${chalk_1.default.bold('npm')} is installed locally.\n` +
                `${chalk_1.default.bold('npm')} is typically installed globally and may cause some confusion about versions when other CLIs use it.\n`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pkg = yield this.getLocalPackageJson('npm');
            return pkg !== undefined;
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [manager, ...managerArgs] = yield npm_1.pkgManagerArgs(this.config.get('npmClient'), { command: 'uninstall', pkg: 'npm' });
            return [
                {
                    message: `Run: ${chalk_1.default.green(manager + ' ' + managerArgs.join(' '))}`,
                    treat: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        yield this.shell.run(manager, managerArgs, {});
                    }),
                },
            ];
        });
    }
}
exports.NpmInstalledLocally = NpmInstalledLocally;
class IonicCLIInstalledLocally extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'ionic-installed-locally';
        this.treatable = true;
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (`The Ionic CLI is installed locally.\n` +
                `While the CLI can run locally, there's no longer a reason to have it installed locally and it may cause some confusion over configuration and versions.\n`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pkg = yield this.getLocalPackageJson('ionic');
            return pkg !== undefined;
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [manager, ...managerArgs] = yield npm_1.pkgManagerArgs(this.config.get('npmClient'), { command: 'uninstall', pkg: 'ionic' });
            return [
                {
                    message: `Run: ${chalk_1.default.green(manager + ' ' + managerArgs.join(' '))}`,
                    treat: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        yield this.shell.run(manager, managerArgs, {});
                    }),
                },
            ];
        });
    }
}
exports.IonicCLIInstalledLocally = IonicCLIInstalledLocally;
class GitNotUsed extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'git-not-used';
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (`Git doesn't appear to be in use.\n` +
                `We highly recommend using source control software such as git (${chalk_1.default.bold('https://git-scm.com')}) to track changes in your code throughout time.\n`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!(yield git_1.isRepoInitialized(this.project.directory))) {
                return true;
            }
            const cmdInstalled = yield this.shell.cmdinfo('git', ['--version']);
            if (!cmdInstalled) {
                return true;
            }
            const [revListCount, status] = yield Promise.all([
                this.shell.output('git', ['rev-list', '--count', 'HEAD'], { fatalOnError: false, showCommand: false, showError: false }),
                this.shell.output('git', ['status', '--porcelain'], { fatalOnError: false, showCommand: false, showError: false }),
            ]);
            this.debug('rev-list count: %s, status: %s', revListCount.trim(), status);
            if (!revListCount) {
                return true;
            }
            const commitCount = Number(revListCount);
            const changes = Boolean(status);
            return commitCount === 1 && changes;
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return [
                { message: `Download git if you don't have it installed: ${chalk_1.default.bold('https://git-scm.com/downloads')}` },
                { message: `Learn the basics if you're unfamiliar with git: ${chalk_1.default.bold('https://try.github.io')}` },
                { message: `Make your first commit and start tracking code changes! 😍` },
            ];
        });
    }
}
exports.GitNotUsed = GitNotUsed;
class GitConfigInvalid extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'git-config-invalid';
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const appflowId = yield this.project.requireAppflowId();
            return (`App linked to ${chalk_1.default.bold(appflowId)} with invalid git configuration.\n` +
                `This app is linked to an app on Ionic (${chalk_1.default.bold(appflowId)}), but the git configuration is not valid.\n`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const isLoggedIn = this.session.isLoggedIn();
            if (!isLoggedIn) {
                return false;
            }
            const appflowId = this.project.config.get('id');
            if (!appflowId) {
                return false;
            }
            if (!(yield git_1.isRepoInitialized(this.project.directory))) {
                return false;
            }
            const remote = yield git_1.getIonicRemote({ shell: this.shell }, this.project.directory);
            if (!remote) {
                return true;
            }
            const token = this.session.getUserToken();
            const appClient = new app_1.AppClient(token, { client: this.client });
            const app = yield appClient.load(appflowId);
            if (app.repo_url !== remote) {
                return true;
            }
            return false;
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return [
                { message: `Run: ${chalk_1.default.green('ionic git remote')}` },
            ];
        });
    }
}
exports.GitConfigInvalid = GitConfigInvalid;
class IonicNativeOldVersionInstalled extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'ionic-native-old-version-installed';
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (`Old version of Ionic Native installed.\n` +
                `Ionic Native ${chalk_1.default.bold('ionic-native')} has been restructured into individual packages under the ${chalk_1.default.bold('@ionic-native/')} namespace to allow for better bundling and faster apps.\n`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pkg = yield this.getLocalPackageJson('ionic-native');
            return pkg !== undefined;
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const args = yield npm_1.pkgManagerArgs(this.config.get('npmClient'), { command: 'uninstall', pkg: 'ionic-native' });
            return [
                { message: `Run ${chalk_1.default.green(args.join(' '))}` },
                { message: `Refer to ${chalk_1.default.bold('https://ion.link/native-docs')} for installation & usage instructions` },
            ];
        });
    }
}
exports.IonicNativeOldVersionInstalled = IonicNativeOldVersionInstalled;
class UnsavedCordovaPlatforms extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'unsaved-cordova-platforms';
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (`Cordova platforms unsaved.\n` +
                `There are Cordova platforms installed that are not saved in ${chalk_1.default.bold('config.xml')} or ${chalk_1.default.bold('package.json')}. It is good practice to manage Cordova platforms and their versions. See the Cordova docs${chalk_1.default.cyan('[1]')} for more information.\n\n` +
                `${chalk_1.default.cyan('[1]')}: ${chalk_1.default.bold('https://cordova.apache.org/docs/en/latest/platform_plugin_versioning_ref/')}\n`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cordova = this.project.getIntegration('cordova');
            if (!cordova || !cordova.enabled) {
                return false;
            }
            const platforms = yield project_1.getPlatforms(cordova.root);
            const conf = yield config_1.loadConfigXml(cordova);
            const engines = conf.getPlatformEngines();
            const engineNames = new Set([...engines.map(e => e.name)]);
            const configXmlDiff = platforms.filter(p => !engineNames.has(p));
            return configXmlDiff.length > 0;
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return [
                { message: `Run: ${chalk_1.default.green('ionic cordova platform save')}` },
            ];
        });
    }
}
exports.UnsavedCordovaPlatforms = UnsavedCordovaPlatforms;
class DefaultCordovaBundleIdUsed extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'default-cordova-bundle-id-used';
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (`Package ID unchanged in ${chalk_1.default.bold('config.xml')}.\n` +
                `The Package Identifier (AKA "Bundle ID" for iOS and "Application ID" for Android) is a unique ID (usually written in reverse DNS notation, such as ${chalk_1.default.bold('com.mycompany.MyApp')}) that Cordova uses when compiling the native build of your app. When your app is submitted to the App Store or Play Store, the Package ID can't be changed. This issue was detected because this app's Package ID is ${chalk_1.default.green('"io.ionic.starter"')}, which is the default Package ID provided after running ${chalk_1.default.green('ionic start')}.`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cordova = this.project.getIntegration('cordova');
            if (!cordova || !cordova.enabled) {
                return false;
            }
            const conf = yield config_1.loadConfigXml(cordova);
            return conf.getBundleId() === 'io.ionic.starter';
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return [
                { message: `Change the ${chalk_1.default.bold('id')} attribute of ${chalk_1.default.bold('<widget>')} (root element) to something other than ${chalk_1.default.green('"io.ionic.starter"')}` },
            ];
        });
    }
}
exports.DefaultCordovaBundleIdUsed = DefaultCordovaBundleIdUsed;
class ViewportFitNotSet extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'viewport-fit-not-set';
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (`${chalk_1.default.bold('viewport-fit=cover')} not set in ${chalk_1.default.bold('index.html')}\n` +
                `iOS 11 introduces new "safe regions" for webviews, which can throw off component sizing, squish the header into the status bar, letterbox the app on iPhone X, etc. Fixing this issue will ensure the webview takes up the full size of the screen. See ${chalk_1.default.bold('https://blog.ionicframework.com/ios-11-checklist')} for more information.`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const indexHtml = yield utils_fs_1.readFile(path.resolve(yield this.project.getSourceDir(), 'index.html'), { encoding: 'utf8' });
            const m = indexHtml.match(/\<meta([\s]*(name="viewport"){1})[\w\d\s\.\-,=]*(content="){1}[\w\d\s\.\-,=]*(viewport-fit=cover){1}[\w\d\s\.\-,="]+\/\>/);
            return !Boolean(m);
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return [
                { message: `Add ${chalk_1.default.bold('viewport-fit=cover')} to the ${chalk_1.default.bold('<meta name="viewport">')} tag in your ${chalk_1.default.bold('index.html')} file` },
            ];
        });
    }
}
exports.ViewportFitNotSet = ViewportFitNotSet;
class CordovaPlatformsCommitted extends base_1.Ailment {
    constructor() {
        super(...arguments);
        this.id = 'cordova-platforms-committed';
    }
    getMessage() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return (`Cordova ${chalk_1.default.bold('platforms/')} directory is committed to git.\n` +
                `Cordova considers ${chalk_1.default.bold('platforms/')} and ${chalk_1.default.bold('plugins/')} build artifacts${chalk_1.default.cyan('[1]')}, and routinely overwrites files.\n\n` +
                `While committing these files might be necessary for some projects${chalk_1.default.cyan('[2]')}, generally platforms should be configured using ${chalk_1.default.bold('config.xml')} and Cordova hooks${chalk_1.default.cyan('[3]')} so that your project is more portable and SDK updates are easier.\n\n` +
                `${chalk_1.default.cyan('[1]')}: ${chalk_1.default.bold('https://cordova.apache.org/docs/en/latest/reference/cordova-cli/#version-control')}\n` +
                `${chalk_1.default.cyan('[2]')}: ${chalk_1.default.bold('https://cordova.apache.org/docs/en/latest/reference/cordova-cli/#platforms')}\n` +
                `${chalk_1.default.cyan('[3]')}: ${chalk_1.default.bold('https://cordova.apache.org/docs/en/latest/guide/appdev/hooks/index.html')}\n\n` +
                `${chalk_1.default.yellow(`${chalk_1.default.bold('WARNING')}: Attempting to fix this could be dangerous. Only proceed if you're sure you haven't made manual modifications to these files.`)}\n`).trim();
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!(yield git_1.isRepoInitialized(this.project.directory))) {
                return false;
            }
            const cmdInstalled = yield this.shell.cmdinfo('git', ['--version']);
            if (!cmdInstalled) {
                return false;
            }
            const files = (yield this.shell.output('git', ['ls-tree', '--name-only', 'HEAD'], { fatalOnError: false, showCommand: false, showError: false })).split('\n');
            return files.includes('platforms'); // TODO
        });
    }
    getTreatmentSteps() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return [
                { message: `Remove ${chalk_1.default.bold('platforms/')} from source control: ${chalk_1.default.green('git rm -rf platforms/')} and ${chalk_1.default.green('git commit')}` },
                { message: `Make sure the ${chalk_1.default.bold('platforms/')} directory has been removed: ${chalk_1.default.green('rm -rf platforms/')}` },
                { message: `Allow Cordova to repopulate your platforms: ${chalk_1.default.green('ionic cordova prepare')}` },
            ];
        });
    }
}
exports.CordovaPlatformsCommitted = CordovaPlatformsCommitted;
