"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const constants_1 = require("../constants");
const guards_1 = require("../guards");
const config_1 = require("./config");
const errors_1 = require("./errors");
class Executor extends cli_framework_1.AbstractExecutor {
    constructor({ namespace }) {
        super();
        this.namespace = namespace;
    }
    execute(argv, env) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pargs = cli_framework_1.stripOptions(argv, {});
            const location = yield this.namespace.locate(pargs);
            if (argv.includes('--version')) {
                return this.execute(['version', ...pargs], env);
            }
            else if (argv.find(arg => arg === '--help' || arg === '-?' || arg === '-h') || !guards_1.isCommand(location.obj)) {
                return this.execute(['help', ...pargs], env);
            }
            const cmd = location.obj;
            const path = location.path;
            const subcommandName = path[path.length - 1][0];
            const subcommandNameArgIdx = argv.findIndex(arg => arg === subcommandName);
            const cmdargs = argv.slice(subcommandNameArgIdx + 1);
            yield this.run(cmd, cmdargs, { location, env, executor: this });
        });
    }
    run(command, cmdargs, { location, env, executor }) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const metadata = yield command.getMetadata();
            const fullNameParts = location.path.map(([p]) => p);
            if (metadata.options) {
                const optMap = metadataToCmdOptsEnv(metadata, fullNameParts.slice(1));
                // TODO: changes opt by reference, which is probably bad
                for (const [opt, envvar] of optMap.entries()) {
                    const envdefault = env[envvar];
                    if (typeof envdefault !== 'undefined') {
                        opt.default = opt.type === Boolean ? (envdefault && envdefault !== '0' ? true : false) : envdefault;
                    }
                }
            }
            const metadataOpts = [...metadata.options ? metadata.options : [], ...config_1.GLOBAL_OPTIONS];
            const minimistOpts = cli_framework_1.metadataOptionsToParseArgsOptions(metadataOpts);
            const cmdoptions = cli_framework_1.parseArgs(cmdargs, minimistOpts);
            const cmdinputs = cmdoptions._;
            if (!this.namespace.project && metadata.type === 'project') {
                throw new errors_1.FatalException(`Sorry! ${chalk_1.default.green(fullNameParts.join(' '))} can only be run in an Ionic project directory.\n` +
                    `If this is a project you'd like to integrate with Ionic, create an ${chalk_1.default.bold(constants_1.PROJECT_FILE)} file.`);
            }
            yield command.execute(cmdinputs, cmdoptions, { location, env, executor });
        });
    }
}
exports.Executor = Executor;
function runCommand(runinfo, argv) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const { env, executor } = runinfo;
        const metadata = yield executor.namespace.getMetadata();
        executor.namespace.env.log.msg(`> ${chalk_1.default.green([metadata.name, ...argv].map(a => a.includes(' ') ? `"${a}"` : a).join(' '))}`);
        yield executor.execute(argv, env);
    });
}
exports.runCommand = runCommand;
function metadataToCmdOptsEnv(metadata, cmdNameParts) {
    const optMap = new Map();
    if (!metadata.options) {
        return optMap;
    }
    const prefix = `IONIC_CMDOPTS_${cmdNameParts.map(s => s.toUpperCase()).join('_')}`;
    for (const option of metadata.options) {
        optMap.set(option, `${prefix}_${option.name.toUpperCase().split('-').join('_')}`);
    }
    return optMap;
}
exports.metadataToCmdOptsEnv = metadataToCmdOptsEnv;
