"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const array_1 = require("@ionic/cli-framework/utils/array");
const format_1 = require("@ionic/cli-framework/utils/format");
const chalk_1 = require("chalk");
const Debug = require("debug");
const lodash = require("lodash");
const path = require("path");
const errors_1 = require("./errors");
const debug = Debug('ionic:lib:hooks');
class Hook {
    constructor(e) {
        this.e = e;
    }
    get script() {
        return `ionic:${this.name}`;
    }
    run(input) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { pkgManagerArgs } = yield Promise.resolve().then(() => require('./utils/npm'));
            const type = this.e.project.type;
            if (!type || !this.e.project.directory) {
                return; // TODO: will we need hooks outside a project?
            }
            const pkg = yield this.e.project.requirePackageJson();
            debug(`Looking for ${chalk_1.default.cyan(this.script)} npm script.`);
            if (pkg.scripts && pkg.scripts[this.script]) {
                debug(`Invoking ${chalk_1.default.cyan(this.script)} npm script.`);
                const [pkgManager, ...pkgArgs] = yield pkgManagerArgs(this.e.config.get('npmClient'), { command: 'run', script: this.script });
                yield this.e.shell.run(pkgManager, pkgArgs, {});
            }
            const projectHooks = this.e.project.config.get('hooks');
            const hooks = projectHooks ? array_1.conform(projectHooks[this.name]) : [];
            for (const h of hooks) {
                const p = path.resolve(this.e.project.directory, h);
                try {
                    if (path.extname(p) !== '.js') {
                        throw new Error(`Hooks must be .js files with a function for its default export.`);
                    }
                    const hook = yield this.loadHookFn(p);
                    if (!hook) {
                        throw new Error(`Module must have a function for its default export.`);
                    }
                    yield hook(lodash.assign({}, input, {
                        project: {
                            type,
                            dir: this.e.project.directory,
                            srcDir: yield this.e.project.getSourceDir(),
                        },
                        argv: process.argv,
                        env: process.env,
                    }));
                }
                catch (e) {
                    throw new errors_1.HookException(`An error occurred while running an Ionic CLI hook defined in ${chalk_1.default.bold(format_1.prettyPath(this.e.project.filePath))}.\n` +
                        `Hook: ${chalk_1.default.bold(this.name)}\n` +
                        `File: ${chalk_1.default.bold(p)}\n\n` +
                        `${chalk_1.default.red(e.stack ? e.stack : e)}`);
                }
            }
        });
    }
    loadHookFn(p) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const module = require(p);
            if (typeof module === 'function') {
                return module;
            }
            else if (typeof module.default === 'function') {
                return module.default;
            }
            debug(`Could not load hook function ${chalk_1.default.bold(p)}: %o not a function`, module);
        });
    }
}
exports.Hook = Hook;
function addHook(baseDir, hooks, hook) {
    const hookPaths = array_1.conform(hooks);
    const resolvedHookPaths = hookPaths.map(p => path.resolve(baseDir, p));
    if (!resolvedHookPaths.includes(path.resolve(baseDir, hook))) {
        hookPaths.push(hook);
    }
    return hookPaths;
}
exports.addHook = addHook;
function removeHook(baseDir, hooks, hook) {
    const hookPaths = array_1.conform(hooks);
    const i = locateHook(baseDir, hookPaths, hook);
    if (i >= 0) {
        hookPaths.splice(i, 1);
    }
    return hookPaths;
}
exports.removeHook = removeHook;
function locateHook(baseDir, hooks, hook) {
    return array_1.conform(hooks).map(p => path.resolve(baseDir, p)).indexOf(path.resolve(baseDir, hook));
}
exports.locateHook = locateHook;
