"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const Debug = require("debug");
const et = require("elementtree");
const path = require("path");
const errors_1 = require("../../errors");
const uuid_1 = require("../../utils/uuid");
const debug = Debug('ionic:lib:integrations:cordova:config');
class ConfigXml {
    constructor(filePath) {
        this.filePath = filePath;
        this.saving = false;
    }
    get doc() {
        if (!this._doc) {
            throw new Error('No doc loaded.');
        }
        return this._doc;
    }
    get sessionid() {
        if (!this._sessionid) {
            throw new Error('No doc loaded.');
        }
        return this._sessionid;
    }
    static load(filePath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!filePath) {
                throw new Error('Must supply file path.');
            }
            const conf = new ConfigXml(filePath);
            yield conf.reload();
            return conf;
        });
    }
    reload() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const configFileContents = yield utils_fs_1.readFile(this.filePath, { encoding: 'utf8' });
            if (!configFileContents) {
                throw new Error(`Cannot load empty config.xml file.`);
            }
            try {
                this._doc = et.parse(configFileContents);
                this._sessionid = uuid_1.shortid();
            }
            catch (e) {
                throw new Error(`Cannot parse config.xml file: ${e.stack ? e.stack : e}`);
            }
        });
    }
    save() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.saving) {
                this.saving = true;
                yield utils_fs_1.writeFile(this.filePath, this.write(), { encoding: 'utf8' });
                this.saving = false;
            }
        });
    }
    setName(name) {
        const root = this.doc.getroot();
        let nameNode = root.find('name');
        if (!nameNode) {
            nameNode = et.SubElement(root, 'name', {});
        }
        nameNode.text = name;
    }
    setBundleId(bundleId) {
        const root = this.doc.getroot();
        root.set('id', bundleId);
    }
    getBundleId() {
        const root = this.doc.getroot();
        return root.get('id');
    }
    /**
     * Update config.xml content src to be a dev server url. As part of this
     * backup the original content src for a reset to occur at a later time.
     */
    writeContentSrc(newSrc) {
        const root = this.doc.getroot();
        let contentElement = root.find('content');
        if (!contentElement) {
            contentElement = et.SubElement(root, 'content', { src: 'index.html' });
        }
        contentElement.set('original-src', contentElement.get('src'));
        contentElement.set('src', newSrc);
        let navElement = root.find(`allow-navigation[@href='${newSrc}']`);
        if (!navElement) {
            navElement = et.SubElement(root, 'allow-navigation', { sessionid: this.sessionid, href: newSrc });
        }
    }
    /**
     * Set config.xml src url back to its original url
     */
    resetContentSrc() {
        const root = this.doc.getroot();
        let contentElement = root.find('content');
        if (!contentElement) {
            contentElement = et.SubElement(root, 'content', { src: 'index.html' });
        }
        const originalSrc = contentElement.get('original-src');
        if (originalSrc) {
            contentElement.set('src', originalSrc);
            delete contentElement.attrib['original-src'];
        }
        const navElements = root.findall(`allow-navigation[@sessionid='${this.sessionid}']`);
        for (const navElement of navElements) {
            root.remove(navElement);
        }
    }
    getPreference(prefName) {
        const root = this.doc.getroot();
        const preferenceElement = root.find(`preference[@name='${prefName}']`);
        if (!preferenceElement) {
            return undefined;
        }
        const value = preferenceElement.get('value');
        if (!value) {
            return undefined;
        }
        return value;
    }
    getProjectInfo() {
        const root = this.doc.getroot();
        let id = root.get('id');
        if (!id) {
            id = '';
        }
        let version = root.get('version');
        if (!version) {
            version = '';
        }
        let nameElement = root.find('name');
        if (!nameElement) {
            nameElement = et.SubElement(root, 'name', {});
        }
        if (!nameElement.text) {
            nameElement.text = 'MyApp';
        }
        const name = nameElement.text.toString();
        return { id, name, version };
    }
    getPlatformEngines() {
        const root = this.doc.getroot();
        const engines = root.findall('engine');
        return engines.map(engine => this.engineElementToPlatformEngine(engine));
    }
    getPlatformEngine(platform) {
        const root = this.doc.getroot();
        const engine = root.find(`engine[@name='${platform}']`);
        if (!engine) {
            return undefined;
        }
        return this.engineElementToPlatformEngine(engine);
    }
    ensurePlatformImages(platform, resourcesPlatform) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const root = this.doc.getroot();
            const orientation = this.getPreference('Orientation') || 'default';
            for (const imgName in resourcesPlatform) {
                const imgType = resourcesPlatform[imgName];
                let platformElement = root.find(`platform[@name='${platform}']`);
                if (!platformElement) {
                    platformElement = et.SubElement(root, 'platform', { name: platform });
                }
                const images = imgType.images.filter(img => orientation === 'default' || typeof img.orientation === 'undefined' || img.orientation === orientation);
                for (const image of images) {
                    // We use forward slashes, (not path.join) here to provide
                    // cross-platform compatibility for paths.
                    const imgPath = ['resources', platform, imgType.nodeName, image.name].join('/'); // TODO: hard-coded 'resources' dir
                    let imgElement = platformElement.find(`${imgType.nodeName}[@src='${imgPath}']`);
                    if (!imgElement) {
                        imgElement = platformElement.find(`${imgType.nodeName}[@src='${imgPath.split('/').join('\\')}']`);
                    }
                    if (!imgElement) {
                        const attrs = {};
                        for (const attr of imgType.nodeAttributes) {
                            let v = image[attr]; // TODO
                            if (attr === 'src') {
                                v = imgPath;
                            }
                            attrs[attr] = v;
                        }
                        imgElement = et.SubElement(platformElement, imgType.nodeName, attrs);
                    }
                    imgElement.set('src', imgPath);
                }
            }
        });
    }
    ensureSplashScreenPreferences() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const root = this.doc.getroot();
            let splashScreenPrefElement = root.find(`preference[@name='SplashScreen']`);
            if (!splashScreenPrefElement) {
                splashScreenPrefElement = et.SubElement(root, 'preference', { name: 'SplashScreen', value: 'screen' });
            }
            let splashShowOnlyFirstTimePrefElement = root.find(`preference[@name='SplashShowOnlyFirstTime']`);
            if (!splashShowOnlyFirstTimePrefElement) {
                splashShowOnlyFirstTimePrefElement = et.SubElement(root, 'preference', { name: 'SplashShowOnlyFirstTime', value: 'false' });
            }
            let splashScreenDelayPrefElement = root.find(`preference[@name='SplashScreenDelay']`);
            if (!splashScreenDelayPrefElement) {
                splashScreenDelayPrefElement = et.SubElement(root, 'preference', { name: 'SplashScreenDelay', value: '3000' });
            }
        });
    }
    write() {
        // Cordova hard codes an indentation of 4 spaces, so we'll follow.
        const contents = this.doc.write({ indent: 4 });
        return contents;
    }
    engineElementToPlatformEngine(engine) {
        const name = engine.get('name');
        const spec = engine.get('spec');
        return Object.assign({ name: name ? name : '', spec: spec ? spec : '' }, engine.attrib);
    }
}
exports.ConfigXml = ConfigXml;
function loadConfigXml(integration) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const filePath = path.resolve(integration.root, 'config.xml');
        debug(`Using config.xml: ${filePath}`);
        try {
            return yield ConfigXml.load(filePath);
        }
        catch (e) {
            const msg = e.code === 'ENOENT'
                ? `Cordova ${chalk_1.default.bold('config.xml')} file not found.\n\nYou can re-add the Cordova integration with the following command: ${chalk_1.default.green('ionic integrations enable cordova --add')}`
                : chalk_1.default.red(e.stack ? e.stack : e);
            throw new errors_1.FatalException(`Cannot load ${chalk_1.default.bold(format_1.prettyPath(filePath))}\n` +
                `${msg}`);
        }
    });
}
exports.loadConfigXml = loadConfigXml;
