"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const Debug = require("debug");
const lodash = require("lodash");
const os = require("os");
const path = require("path");
const errors_1 = require("../errors");
var guards_1 = require("../../guards");
exports.INTEGRATION_NAMES = guards_1.INTEGRATION_NAMES;
const debug = Debug('ionic:lib:integrations');
class BaseIntegration {
    constructor(e) {
        this.e = e;
    }
    static createFromName(deps, name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (name === 'capacitor') {
                const { Integration } = yield Promise.resolve().then(() => require('./capacitor'));
                return new Integration(deps);
            }
            else if (name === 'cordova') {
                const { Integration } = yield Promise.resolve().then(() => require('./cordova'));
                return new Integration(deps);
            }
            throw new errors_1.IntegrationNotFoundException(`Bad integration name: ${chalk_1.default.bold(name)}`); // TODO?
        });
    }
    getInfo() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return [];
        });
    }
    enable() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // optionally overwritten by subclasses
        });
    }
    disable() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // optionally overwritten by subclasses
        });
    }
    personalize(details) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // optionally overwritten by subclasses
        });
    }
    add(details, handlers = {}) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.archiveUrl) {
                return;
            }
            const onFileCreate = handlers.onFileCreate ? handlers.onFileCreate : lodash.noop;
            const conflictHandler = handlers.conflictHandler ? handlers.conflictHandler : () => tslib_1.__awaiter(this, void 0, void 0, function* () { return false; });
            const { createRequest, download } = yield Promise.resolve().then(() => require('../utils/http'));
            const { tar } = yield Promise.resolve().then(() => require('../utils/archive'));
            this.e.log.info(`Downloading integration ${chalk_1.default.green(this.name)}`);
            const tmpdir = path.resolve(os.tmpdir(), `ionic-integration-${this.name}`);
            // TODO: etag
            if (yield utils_fs_1.pathExists(tmpdir)) {
                yield utils_fs_1.remove(tmpdir);
            }
            yield utils_fs_1.mkdirp(tmpdir);
            const ws = tar.extract({ cwd: tmpdir });
            const { req } = yield createRequest('GET', this.archiveUrl, this.e.config.getHTTPConfig());
            yield download(req, ws, {});
            const contents = yield utils_fs_1.readdirSafe(tmpdir);
            const blacklist = [];
            debug(`Integration files downloaded to ${chalk_1.default.bold(tmpdir)} (files: ${contents.map(f => chalk_1.default.bold(f)).join(', ')})`);
            for (const f of contents) {
                const projectf = path.resolve(this.e.project.directory, f);
                try {
                    const stats = yield utils_fs_1.stat(projectf);
                    const overwrite = yield conflictHandler(projectf, stats);
                    if (!overwrite) {
                        blacklist.push(f);
                    }
                }
                catch (e) {
                    if (e.code !== 'ENOENT') {
                        throw e;
                    }
                }
            }
            this.e.log.info(`Copying integrations files to project`);
            debug(`Blacklist: ${blacklist.map(f => chalk_1.default.bold(f)).join(', ')}`);
            yield utils_fs_1.mkdirp(details.root);
            yield utils_fs_1.copy(tmpdir, details.root, {
                filter: f => {
                    if (f === tmpdir) {
                        return true;
                    }
                    const projectf = f.substring(tmpdir.length + 1);
                    for (const item of blacklist) {
                        if (item.slice(-1) === '/' && `${projectf}/` === item) {
                            return false;
                        }
                        if (projectf.startsWith(item)) {
                            return false;
                        }
                    }
                    onFileCreate(projectf);
                    return true;
                },
            });
            yield this.enable();
        });
    }
}
exports.BaseIntegration = BaseIntegration;
