"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const Debug = require("debug");
const lodash = require("lodash");
const config_1 = require("../../config");
const errors_1 = require("../../errors");
const generate_1 = require("../../generate");
// https://github.com/ionic-team/angular-toolkit/blob/master/collection.json
const SCHEMATICS = ['page', 'component', 'service', 'module', 'class', 'directive', 'guard', 'pipe', 'interface', 'enum'];
const SCHEMATIC_ALIAS = new Map([
    ['pg', 'page'],
    ['cl', 'class'],
    ['c', 'component'],
    ['d', 'directive'],
    ['e', 'enum'],
    ['g', 'guard'],
    ['i', 'interface'],
    ['m', 'module'],
    ['p', 'pipe'],
    ['s', 'service'],
]);
const debug = Debug('ionic:lib:project:angular:generate');
class AngularGenerateRunner extends generate_1.GenerateRunner {
    constructor(e) {
        super();
        this.e = e;
    }
    getCommandMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                groups: [cli_framework_1.CommandGroup.Beta],
                description: `
This command uses the Angular CLI to generate features such as ${['pages', 'components', 'directives', 'services'].map(c => chalk_1.default.green(c)).join(', ')}, etc.

 - For a full list of available types, use ${chalk_1.default.green('npx ng g --help')}
 - For a list of options for a types, use ${chalk_1.default.green('npx ng g <type> --help')}

You can specify a path to nest your feature within any number of subdirectories. For example, specify a name of ${chalk_1.default.green('"pages/New Page"')} to generate page files at ${chalk_1.default.bold('src/app/pages/new-page/')}.

To test a generator before file modifications are made, use the ${chalk_1.default.green('--dry-run')} option.
      `,
                exampleCommands: [
                    'page',
                    'page contact',
                    'component contact/form',
                    'component login-form --change-detection=OnPush',
                    'directive ripple --skip-import',
                    'service api/user',
                ],
                inputs: [
                    {
                        name: 'type',
                        summary: `The type of feature (e.g. ${['page', 'component', 'directive', 'service'].map(c => chalk_1.default.green(c)).join(', ')})`,
                        validators: [cli_framework_1.validators.required],
                    },
                    {
                        name: 'name',
                        summary: 'The name/path of the feature being generated',
                        validators: [cli_framework_1.validators.required],
                    },
                ],
            };
        });
    }
    ensureCommandLine(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (inputs[0]) {
                this.validateFeatureType(inputs[0]);
            }
            else {
                const type = yield this.e.prompt({
                    type: 'list',
                    name: 'type',
                    message: 'What would you like to generate?',
                    choices: SCHEMATICS,
                });
                inputs[0] = type;
            }
            if (!inputs[1]) {
                const type = SCHEMATIC_ALIAS.get(inputs[0]) || inputs[0];
                const name = yield this.e.prompt({
                    type: 'input',
                    name: 'name',
                    message: `Name/path of ${chalk_1.default.green(type)}:`,
                    validate: v => cli_framework_1.validators.required(v),
                });
                inputs[1] = name.trim();
            }
        });
    }
    createOptionsFromCommandLine(inputs, options) {
        const baseOptions = super.createOptionsFromCommandLine(inputs, options);
        // TODO: this is a little gross, is there a better way to pass in all the
        // options that the command got?
        return Object.assign({}, lodash.omit(options, '_', '--', ...config_1.GLOBAL_OPTIONS.map(opt => opt.name)), { project: options['project'] }, baseOptions);
    }
    run(options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { name } = options;
            const type = SCHEMATIC_ALIAS.get(options.type) || options.type;
            try {
                yield this.generateComponent(type, name, lodash.omit(options, 'type', 'name'));
            }
            catch (e) {
                debug(e);
                throw new errors_1.FatalException(`Could not generate ${chalk_1.default.green(type)}.`);
            }
            if (!options['dry-run']) {
                this.e.log.ok(`Generated ${chalk_1.default.green(type)}!`);
            }
        });
    }
    validateFeatureType(type) {
        if (type === 'provider') {
            throw new errors_1.FatalException(`Please use ${chalk_1.default.green('ionic generate service')} for generating service providers.\n` +
                `For more information, please see the Angular documentation${chalk_1.default.cyan('[1]')} on services.\n\n` +
                `${chalk_1.default.cyan('[1]')}: ${chalk_1.default.bold('https://angular.io/guide/architecture-services')}`);
        }
        if (!SCHEMATICS.includes(type) && !SCHEMATIC_ALIAS.get(type)) {
            throw new errors_1.FatalException(`${chalk_1.default.green(type)} is not a known feature.\n` +
                `Use ${chalk_1.default.green('npx ng g --help')} to list available types of features.`);
        }
    }
    generateComponent(type, name, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ngArgs = cli_framework_1.unparseArgs(Object.assign({ _: ['generate', type, name] }, options), {});
            const shellOptions = { cwd: this.e.ctx.execPath };
            yield this.e.shell.run('ng', ngArgs, shellOptions);
        });
    }
}
exports.AngularGenerateRunner = AngularGenerateRunner;
