"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_network_1 = require("@ionic/utils-network");
const chalk_1 = require("chalk");
const serve_1 = require("../../serve");
class AngularServeRunner extends serve_1.ServeRunner {
    constructor(e) {
        super();
        this.e = e;
    }
    getCommandMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                groups: [cli_framework_1.CommandGroup.Beta],
                description: `
${chalk_1.default.green('ionic serve')} uses the Angular CLI. Use ${chalk_1.default.green('ng serve --help')} to list all Angular CLI options for serving your app. See the ${chalk_1.default.green('ng serve')} docs[^ng-serve-docs] for explanations. Options not listed below are considered advanced and can be passed to the Angular CLI using the ${chalk_1.default.green('--')} separator after the Ionic CLI arguments. See the examples.

The dev server can use HTTPS via the ${chalk_1.default.green('--ssl')} option ${chalk_1.default.bold.red('(experimental)')}. There are several known issues with HTTPS. See issue #3305[^issue-3305].
`,
                footnotes: [
                    {
                        id: 'ng-serve-docs',
                        url: 'https://angular.io/cli/serve',
                    },
                    {
                        id: 'issue-3305',
                        url: 'https://github.com/ionic-team/ionic-cli/issues/3305',
                    },
                ],
                options: [
                    {
                        name: 'ssl',
                        summary: 'Use HTTPS for the dev server',
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Experimental, 'cordova'],
                        hint: chalk_1.default.dim('[ng]'),
                    },
                    {
                        name: 'prod',
                        summary: `Flag to use the ${chalk_1.default.green('production')} configuration`,
                        type: Boolean,
                        groups: ['cordova'],
                        hint: chalk_1.default.dim('[ng]'),
                    },
                    {
                        name: 'configuration',
                        summary: 'Specify the configuration to use.',
                        type: String,
                        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova'],
                        hint: chalk_1.default.dim('[ng]'),
                        spec: { value: 'conf' },
                    },
                    {
                        name: 'source-map',
                        summary: 'Output sourcemaps',
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova'],
                        hint: chalk_1.default.dim('[ng]'),
                    },
                    {
                        name: 'devapp',
                        summary: 'Publish DevApp service',
                        type: Boolean,
                        default: false,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                ],
                exampleCommands: [
                    '-- --proxy-config proxy.conf.json',
                ],
            };
        });
    }
    createOptionsFromCommandLine(inputs, options) {
        const baseOptions = super.createOptionsFromCommandLine(inputs, options);
        const prod = options['prod'] ? Boolean(options['prod']) : undefined;
        const ssl = options['ssl'] ? Boolean(options['ssl']) : undefined;
        const configuration = options['configuration'] ? String(options['configuration']) : (prod ? 'production' : undefined);
        const sourcemaps = typeof options['source-map'] === 'boolean' ? Boolean(options['source-map']) : undefined;
        return Object.assign({}, baseOptions, { ssl,
            configuration,
            sourcemaps });
    }
    determineEngineFromCommandLine(options) {
        if (options['devapp']) {
            return 'cordova';
        }
        return super.determineEngineFromCommandLine(options);
    }
    platformToMode(platform) {
        if (platform === 'ios') {
            return 'ios';
        }
        return 'md';
    }
    modifyOpenURL(url, options) {
        return `${url}${options.browserOption ? options.browserOption : ''}${options.platform ? `?ionic:mode=${this.platformToMode(options.platform)}&ionic:persistConfig=true` : ''}`;
    }
    serveProject(options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [externalIP, availableInterfaces] = yield this.selectExternalIP(options);
            const port = options.port = yield utils_network_1.findClosestOpenPort(options.port);
            const ng = new AngularServeCLI(this.e);
            yield ng.serve(options);
            return {
                custom: ng.resolvedProgram !== ng.program,
                protocol: options.ssl ? 'https' : 'http',
                localAddress: 'localhost',
                externalAddress: externalIP,
                externalNetworkInterfaces: availableInterfaces,
                port,
                externallyAccessible: ![serve_1.BIND_ALL_ADDRESS, ...serve_1.LOCAL_ADDRESSES].includes(externalIP),
            };
        });
    }
}
exports.AngularServeRunner = AngularServeRunner;
class AngularServeCLI extends serve_1.ServeCLI {
    constructor() {
        super(...arguments);
        this.name = 'Angular CLI';
        this.pkg = '@angular/cli';
        this.program = 'ng';
        this.prefix = 'ng';
        this.script = serve_1.SERVE_SCRIPT;
        this.chunks = 0;
    }
    serve(options) {
        const _super = Object.create(null, {
            serve: { get: () => super.serve }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.on('compile', chunks => {
                if (chunks > 0) {
                    this.e.log.info(`... and ${chalk_1.default.bold(chunks.toString())} additional chunks`);
                }
            });
            return _super.serve.call(this, options);
        });
    }
    stdoutFilter(line) {
        if (this.resolvedProgram !== this.program) {
            return super.stdoutFilter(line);
        }
        const strippedLine = format_1.stripAnsi(line);
        if (strippedLine.includes('Development Server is listening')) {
            this.emit('ready');
            return false;
        }
        if (strippedLine.match(/.*chunk\s{\d+}.+/)) {
            this.chunks++;
            return false;
        }
        if (strippedLine.includes('Compiled successfully')) {
            this.emit('compile', this.chunks);
            this.chunks = 0;
        }
        return true;
    }
    buildArgs(options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { pkgManagerArgs } = yield Promise.resolve().then(() => require('../../utils/npm'));
            const args = yield this.serveOptionsToNgArgs(options);
            if (this.resolvedProgram === this.program) {
                return [...this.buildArchitectCommand(options), ...args];
            }
            else {
                const [, ...pkgArgs] = yield pkgManagerArgs(this.e.config.get('npmClient'), { command: 'run', script: this.script, scriptArgs: [...args] });
                return pkgArgs;
            }
        });
    }
    serveOptionsToNgArgs(options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const args = {
                _: [],
                host: options.address,
                port: options.port ? options.port.toString() : undefined,
                'source-map': options.sourcemaps !== false ? options.sourcemaps : 'false',
                'ssl': options.ssl !== false ? options.ssl : 'false',
            };
            const projectArgs = [];
            let separatedArgs = options['--'];
            if (options.engine === 'cordova') {
                const integration = this.e.project.requireIntegration('cordova');
                args.platform = options.platform;
                if (this.e.project.rootDirectory !== integration.root) {
                    args.cordovaBasePath = integration.root;
                }
                separatedArgs = [];
                if (options.devapp) {
                    args.cordovaMock = true;
                }
            }
            if (this.resolvedProgram !== this.program) {
                if (options.configuration) {
                    projectArgs.push(`--configuration=${options.configuration}`);
                }
                if (options.project) {
                    projectArgs.push(`--project=${options.project}`);
                }
            }
            return [...cli_framework_1.unparseArgs(args), ...projectArgs, ...separatedArgs];
        });
    }
    buildArchitectCommand(options) {
        const cmd = options.engine === 'cordova' ? 'ionic-cordova-serve' : 'serve';
        const project = options.project ? options.project : 'app';
        return ['run', `${project}:${cmd}${options.configuration ? `:${options.configuration}` : ''}`];
    }
}
exports.AngularServeCLI = AngularServeCLI;
