"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const fn_1 = require("@ionic/cli-framework/utils/fn");
const format_1 = require("@ionic/cli-framework/utils/format");
const node_1 = require("@ionic/cli-framework/utils/node");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const Debug = require("debug");
const lodash = require("lodash");
const path = require("path");
const constants_1 = require("../../constants");
const guards_1 = require("../../guards");
const errors_1 = require("../errors");
const integrations_1 = require("../integrations");
const debug = Debug('ionic:lib:project');
class ProjectDetailsError extends errors_1.BaseException {
    constructor(msg, 
    /**
     * Unique code for this error.
     */
    code, 
    /**
     * The underlying error that caused this error.
     */
    error) {
        super(msg);
        this.code = code;
        this.error = error;
    }
}
exports.ProjectDetailsError = ProjectDetailsError;
class ProjectDetails {
    constructor({ rootDirectory, args = { _: [] }, e }) {
        this.rootDirectory = rootDirectory;
        this.e = e;
        this.args = args;
    }
    getIdFromArgs() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const id = this.args && this.args['project'] ? String(this.args['project']) : undefined;
            if (id) {
                debug(`Project id from args: ${chalk_1.default.bold(id)}`);
                return id;
            }
        });
    }
    getIdFromPathMatch(config) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { ctx } = this.e;
            for (const [key, value] of lodash.entries(config.projects)) {
                const id = key;
                if (value && value.root) {
                    const projectDir = path.resolve(this.rootDirectory, value.root);
                    if (ctx.execPath.startsWith(projectDir)) {
                        debug(`Project id from path match: ${chalk_1.default.bold(id)}`);
                        return id;
                    }
                }
            }
        });
    }
    getIdFromDefaultProject(config) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const id = config.defaultProject;
            if (id) {
                debug(`Project id from defaultProject: ${chalk_1.default.bold(id)}`);
                return id;
            }
        });
    }
    getTypeFromConfig(config) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { type } = config;
            if (type) {
                debug(`Project type from config: ${chalk_1.default.bold(prettyProjectName(type))} ${type ? chalk_1.default.bold(`(${type})`) : ''}`);
                return type;
            }
        });
    }
    getTypeFromDetection() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            for (const projectType of constants_1.PROJECT_TYPES) {
                const p = yield createProjectFromDetails({ context: 'app', configPath: path.resolve(this.rootDirectory, constants_1.PROJECT_FILE), type: projectType, errors: [] }, this.e);
                const type = p.type;
                if (yield p.detected()) {
                    debug(`Project type from detection: ${chalk_1.default.bold(prettyProjectName(type))} ${type ? chalk_1.default.bold(`(${type})`) : ''}`);
                    return type;
                }
            }
        });
    }
    determineSingleApp(config) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const errors = [];
            let type = yield fn_1.resolveValue(() => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.getTypeFromConfig(config); }), () => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.getTypeFromDetection(); }));
            if (!type) {
                errors.push(new ProjectDetailsError('Could not determine project type', 'ERR_MISSING_PROJECT_TYPE'));
            }
            else if (!constants_1.PROJECT_TYPES.includes(type)) {
                errors.push(new ProjectDetailsError(`Invalid project type: ${type}`, 'ERR_INVALID_PROJECT_TYPE'));
                type = undefined;
            }
            return { context: 'app', type, errors };
        });
    }
    determineMultiApp(config) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const errors = [];
            const id = yield fn_1.resolveValue(() => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.getIdFromArgs(); }), () => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.getIdFromPathMatch(config); }), () => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.getIdFromDefaultProject(config); }));
            let type;
            if (id) {
                const app = config.projects[id];
                if (app) {
                    const r = yield this.determineSingleApp(app);
                    type = r.type;
                    errors.push(...r.errors);
                }
                else {
                    errors.push(new ProjectDetailsError('Could not find project in config', 'ERR_MULTI_MISSING_CONFIG'));
                }
            }
            else {
                errors.push(new ProjectDetailsError('Could not determine project id', 'ERR_MULTI_MISSING_ID'));
            }
            return { context: 'multiapp', id, type, errors };
        });
    }
    processResult(result) {
        const { log } = this.e;
        const errorCodes = result.errors.map(e => e.code);
        const e1 = result.errors.find(e => e.code === 'ERR_INVALID_PROJECT_FILE');
        const e2 = result.errors.find(e => e.code === 'ERR_INVALID_PROJECT_TYPE');
        if (e1) {
            log.error(`Error while loading config (project config: ${chalk_1.default.bold(format_1.prettyPath(result.configPath))})\n` +
                `${e1.error ? `${e1.message}: ${chalk_1.default.red(e1.error.toString())}` : chalk_1.default.red(e1.message)}. ` +
                `Run ${chalk_1.default.green('ionic init')} to re-initialize your Ionic project. Without a valid project config, the CLI will not have project context.`);
            log.nl();
        }
        if (result.context === 'multiapp') {
            if (errorCodes.includes('ERR_MULTI_MISSING_ID')) {
                log.warn(`Multi-app workspace detected, but cannot determine which project to use.\n` +
                    `Please set a ${chalk_1.default.green('defaultProject')} in ${chalk_1.default.bold(format_1.prettyPath(result.configPath))} or specify the project using the global ${chalk_1.default.green('--project')} option. Read the documentation${chalk_1.default.cyan('[1]')} for more information.\n\n` +
                    `${chalk_1.default.cyan('[1]')}: ${chalk_1.default.bold('https://beta.ionicframework.com/docs/cli/configuration#multi-app-projects')}`);
                log.nl();
            }
            if (result.id && errorCodes.includes('ERR_MULTI_MISSING_CONFIG')) {
                log.warn(`Multi-app workspace detected, but project was not found in configuration.\n` +
                    `Project ${chalk_1.default.green(result.id)} could not be found in the workspace. Did you add it to ${chalk_1.default.bold(format_1.prettyPath(result.configPath))}?`);
            }
        }
        if (errorCodes.includes('ERR_MISSING_PROJECT_TYPE')) {
            const listWrapOptions = { width: format_1.TTY_WIDTH - 8 - 3, indentation: 1 };
            log.warn(`Could not determine project type (project config: ${chalk_1.default.bold(format_1.prettyPath(result.configPath))}).\n` +
                `- ${format_1.wordWrap(`For ${chalk_1.default.bold(prettyProjectName('angular'))} projects, make sure ${chalk_1.default.green('@ionic/angular')} is listed as a dependency in ${chalk_1.default.bold('package.json')}.`, listWrapOptions)}\n` +
                `- ${format_1.wordWrap(`For ${chalk_1.default.bold(prettyProjectName('ionic-angular'))} projects, make sure ${chalk_1.default.green('ionic-angular')} is listed as a dependency in ${chalk_1.default.bold('package.json')}.`, listWrapOptions)}\n` +
                `- ${format_1.wordWrap(`For ${chalk_1.default.bold(prettyProjectName('ionic1'))} projects, make sure ${chalk_1.default.green('ionic')} is listed as a dependency in ${chalk_1.default.bold('bower.json')}.`, listWrapOptions)}\n\n` +
                `Alternatively, set ${chalk_1.default.bold('type')} attribute in ${chalk_1.default.bold(format_1.prettyPath(result.configPath))} to one of: ${constants_1.PROJECT_TYPES.map(v => chalk_1.default.green(v)).join(', ')}.\n\n` +
                `If the Ionic CLI does not know what type of project this is, ${chalk_1.default.green('ionic build')}, ${chalk_1.default.green('ionic serve')}, and other commands may not work. You can use the ${chalk_1.default.green('custom')} project type if that's okay.`);
            log.nl();
        }
        if (e2) {
            log.error(`${e2.message} (project config: ${chalk_1.default.bold(format_1.prettyPath(result.configPath))}).\n` +
                `Project type must be one of: ${constants_1.PROJECT_TYPES.map(v => chalk_1.default.green(v)).join(', ')}`);
            log.nl();
        }
    }
    readConfig(p) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                let configContents = yield utils_fs_1.readFile(p, { encoding: 'utf8' });
                if (!configContents) {
                    configContents = '{}\n';
                    yield utils_fs_1.writeFile(p, configContents, { encoding: 'utf8' });
                }
                return yield JSON.parse(configContents);
            }
            catch (e) {
                throw new ProjectDetailsError('Could not read project file', 'ERR_INVALID_PROJECT_FILE', e);
            }
        });
    }
    /**
     * Gather project details from specified configuration.
     *
     * This method will always resolve with a result object, with an array of
     * errors. Use `processResult()` to log warnings & errors.
     */
    result() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const errors = [];
            const configPath = path.resolve(this.rootDirectory, constants_1.PROJECT_FILE);
            let config;
            try {
                config = yield this.readConfig(configPath);
                if (guards_1.isProjectConfig(config)) {
                    const r = yield this.determineSingleApp(config);
                    errors.push(...r.errors);
                    return Object.assign({ configPath, errors }, r);
                }
                if (guards_1.isMultiProjectConfig(config)) {
                    const r = yield this.determineMultiApp(config);
                    errors.push(...r.errors);
                    return Object.assign({ configPath, errors }, r);
                }
                throw new ProjectDetailsError('Unknown project file structure', 'ERR_INVALID_PROJECT_FILE');
            }
            catch (e) {
                errors.push(e);
            }
            return { configPath, context: 'unknown', errors };
        });
    }
}
exports.ProjectDetails = ProjectDetails;
function createProjectFromDetails(details, deps) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const { context, type } = details;
        switch (type) {
            case 'angular':
                const { AngularProject } = yield Promise.resolve().then(() => require('./angular'));
                return new AngularProject(details, deps);
            case 'ionic-angular':
                const { IonicAngularProject } = yield Promise.resolve().then(() => require('./ionic-angular'));
                return new IonicAngularProject(details, deps);
            case 'ionic1':
                const { Ionic1Project } = yield Promise.resolve().then(() => require('./ionic1'));
                return new Ionic1Project(details, deps);
            case 'custom':
                const { CustomProject } = yield Promise.resolve().then(() => require('./custom'));
                return new CustomProject(details, deps);
        }
        // If we can't match any of the types above, but we've detected a multi-app
        // setup, it likely means this is a "bare" project, or a project without
        // apps. This can occur when `ionic start` is used for the first time in a
        // new multi-app setup.
        if (context === 'multiapp') {
            const { BareProject } = yield Promise.resolve().then(() => require('./bare'));
            return new BareProject(details, deps);
        }
        throw new errors_1.FatalException(`Bad project type: ${chalk_1.default.bold(String(type))}`); // TODO?
    });
}
exports.createProjectFromDetails = createProjectFromDetails;
function findProjectDirectory(cwd) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        return utils_fs_1.findBaseDirectory(cwd, constants_1.PROJECT_FILE);
    });
}
exports.findProjectDirectory = findProjectDirectory;
function createProjectFromDirectory(rootDirectory, args, deps, { logErrors = true } = {}) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const details = new ProjectDetails({ rootDirectory, args, e: deps });
        const result = yield details.result();
        debug('Project details: %o', Object.assign({}, result, { errors: result.errors.map(e => e.code) }));
        if (logErrors) {
            details.processResult(result);
        }
        if (result.context === 'unknown') {
            return;
        }
        return createProjectFromDetails(result, deps);
    });
}
exports.createProjectFromDirectory = createProjectFromDirectory;
class ProjectConfig extends cli_framework_1.BaseConfig {
    constructor(p, _a = {}) {
        var { type } = _a, options = tslib_1.__rest(_a, ["type"]);
        super(p, options);
        this.type = type;
        const c = this.c;
        if (typeof c.app_id === 'string') { // <4.0.0 project config migration
            if (c.app_id && !c.id) {
                // set `id` only if it has not been previously set and if `app_id`
                // isn't an empty string (which it used to be, sometimes)
                this.set('id', c.app_id);
            }
            this.unset('app_id');
        }
        else if (typeof c.pro_id === 'string') {
            if (!c.id) {
                // set `id` only if it has not been previously set
                this.set('id', c.pro_id);
            }
            // we do not unset `pro_id` because it would break things
        }
    }
    provideDefaults(c) {
        return lodash.assign({
            name: 'New Ionic App',
            integrations: {},
            type: this.type,
        }, c);
    }
}
exports.ProjectConfig = ProjectConfig;
class Project {
    constructor(details, e) {
        this.details = details;
        this.e = e;
        this.rootDirectory = path.dirname(details.configPath);
    }
    get filePath() {
        return this.details.configPath;
    }
    get directory() {
        const root = this.config.get('root');
        if (!root) {
            return this.rootDirectory;
        }
        return path.resolve(this.rootDirectory, root);
    }
    get config() {
        const id = this.details.context === 'multiapp' ? this.details.id : undefined;
        const options = { type: this.type, pathPrefix: id ? ['projects', id] : [] };
        return new ProjectConfig(this.filePath, options);
    }
    getBuildRunner() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                return yield this.requireBuildRunner();
            }
            catch (e) {
                if (!(e instanceof errors_1.RunnerNotFoundException)) {
                    throw e;
                }
            }
        });
    }
    getServeRunner() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                return yield this.requireServeRunner();
            }
            catch (e) {
                if (!(e instanceof errors_1.RunnerNotFoundException)) {
                    throw e;
                }
            }
        });
    }
    getGenerateRunner() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                return yield this.requireGenerateRunner();
            }
            catch (e) {
                if (!(e instanceof errors_1.RunnerNotFoundException)) {
                    throw e;
                }
            }
        });
    }
    requireAppflowId() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const appflowId = this.config.get('id');
            if (!appflowId) {
                throw new errors_1.FatalException(`Your project file (${chalk_1.default.bold(format_1.prettyPath(this.filePath))}) does not contain '${chalk_1.default.bold('id')}'. ` +
                    `Run ${chalk_1.default.green('ionic link')}.`);
            }
            return appflowId;
        });
    }
    get packageJsonPath() {
        return path.resolve(this.directory, 'package.json');
    }
    getPackageJson(pkgName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let pkg;
            let pkgPath;
            try {
                pkgPath = pkgName ? node_1.resolve(`${pkgName}/package`, { paths: node_1.compileNodeModulesPaths(this.directory) }) : this.packageJsonPath;
                pkg = yield node_1.readPackageJsonFile(pkgPath);
            }
            catch (e) {
                this.e.log.error(`Error loading ${chalk_1.default.bold(pkgName ? pkgName : `project's`)} ${chalk_1.default.bold('package.json')}: ${e}`);
            }
            return [pkg, pkgPath ? path.dirname(pkgPath) : undefined];
        });
    }
    requirePackageJson(pkgName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const pkgPath = pkgName ? node_1.resolve(`${pkgName}/package`, { paths: node_1.compileNodeModulesPaths(this.directory) }) : this.packageJsonPath;
                return yield node_1.readPackageJsonFile(pkgPath);
            }
            catch (e) {
                if (e instanceof SyntaxError) {
                    throw new errors_1.FatalException(`Could not parse ${chalk_1.default.bold(pkgName ? pkgName : `project's`)} ${chalk_1.default.bold('package.json')}. Is it a valid JSON file?`);
                }
                else if (e === node_1.ERROR_INVALID_PACKAGE_JSON) {
                    throw new errors_1.FatalException(`The ${chalk_1.default.bold(pkgName ? pkgName : `project's`)} ${chalk_1.default.bold('package.json')} file seems malformed.`);
                }
                throw e; // Probably file not found
            }
        });
    }
    getDocsUrl() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return 'https://ion.link/docs';
        });
    }
    getSourceDir() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return path.resolve(this.directory, 'src');
        });
    }
    getDistDir() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return path.resolve(this.directory, 'www');
        });
    }
    getInfo() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const integrations = yield this.getIntegrations();
            const integrationInfo = lodash.flatten(yield Promise.all(integrations.map((i) => tslib_1.__awaiter(this, void 0, void 0, function* () { return i.getInfo(); }))));
            return integrationInfo;
        });
    }
    personalize(details) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { name, projectId, description, version } = details;
            this.config.set('name', name);
            const pkg = yield this.requirePackageJson();
            pkg.name = projectId;
            pkg.version = version ? version : '0.0.1';
            pkg.description = description ? description : 'An Ionic project';
            yield utils_fs_1.writeJson(this.packageJsonPath, pkg);
            const integrations = yield this.getIntegrations();
            yield Promise.all(integrations.map((i) => tslib_1.__awaiter(this, void 0, void 0, function* () { return i.personalize(details); })));
        });
    }
    registerAilments(registry) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ailments = yield Promise.resolve().then(() => require('../doctor/ailments'));
            const deps = Object.assign({}, this.e, { project: this });
            registry.register(new ailments.NpmInstalledLocally(deps));
            registry.register(new ailments.IonicCLIInstalledLocally(deps));
            registry.register(new ailments.GitNotUsed(deps));
            registry.register(new ailments.GitConfigInvalid(deps));
            registry.register(new ailments.IonicNativeOldVersionInstalled(deps));
            registry.register(new ailments.UnsavedCordovaPlatforms(deps));
            registry.register(new ailments.DefaultCordovaBundleIdUsed(deps));
            registry.register(new ailments.ViewportFitNotSet(deps));
            registry.register(new ailments.CordovaPlatformsCommitted(deps));
        });
    }
    createIntegration(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return integrations_1.BaseIntegration.createFromName({
                config: this.e.config,
                project: this,
                shell: this.e.shell,
                log: this.e.log,
            }, name);
        });
    }
    getIntegration(name) {
        const integration = this.config.get('integrations')[name];
        if (integration) {
            return {
                enabled: integration.enabled !== false,
                root: integration.root === undefined ? this.directory : path.resolve(this.rootDirectory, integration.root),
            };
        }
    }
    requireIntegration(name) {
        const id = this.details.context === 'multiapp' ? this.details.id : undefined;
        const integration = this.getIntegration(name);
        if (!integration) {
            throw new errors_1.FatalException(`Could not find ${chalk_1.default.bold(name)} integration in the ${chalk_1.default.bold(id ? id : 'default')} project.`);
        }
        if (!integration.enabled) {
            throw new errors_1.FatalException(`${chalk_1.default.bold(name)} integration is disabled in the ${chalk_1.default.bold(id ? id : 'default')} project.`);
        }
        return integration;
    }
    getIntegrations() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const integrationsFromConfig = this.config.get('integrations');
            const names = Object.keys(integrationsFromConfig); // TODO
            const integrationNames = names.filter(n => {
                const c = integrationsFromConfig[n];
                return c && c.enabled !== false;
            });
            const integrations = yield Promise.all(integrationNames.map((name) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                try {
                    return yield this.createIntegration(name);
                }
                catch (e) {
                    if (!(e instanceof errors_1.IntegrationNotFoundException)) {
                        throw e;
                    }
                    this.e.log.warn(e.message);
                }
            })));
            return integrations.filter((i) => typeof i !== 'undefined');
        });
    }
}
exports.Project = Project;
function prettyProjectName(type) {
    if (!type) {
        return 'Unknown';
    }
    if (type === 'angular') {
        return '@ionic/angular';
    }
    else if (type === 'ionic-angular') {
        return 'Ionic 2/3';
    }
    else if (type === 'ionic1') {
        return 'Ionic 1';
    }
    return type;
}
exports.prettyProjectName = prettyProjectName;
function isValidProjectId(projectId) {
    return projectId !== '.' && node_1.isValidPackageName(projectId) && projectId === path.basename(projectId);
}
exports.isValidProjectId = isValidProjectId;
