"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const chalk_1 = require("chalk");
const generate_1 = require("../../generate");
const app_scripts_1 = require("./app-scripts");
const GENERATOR_TYPES = ['component', 'directive', 'page', 'pipe', 'provider', 'tabs'];
class IonicAngularGenerateRunner extends generate_1.GenerateRunner {
    constructor(e) {
        super();
        this.e = e;
    }
    getCommandMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                groups: [],
                summary: `Generate pipes, components, pages, directives, providers, and tabs`,
                description: `
Automatically create components for your Ionic app.

The given ${chalk_1.default.green('name')} is normalized into an appropriate naming convention. For example, ${chalk_1.default.green('ionic generate page neat')} creates a page by the name of ${chalk_1.default.green('NeatPage')} in ${chalk_1.default.green('src/pages/neat/')}.
      `,
                exampleCommands: [
                    '',
                    ...GENERATOR_TYPES,
                    'component foo',
                    'page Login',
                    'page Detail --no-module',
                    'page About --constants',
                    'pipe MyFilterPipe',
                ],
                inputs: [
                    {
                        name: 'type',
                        summary: `The type of generator (e.g. ${GENERATOR_TYPES.map(t => chalk_1.default.green(t)).join(', ')})`,
                        validators: [cli_framework_1.validators.required, cli_framework_1.contains(GENERATOR_TYPES, {})],
                    },
                    {
                        name: 'name',
                        summary: 'The name of the component being generated',
                        validators: [cli_framework_1.validators.required],
                    },
                ],
                options: [
                    {
                        name: 'module',
                        summary: 'Do not generate an NgModule for the component',
                        type: Boolean,
                        default: true,
                    },
                    {
                        name: 'constants',
                        summary: 'Generate a page constant file for lazy-loaded pages',
                        type: Boolean,
                        default: false,
                    },
                ],
            };
        });
    }
    ensureCommandLine(inputs, options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!inputs[0]) {
                const generatorType = yield this.e.prompt({
                    type: 'list',
                    name: 'generatorType',
                    message: 'What would you like to generate:',
                    choices: GENERATOR_TYPES,
                });
                inputs[0] = generatorType;
            }
            if (!inputs[1]) {
                const generatorName = yield this.e.prompt({
                    type: 'input',
                    name: 'generatorName',
                    message: 'What should the name be?',
                    validate: v => cli_framework_1.validators.required(v),
                });
                inputs[1] = generatorName;
            }
        });
    }
    createOptionsFromCommandLine(inputs, options) {
        const baseOptions = super.createOptionsFromCommandLine(inputs, options);
        return Object.assign({}, baseOptions, { module: options['module'] ? true : false, constants: options['constants'] ? true : false });
    }
    run(options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const AppScripts = yield app_scripts_1.importAppScripts(this.e.project.directory);
            const appScriptsArgs = cli_framework_1.unparseArgs({ _: [], module: options.module, constants: options.constants }, { useEquals: false, ignoreFalse: true, allowCamelCase: true });
            AppScripts.setProcessArgs(['node', 'appscripts'].concat(appScriptsArgs));
            AppScripts.setCwd(this.e.project.directory);
            const context = AppScripts.generateContext();
            switch (options.type) {
                case 'page':
                    yield AppScripts.processPageRequest(context, options.name, options);
                    break;
                case 'component':
                    const componentData = yield this.getModules(context, 'component');
                    yield AppScripts.processComponentRequest(context, options.name, componentData);
                    break;
                case 'directive':
                    const directiveData = yield this.getModules(context, 'directive');
                    yield AppScripts.processDirectiveRequest(context, options.name, directiveData);
                    break;
                case 'pipe':
                    const pipeData = yield this.getModules(context, 'pipe');
                    yield AppScripts.processPipeRequest(context, options.name, pipeData);
                    break;
                case 'provider':
                    const providerData = context.appNgModulePath;
                    yield AppScripts.processProviderRequest(context, options.name, providerData);
                    break;
                case 'tabs':
                    const tabsData = yield this.tabsPrompt();
                    yield AppScripts.processTabsRequest(context, options.name, tabsData, options);
                    break;
            }
            this.e.log.ok(`Generated a ${chalk_1.default.bold(options.type)}${options.type === 'tabs' ? ' page' : ''} named ${chalk_1.default.bold(options.name)}!`);
        });
    }
    tabsPrompt() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const tabNames = [];
            const howMany = yield this.e.prompt({
                type: 'input',
                name: 'howMany',
                message: 'How many tabs?',
                validate: v => cli_framework_1.validators.numeric(v),
            });
            for (let i = 0; i < parseInt(howMany, 10); i++) {
                const tabName = yield this.e.prompt({
                    type: 'input',
                    name: 'tabName',
                    message: 'Name of this tab:',
                });
                tabNames.push(tabName);
            }
            return tabNames;
        });
    }
    getModules(context, kind) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            switch (kind) {
                case 'component':
                    return context.componentsNgModulePath ? context.componentsNgModulePath : context.appNgModulePath;
                case 'pipe':
                    return context.pipesNgModulePath ? context.pipesNgModulePath : context.appNgModulePath;
                case 'directive':
                    return context.directivesNgModulePath ? context.directivesNgModulePath : context.appNgModulePath;
            }
        });
    }
}
exports.IonicAngularGenerateRunner = IonicAngularGenerateRunner;
