"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_framework_1 = require("@ionic/cli-framework");
const string_1 = require("@ionic/cli-framework/utils/string");
const chalk_1 = require("chalk");
const Debug = require("debug");
const serve_1 = require("../../serve");
const common_1 = require("../common");
const app_scripts_1 = require("./app-scripts");
const debug = Debug('ionic:lib:project:ionic-angular:serve');
const DEFAULT_PROGRAM = 'ionic-app-scripts';
exports.DEFAULT_SERVE_SCRIPT_VALUE = `${DEFAULT_PROGRAM} serve`;
class IonicAngularServeRunner extends serve_1.ServeRunner {
    constructor(e) {
        super();
        this.e = e;
    }
    getCommandMetadata() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return {
                options: [
                    {
                        name: 'consolelogs',
                        summary: 'Print app console logs to Ionic CLI',
                        type: Boolean,
                        groups: ['cordova'],
                        aliases: ['c'],
                        hint: chalk_1.default.dim('[app-scripts]'),
                    },
                    {
                        name: 'serverlogs',
                        summary: 'Print dev server logs to Ionic CLI',
                        type: Boolean,
                        aliases: ['s'],
                        groups: [cli_framework_1.OptionGroup.Hidden, 'cordova'],
                        hint: chalk_1.default.dim('[app-scripts]'),
                    },
                    {
                        name: 'devapp',
                        summary: 'Do not publish DevApp service',
                        type: Boolean,
                        default: true,
                        groups: [cli_framework_1.OptionGroup.Advanced],
                    },
                    {
                        name: 'livereload-port',
                        summary: 'Use specific port for live-reload',
                        default: serve_1.DEFAULT_LIVERELOAD_PORT.toString(),
                        aliases: ['r'],
                        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova'],
                        hint: chalk_1.default.dim('[app-scripts]'),
                        spec: { value: 'port' },
                    },
                    {
                        name: 'dev-logger-port',
                        summary: 'Use specific port for dev server',
                        default: serve_1.DEFAULT_DEV_LOGGER_PORT.toString(),
                        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova'],
                        hint: chalk_1.default.dim('[app-scripts]'),
                        spec: { value: 'port' },
                    },
                    {
                        name: 'proxy',
                        summary: 'Do not add proxies',
                        type: Boolean,
                        default: true,
                        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova'],
                        hint: chalk_1.default.dim('[app-scripts]'),
                    },
                    {
                        name: 'source-map',
                        summary: 'Output sourcemaps',
                        type: Boolean,
                        groups: [cli_framework_1.OptionGroup.Advanced, 'cordova'],
                        hint: chalk_1.default.dim('[app-scripts]'),
                    },
                    ...app_scripts_1.APP_SCRIPTS_OPTIONS,
                ],
                exampleCommands: [
                    '-c', '-- --enableLint false',
                ],
            };
        });
    }
    createOptionsFromCommandLine(inputs, options) {
        const baseOptions = super.createOptionsFromCommandLine(inputs, options);
        const sourcemaps = typeof options['source-map'] === 'boolean' ? Boolean(options['source-map']) : undefined;
        const livereloadPort = string_1.str2num(options['livereload-port'], serve_1.DEFAULT_LIVERELOAD_PORT);
        const notificationPort = string_1.str2num(options['dev-logger-port'], serve_1.DEFAULT_DEV_LOGGER_PORT);
        return Object.assign({}, baseOptions, { sourcemaps, consolelogs: options['consolelogs'] ? true : false, serverlogs: options['serverlogs'] ? true : false, livereloadPort,
            notificationPort, env: options['env'] ? String(options['env']) : undefined });
    }
    modifyOpenURL(url, options) {
        return `${url}${options.browserOption ? options.browserOption : ''}${options.platform ? `?ionicplatform=${options.platform}` : ''}`;
    }
    serveProject(options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [externalIP, availableInterfaces] = yield this.selectExternalIP(options);
            const { port, livereloadPort, notificationPort } = yield common_1.findOpenIonicPorts(options.address, options);
            options.port = port;
            options.livereloadPort = livereloadPort;
            options.notificationPort = notificationPort;
            const appscripts = new IonicAngularServeCLI(this.e);
            yield appscripts.serve(options);
            return {
                custom: appscripts.resolvedProgram !== appscripts.program,
                protocol: 'http',
                localAddress: 'localhost',
                externalAddress: externalIP,
                externalNetworkInterfaces: availableInterfaces,
                port,
                externallyAccessible: ![serve_1.BIND_ALL_ADDRESS, ...serve_1.LOCAL_ADDRESSES].includes(externalIP),
            };
        });
    }
}
exports.IonicAngularServeRunner = IonicAngularServeRunner;
class IonicAngularServeCLI extends serve_1.ServeCLI {
    constructor() {
        super(...arguments);
        this.name = 'Ionic App Scripts';
        this.pkg = '@ionic/app-scripts';
        this.program = DEFAULT_PROGRAM;
        this.prefix = 'app-scripts';
        this.script = serve_1.SERVE_SCRIPT;
    }
    stdoutFilter(line) {
        if (this.resolvedProgram !== this.program) {
            return super.stdoutFilter(line);
        }
        if (line.includes('server running')) {
            this.emit('ready');
            return false;
        }
        return true;
    }
    buildArgs(options) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { pkgManagerArgs } = yield Promise.resolve().then(() => require('../../utils/npm'));
            const args = this.serveOptionsToAppScriptsArgs(options);
            if (this.resolvedProgram === this.program) {
                return ['serve', ...args];
            }
            else {
                const [, ...pkgArgs] = yield pkgManagerArgs(this.e.config.get('npmClient'), { command: 'run', script: this.script, scriptArgs: [...args] });
                return pkgArgs;
            }
        });
    }
    serveOptionsToAppScriptsArgs(options) {
        const args = {
            _: [],
            address: options.address,
            port: String(options.port),
            'livereload-port': String(options.livereloadPort),
            'dev-logger-port': String(options.notificationPort),
            consolelogs: options.consolelogs,
            serverlogs: options.serverlogs,
            nobrowser: true,
            nolivereload: !options.livereload,
            noproxy: !options.proxy,
            iscordovaserve: options.engine === 'cordova',
            generateSourceMap: typeof options.sourcemaps !== 'undefined' ? options.sourcemaps ? 'true' : 'false' : undefined,
            platform: options.platform,
            target: options.engine === 'cordova' ? 'cordova' : undefined,
            env: options.env,
        };
        return [...cli_framework_1.unparseArgs(args, { allowCamelCase: true, useEquals: false }), ...options['--']];
    }
    resolveProgram() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (typeof this.script !== 'undefined') {
                debug(`Looking for ${chalk_1.default.cyan(this.script)} npm script.`);
                const pkg = yield this.e.project.requirePackageJson();
                if (pkg.scripts && pkg.scripts[this.script]) {
                    if (pkg.scripts[this.script] === exports.DEFAULT_SERVE_SCRIPT_VALUE) {
                        debug(`Found ${chalk_1.default.cyan(this.script)}, but it is the default. Not running.`);
                    }
                    else {
                        debug(`Using ${chalk_1.default.cyan(this.script)} npm script.`);
                        return this.e.config.get('npmClient');
                    }
                }
            }
            return this.program;
        });
    }
}
