"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const format_1 = require("@ionic/cli-framework/utils/format");
const utils_fs_1 = require("@ionic/utils-fs");
const chalk_1 = require("chalk");
const Debug = require("debug");
const lodash = require("lodash");
const path = require("path");
const __1 = require("../");
const errors_1 = require("../../errors");
const debug = Debug('ionic:lib:project:angular');
exports.ERROR_INVALID_BOWER_JSON = 'INVALID_BOWER_JSON';
function isBowerJson(obj) {
    return obj && typeof obj.name === 'string';
}
function readBowerJsonFile(p) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const bowerJson = yield utils_fs_1.readJson(p);
        if (!isBowerJson(bowerJson)) {
            throw exports.ERROR_INVALID_BOWER_JSON;
        }
        return bowerJson;
    });
}
class Ionic1Project extends __1.Project {
    constructor() {
        super(...arguments);
        this.type = 'ionic1';
    }
    getInfo() {
        const _super = Object.create(null, {
            getInfo: { get: () => super.getInfo }
        });
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const [ionic1Version, [v1ToolkitPkg],] = yield Promise.all([
                this.getFrameworkVersion(),
                this.getPackageJson('@ionic/v1-toolkit'),
            ]);
            return [
                ...(yield _super.getInfo.call(this)),
                { group: 'ionic', key: 'Ionic Framework', value: ionic1Version ? `ionic1 ${ionic1Version}` : 'unknown' },
                { group: 'ionic', key: '@ionic/v1-toolkit', value: v1ToolkitPkg ? v1ToolkitPkg.version : 'not installed' },
            ];
        });
    }
    detected() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const bwr = yield readBowerJsonFile(path.resolve(this.directory, 'bower.json'));
                const deps = lodash.assign({}, bwr.dependencies, bwr.devDependencies);
                if (typeof deps['ionic'] === 'string') {
                    debug(`${chalk_1.default.bold('ionic')} detected in ${chalk_1.default.bold('bower.json')}`);
                    return true;
                }
            }
            catch (e) {
                // ignore
            }
            return false;
        });
    }
    getSourceDir() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.getDistDir(); // ionic1's source directory is the dist directory
        });
    }
    getDocsUrl() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return 'https://ion.link/v1-docs';
        });
    }
    // this method search not only package.json
    getFrameworkVersion() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ionicVersionFilePath = path.resolve(yield this.getDistDir(), 'lib', 'ionic', 'version.json'); // TODO
            const bowerJsonPath = path.resolve(this.directory, 'bower.json');
            try {
                try {
                    const ionicVersionJson = yield utils_fs_1.readJson(ionicVersionFilePath);
                    return ionicVersionJson['version'];
                }
                catch (e) {
                    const bwr = yield this.loadBowerJson();
                    const deps = lodash.assign({}, bwr.dependencies, bwr.devDependencies);
                    const ionicEntry = deps['ionic'];
                    if (!ionicEntry) {
                        return;
                    }
                    const m = ionicEntry.match(/.+#(.+)/);
                    if (m && m[1]) {
                        return m[1];
                    }
                }
            }
            catch (e) {
                this.e.log.error(`Error with ${chalk_1.default.bold(format_1.prettyPath(bowerJsonPath))} file: ${e}`);
            }
        });
    }
    loadBowerJson() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.bowerJsonFile) {
                const bowerJsonPath = path.resolve(this.directory, 'bower.json');
                try {
                    this.bowerJsonFile = yield readBowerJsonFile(bowerJsonPath);
                }
                catch (e) {
                    if (e instanceof SyntaxError) {
                        throw new errors_1.FatalException(`Could not parse ${chalk_1.default.bold('bower.json')}. Is it a valid JSON file?`);
                    }
                    else if (e === exports.ERROR_INVALID_BOWER_JSON) {
                        throw new errors_1.FatalException(`The ${chalk_1.default.bold('bower.json')} file seems malformed.`);
                    }
                    throw e; // Probably file not found
                }
            }
            return this.bowerJsonFile;
        });
    }
    requireBuildRunner() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { Ionic1BuildRunner } = yield Promise.resolve().then(() => require('./build'));
            const deps = Object.assign({}, this.e, { project: this });
            return new Ionic1BuildRunner(deps);
        });
    }
    requireServeRunner() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { Ionic1ServeRunner } = yield Promise.resolve().then(() => require('./serve'));
            const deps = Object.assign({}, this.e, { project: this });
            return new Ionic1ServeRunner(deps);
        });
    }
    requireGenerateRunner() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            throw new errors_1.RunnerNotFoundException('Generators are not supported in Ionic 1 projects.');
        });
    }
}
exports.Ionic1Project = Ionic1Project;
