"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const chalk_1 = require("chalk");
const guards_1 = require("../guards");
const errors_1 = require("./errors");
const http_1 = require("./http");
class BaseSession {
    constructor(e) {
        this.e = e;
    }
    logout() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.e.config.unset('org.id');
            this.e.config.unset('user.id');
            this.e.config.unset('user.email');
            this.e.config.unset('tokens.user');
            this.e.config.set('git.setup', false);
        });
    }
    isLoggedIn() {
        return typeof this.e.config.get('tokens.user') === 'string';
    }
    getUser() {
        const userId = this.e.config.get('user.id');
        if (!userId) {
            throw new errors_1.SessionException(`Oops, sorry! You'll need to log in:\n    ${chalk_1.default.green('ionic login')}\n\n` +
                `You can create a new account by signing up:\n\n    ${chalk_1.default.green('ionic signup')}\n`);
        }
        return { id: userId };
    }
    getUserToken() {
        const userToken = this.e.config.get('tokens.user');
        if (!userToken) {
            throw new errors_1.SessionException(`Oops, sorry! You'll need to log in:\n    ${chalk_1.default.green('ionic login')}\n\n` +
                `You can create a new account by signing up:\n\n    ${chalk_1.default.green('ionic signup')}\n`);
        }
        return userToken;
    }
}
exports.BaseSession = BaseSession;
class ProSession extends BaseSession {
    login(email, password) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.e.client.make('POST', '/login');
            req.send({ email, password, source: 'cli' });
            try {
                const res = yield this.e.client.do(req);
                if (!guards_1.isLoginResponse(res)) {
                    const data = res.data;
                    if (hasTokenAttribute(data)) {
                        data.token = '*****';
                    }
                    throw new errors_1.FatalException('API request was successful, but the response format was unrecognized.\n' +
                        http_1.formatResponseError(req, res.meta.status, data));
                }
                const { token, user } = res.data;
                if (this.e.config.get('user.id') !== user.id) { // User changed
                    yield this.logout();
                }
                this.e.config.set('user.id', user.id);
                this.e.config.set('user.email', email);
                this.e.config.set('tokens.user', token);
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e) && (e.response.status === 401 || e.response.status === 403)) {
                    throw new errors_1.SessionException('Incorrect email or password.');
                }
                throw e;
            }
        });
    }
    ssoLogin(email) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { AuthClient } = yield Promise.resolve().then(() => require('./auth'));
            const { Auth0OAuth2Flow } = yield Promise.resolve().then(() => require('./sso'));
            const authClient = new AuthClient(this.e);
            const { uuid: connection } = yield authClient.connections.load(email);
            const flow = new Auth0OAuth2Flow({ email, connection }, this.e);
            const token = yield flow.run();
            yield this.tokenLogin(token);
            this.e.config.set('org.id', connection);
        });
    }
    tokenLogin(token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { UserClient } = yield Promise.resolve().then(() => require('./user'));
            const userClient = new UserClient(token, this.e);
            try {
                const user = yield userClient.loadSelf();
                const user_id = user.id;
                if (this.e.config.get('user.id') !== user_id) { // User changed
                    yield this.logout();
                }
                this.e.config.set('user.id', user_id);
                this.e.config.set('user.email', user.email);
                this.e.config.set('tokens.user', token);
            }
            catch (e) {
                if (guards_1.isSuperAgentError(e) && (e.response.status === 401 || e.response.status === 403)) {
                    throw new errors_1.SessionException('Invalid auth token.');
                }
                throw e;
            }
        });
    }
}
exports.ProSession = ProSession;
function promptToLogin(env) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const { validators } = yield Promise.resolve().then(() => require('@ionic/cli-framework'));
        env.log.msg(`Log into your Ionic Appflow account\n` +
            `If you don't have one yet, create yours by running: ${chalk_1.default.green(`ionic signup`)}\n`);
        const email = yield env.prompt({
            type: 'input',
            name: 'email',
            message: 'Email:',
            validate: v => validators.required(v) && validators.email(v),
        });
        const password = yield env.prompt({
            type: 'password',
            name: 'password',
            message: 'Password:',
            mask: '*',
            validate: v => validators.required(v),
        });
        yield env.session.login(email, password);
    });
}
exports.promptToLogin = promptToLogin;
function hasTokenAttribute(r) {
    return r && typeof r === 'object' && typeof r.token === 'string';
}
