"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const Debug = require("debug");
const helper_1 = require("./helper");
const uuid_1 = require("./utils/uuid");
const debug = Debug('ionic:lib:telemetry');
const GA_CODE = 'UA-44023830-30';
let _gaTracker;
class Telemetry {
    constructor({ config, client, getInfo, ctx, project, session }) {
        this.client = client;
        this.config = config;
        this.getInfo = getInfo;
        this.ctx = ctx;
        this.project = project;
        this.session = session;
    }
    sendCommand(command, args) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            debug('Sending telemetry for command: %O %O', command, args);
            yield helper_1.sendMessage({ config: this.config, ctx: this.ctx }, { type: 'telemetry', data: { command, args } });
        });
    }
}
exports.Telemetry = Telemetry;
function getLeek({ config, version }) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (!_gaTracker) {
            const Leek = yield Promise.resolve().then(() => require('leek'));
            let telemetryToken = config.get('tokens.telemetry');
            if (!telemetryToken) {
                telemetryToken = uuid_1.generateUUID();
                config.set('tokens.telemetry', telemetryToken);
                debug(`setting telemetry token to ${telemetryToken}`);
            }
            _gaTracker = new Leek({
                name: telemetryToken,
                trackingCode: GA_CODE,
                globalName: 'ionic',
                version,
                silent: !config.get('telemetry'),
            });
        }
        return _gaTracker;
    });
}
function sendCommand({ config, client, getInfo, ctx, session, project }, command, args) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const messageList = [];
        const name = 'command execution';
        const prettyArgs = args.map(a => a.includes(' ') ? `"${a}"` : a);
        const message = messageList.concat([command], prettyArgs).join(' ');
        yield Promise.all([
            (() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const leek = yield getLeek({ config, version: ctx.version });
                try {
                    yield leek.track({ name, message });
                }
                catch (e) {
                    debug(`leek track error: ${e.stack ? e.stack : e}`);
                }
            }))(),
            (() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const now = new Date().toISOString();
                const appflowId = project ? project.config.get('id') : undefined;
                const { req } = yield client.make('POST', '/events/metrics');
                const metric = {
                    'name': 'cli_command_metrics',
                    'timestamp': now,
                    'session_id': config.get('tokens.telemetry'),
                    'source': 'cli',
                    'value': {
                        'command': command,
                        'arguments': prettyArgs.join(' '),
                        'version': ctx.version,
                        'node_version': process.version,
                        'app_id': appflowId,
                        'backend': 'pro',
                    },
                };
                const isLoggedIn = session.isLoggedIn();
                const info = yield getInfo();
                if (isLoggedIn) {
                    const token = session.getUserToken();
                    req.set('Authorization', `Bearer ${token}`);
                }
                const frameworkInfo = info.find(item => item.key === 'Ionic Framework');
                const npmInfo = info.find(item => item.key === 'npm');
                const osInfo = info.find(item => item.key === 'OS');
                const xcodeInfo = info.find(item => item.key === 'Xcode');
                const androidSdkInfo = info.find(item => item.key === 'Android SDK Tools');
                const cordovaInfo = info.find(item => item.key === 'Cordova CLI');
                const cordovaPlatformsInfo = info.find(item => item.key === 'Cordova Platforms');
                const appScriptsInfo = info.find(item => item.key === '@ionic/app-scripts');
                if (frameworkInfo) {
                    metric['value']['framework'] = frameworkInfo.value;
                }
                if (npmInfo) {
                    metric['value']['npm_version'] = npmInfo.value;
                }
                if (osInfo) {
                    metric['value']['os'] = osInfo.value;
                }
                if (xcodeInfo) {
                    metric['value']['xcode_version'] = xcodeInfo.value;
                }
                if (androidSdkInfo) {
                    metric['value']['android_sdk_version'] = androidSdkInfo.value;
                }
                if (cordovaInfo) {
                    metric['value']['cordova_version'] = cordovaInfo.value;
                }
                if (cordovaPlatformsInfo) {
                    metric['value']['cordova_platforms'] = cordovaPlatformsInfo.value;
                }
                if (appScriptsInfo) {
                    metric['value']['app_scripts_version'] = appScriptsInfo.value;
                }
                debug('metric: %o', metric);
                req.send({
                    'metrics': [metric],
                    'sent_at': now,
                });
                try {
                    yield client.do(req);
                }
                catch (e) {
                    debug(`metric send error: ${e.stack ? e.stack : e}`);
                }
            }))(),
        ]);
    });
}
exports.sendCommand = sendCommand;
