"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const guards_1 = require("../guards");
const http_1 = require("./http");
class UserClient extends http_1.ResourceClient {
    constructor(token, e) {
        super();
        this.token = token;
        this.e = e;
    }
    load(id, modifiers) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.e.client.make('GET', `/users/${id}`);
            this.applyAuthentication(req, this.token);
            this.applyModifiers(req, modifiers);
            const res = yield this.e.client.do(req);
            if (!guards_1.isUserResponse(res)) {
                throw http_1.createFatalAPIFormat(req, res);
            }
            return res.data;
        });
    }
    loadSelf() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.e.client.make('GET', '/users/self');
            this.applyAuthentication(req, this.token);
            const res = yield this.e.client.do(req);
            if (!guards_1.isUserResponse(res)) {
                throw http_1.createFatalAPIFormat(req, res);
            }
            return res.data;
        });
    }
    oAuthGithubLogin(id) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { req } = yield this.e.client.make('POST', `/users/${id}/oauth/github`);
            this.applyAuthentication(req, this.token);
            req.send({ source: 'cli' });
            const res = yield this.e.client.do(req);
            if (!guards_1.isOAuthLoginResponse(res)) {
                throw http_1.createFatalAPIFormat(req, res);
            }
            return res.data.redirect_url;
        });
    }
    paginateGithubRepositories(id) {
        return new http_1.TokenPaginator({
            client: this.e.client,
            reqgen: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const { req } = yield this.e.client.make('GET', `/users/${id}/oauth/github/repositories`);
                req.set('Authorization', `Bearer ${this.token}`);
                return { req };
            }),
            guard: guards_1.isGithubRepoListResponse,
        });
    }
    paginateGithubBranches(userId, repoId) {
        return new http_1.TokenPaginator({
            client: this.e.client,
            reqgen: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const { req } = yield this.e.client.make('GET', `/users/${userId}/oauth/github/repositories/${repoId}/branches`);
                req.set('Authorization', `Bearer ${this.token}`);
                return { req };
            }),
            guard: guards_1.isGithubBranchListResponse,
        });
    }
}
exports.UserClient = UserClient;
