'use strict';

// Load modules

const Lab = require('lab');
const Code = require('code');
const Isemail = require('..');
const Tests = require('./tests.json');


// declare internals

const internals = {
    defaultThreshold: 16
};


// Test shortcuts

const lab = exports.lab = Lab.script();
const describe = lab.describe;
const it = lab.it;
const expect = Code.expect;


// Diagnoses

const diag = Isemail.validate.diagnoses;


// Expectations

const expectations = Tests.map((value) => {

    value[1] = diag[value[1]];
    return value;
});


// Null characters aren't supported in JSON

expectations.push(['test@[\0', diag.errExpectingDTEXT]);
expectations.push(['(\0)test@example.com', diag.errExpectingCTEXT]);


const tldExpectations = [
    ['shouldbe@invalid', diag.errUnknownTLD],
    ['shouldbe@example.com', diag.valid]
];


describe('validate()', () => {

    it('should check options.tldWhitelist', (done) => {

        expect(Isemail.validate('person@top', {
            tldWhitelist: 'top',
            checkDNS: false
        })).to.equal(true);

        expect(Isemail.validate('person@top', {
            tldWhitelist: ['com'],
            checkDNS: false
        })).to.equal(false);

        expect(Isemail.validate('person@top', {
            tldWhitelist: { com: true },
            checkDNS: false
        })).to.equal(false);

        expect(() => {

            Isemail.validate('', {
                tldWhitelist: 77
            });
        }).to.throw(/tldWhitelist/);
        done();
    });

    it('should check options.tldBlacklist', (done) => {

        expect(Isemail.validate('person@top', {
            tldBlacklist: 'top',
            checkDNS: false
        })).to.equal(false);

        expect(Isemail.validate('person@top', {
            tldBlacklist: ['com'],
            checkDNS: false
        })).to.equal(true);

        expect(Isemail.validate('person@top', {
            tldBlacklist: { com: true },
            checkDNS: false
        })).to.equal(true);

        expect(() => {

            Isemail.validate('', {
                tldBlacklist: 77
            });
        }).to.throw(/tldBlacklist/);
        done();
    });

    it('should check options.minDomainAtoms', (done) => {

        expect(() => {

            Isemail.validate('person@top', {
                minDomainAtoms: -1
            });
        }).to.throw(/minDomainAtoms/);

        expect(() => {

            Isemail.validate('person@top', {
                minDomainAtoms: 1.5
            });
        }).to.throw(/minDomainAtoms/);
        done();
    });

    it('should use options.errorLevel', (done) => {

        expect(Isemail.validate('person@123', {
            errorLevel: diag.rfc5321TLDNumeric + 1
        })).to.equal(0);

        expect(Isemail.validate('person@123', {
            errorLevel: diag.rfc5321TLDNumeric
        })).to.equal(diag.rfc5321TLDNumeric);
        done();
    });

    it('should ensure callback provided with checkDNS', (done) => {

        expect(() => {

            Isemail.validate('person@top', {
                checkDNS: true
            });
        }).to.throw(/(?=.*\bcheckDNS\b)(?=.*\bcallback\b)/);
        done();
    });

    it('should handle omitted options', (done) => {

        expect(Isemail.validate(expectations[0][0])).to.equal(expectations[0][1] < internals.defaultThreshold);
        done();
    });

    it('should handle omitted options with callback', (done) => {

        Isemail.validate(expectations[0][0], (res) => {

            expect(res).to.equal(expectations[0][1] < internals.defaultThreshold);
            done();
        });
    });

    expectations.forEach((obj, i) => {

        const email = obj[0], result = obj[1];
        it('should handle test ' + (i + 1), (done) => {

            Isemail.validate(email, {
                errorLevel: 0,
                checkDNS: true
            }, (res) => {

                expect(res).to.equal(result);
                done();
            });
        });
    });

    tldExpectations.forEach((obj, i) => {

        const email = obj[0];
        const result = obj[1];

        it('should handle tld test ' + (i + 1), (done) => {

            expect(Isemail.validate(email, {
                errorLevel: 0,
                tldWhitelist: { com: true }
            })).to.equal(result);

            expect(Isemail.validate(email, {
                errorLevel: 0,
                tldWhitelist: ['com']
            })).to.equal(result);

            expect(Isemail.validate(email, {
                errorLevel: 0,
                tldBlacklist: { invalid: true }
            })).to.equal(result);

            expect(Isemail.validate(email, {
                errorLevel: 0,
                tldBlacklist: ['invalid']
            })).to.equal(result);

            done();
        });
    });

    it('should handle domain atom test 1', (done) => {

        expect(Isemail.validate('shouldbe@invalid', {
            errorLevel: 0,
            minDomainAtoms: 2
        })).to.equal(diag.errDomainTooShort);

        done();
    });

    it('should handle domain atom test 2', (done) => {

        expect(Isemail.validate('valid@example.com', {
            errorLevel: 0,
            minDomainAtoms: 2
        })).to.equal(diag.valid);

        done();
    });
});
