'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var _Map = _interopDefault(require('babel-runtime/core-js/map'));
var path = _interopDefault(require('path'));
var _Promise = _interopDefault(require('babel-runtime/core-js/promise'));
var debug = _interopDefault(require('debug'));
var pify = _interopDefault(require('pify'));
var _extends = _interopDefault(require('babel-runtime/helpers/extends'));
var _Math$sign = _interopDefault(require('babel-runtime/core-js/math/sign'));
var buffer = require('buffer');
var _Object$keys = _interopDefault(require('babel-runtime/core-js/object/keys'));
var ini = _interopDefault(require('ini'));
var get = _interopDefault(require('lodash.get'));
var set = _interopDefault(require('lodash.set'));
var unset = _interopDefault(require('lodash.unset'));
var shasum = _interopDefault(require('shasum'));
var BufferCursor = _interopDefault(require('buffercursor'));
var pad = _interopDefault(require('pad'));
var _Object$values = _interopDefault(require('babel-runtime/core-js/object/values'));
var _Object$assign = _interopDefault(require('babel-runtime/core-js/object/assign'));
var applyDelta = _interopDefault(require('git-apply-delta'));
var listpack = _interopDefault(require('git-list-pack'));
var crc32 = _interopDefault(require('crc/lib/crc32.js'));
var stream = require('stream');
var pako = _interopDefault(require('pako'));
var marky = _interopDefault(require('marky'));
var _Symbol$iterator = _interopDefault(require('babel-runtime/core-js/symbol/iterator'));
var sortby = _interopDefault(require('lodash.sortby'));
var _Set = _interopDefault(require('babel-runtime/core-js/set'));
var AsyncLock = _interopDefault(require('async-lock'));
var ignore = _interopDefault(require('ignore'));
var simpleGet = _interopDefault(require('simple-get'));
var concat = _interopDefault(require('simple-concat'));
var _objectWithoutProperties = _interopDefault(require('babel-runtime/helpers/objectWithoutProperties'));
var _Number$isNaN = _interopDefault(require('babel-runtime/core-js/number/is-nan'));
var through2 = _interopDefault(require('through2'));
var peek = _interopDefault(require('buffer-peek-stream'));
var split2 = _interopDefault(require('split2'));
var createHash = _interopDefault(require('sha.js'));

// @flow
/*::
type Node = {
  type: string,
  fullpath: string,
  basename: string,
  metadata: Object, // mode, oid
  parent?: Node,
  children: Array<Node>
}
*/

/** @ignore */
function flatFileListToDirectoryStructure(files /*: Array<{path: string}> */
) /*: Node|void */{
  const inodes /*: Map<string, Node> */ = new _Map();
  const mkdir = function (name) /*: Node|void */{
    if (!inodes.has(name)) {
      let dir /*: Node */ = {
        type: 'tree',
        fullpath: name,
        basename: path.basename(name),
        metadata: {},
        children: []
      };
      inodes.set(name, dir);
      // This recursively generates any missing parent folders.
      // We do it after we've added the inode to the set so that
      // we don't recurse infinitely trying to create the root '.' dirname.
      dir.parent = mkdir(path.dirname(name));
      if (dir.parent && dir.parent !== dir) dir.parent.children.push(dir);
    }
    return inodes.get(name);
  };

  const mkfile = function (name, metadata) /*: Node|void */{
    if (!inodes.has(name)) {
      let file /*: Node */ = {
        type: 'blob',
        fullpath: name,
        basename: path.basename(name),
        metadata: metadata,
        // This recursively generates any missing parent folders.
        parent: mkdir(path.dirname(name)),
        children: []
      };
      if (file.parent) file.parent.children.push(file);
      inodes.set(name, file);
    }
    return inodes.get(name);
  };

  mkdir('.');
  for (let file of files) {
    mkfile(file.path, file);
  }
  return inodes.get('.');
}

/** @ignore */
async function sleep(ms) {
  return new _Promise((resolve, reject) => setTimeout(resolve, ms));
}

var name = "isomorphic-git";
var version = "0.0.0-development";

/**
 *
 * Use with {@link push} and {@link fetch} to set Basic Authentication headers.
 * This for is for *actual* OAuth2 tokens (not "personal access tokens").
 * Unfortunately, all the major git hosting companies have chosen different conventions!
 * Lucky for you, I already looked up and codified it for you.
 *
 * - oauth2('github', token) - Github uses `token` as the username, and 'x-oauth-basic' as the password.
 * - oauth2('bitbucket', token) - Bitbucket uses 'x-token-auth' as the username, and `token` as the password.
 * - oauth2('gitlab', token) - Gitlab uses 'oauth2' as the username, and `token` as the password.
 *
 * I will gladly accept pull requests for more companies' conventions.
 *
 * @param {string} company
 * @param {string} token
 * @returns {{username: string, password: string}}
 *
 */
function oauth2(company, token) {
  switch (company) {
    case 'github':
      return {
        username: token,
        password: 'x-oauth-basic'
      };
    case 'bitbucket':
      return {
        username: 'x-token-auth',
        password: token
      };
    case 'gitlab':
      return {
        username: 'oauth2',
        password: token
      };
    default:
      throw new Error(`I don't know how ${company} expects its Basic Auth headers to be formatted for OAuth2 usage. If you do, you can use the regular '.auth(username, password)' to set the basic auth header yourself.`);
  }
}

/**
 *
 * Use with {@link push} and {@link fetch} to set Basic Authentication headers.
 * This works for basic username / password auth, or the newer username / token auth
 * that is often required if 2FA is enabled.
 *
 * Authentication is normally required for pushing to a git repository.
 * It may also be required to clone or fetch from a private repository.
 * Git does all its authentication using HTTPS Basic Authentication.
 * Usually this is straightforward, but there are some things to watch out for.
 *
 * If you have two-factor authentication (2FA) enabled on your account, you
 * probably cannot push or pull using your regular username and password.
 * Instead, you may have to create a Personal Access Token (or an App
 * Password in Bitbucket lingo) and use that to authenticate.
 *
 * @param {string} username
 * @param {string} password
 * @returns {{username: string, password: string}}
 *
 * @example
 * let {username, password} = auth('username', 'password')
 *
 * // a one-argument version is also supported
 * let {username, password} = auth('username:password')
 *
 * // Personal Access Token Authentication
 * // (note Bitbucket calls theirs App Passwords instead for some reason)
 * let {username, password} = auth('username', 'personal access token')
 * let {username, password} = auth('username', 'app password')
 * let {username, password} = auth('personal access token') // Github (only) lets you leave out the username
 */
function auth(username, password) {
  // Allow specifying it as one argument (mostly for CLI inputability)
  if (password === undefined) {
    let i = username.indexOf(':');
    if (i > -1) {
      password = username.slice(i + 1);
      username = username.slice(0, i);
    } else {
      password = ''; // Enables the .auth(GITHUB_TOKEN) no-username shorthand
    }
  }
  return { username, password };
}

const log = debug('isomorphic-git');

log.log = console.log.bind(console);

const delayedReleases = new _Map();
/**
 * @ignore
 * This is just a collection of helper functions really. At least that's how it started.
 */
class FileSystem {
  constructor(fs) {
    if (typeof fs._readFile !== 'undefined') return fs;
    this._readFile = pify(fs.readFile.bind(fs));
    this._writeFile = pify(fs.writeFile.bind(fs));
    this._mkdir = pify(fs.mkdir.bind(fs));
    this._rmdir = pify(fs.rmdir.bind(fs));
    this._unlink = pify(fs.unlink.bind(fs));
    this._stat = pify(fs.stat.bind(fs));
    this._lstat = pify(fs.lstat.bind(fs));
    this._readdir = pify(fs.readdir.bind(fs));
  }
  /**
   * Return true if a file exists, false if it doesn't exist.
   * Rethrows errors that aren't related to file existance.
   */
  async exists(filepath /*: string */, options /*: Object */ = {}) {
    try {
      await this._stat(filepath);
      return true;
    } catch (err) {
      if (err.code === 'ENOENT' || err.code === 'ENOTDIR') {
        return false;
      } else {
        console.log('Unhandled error in "FileSystem.exists()" function', err);
        throw err;
      }
    }
  }
  /**
   * Return the contents of a file if it exists, otherwise returns null.
   */
  async read(filepath /*: string */, options /*: Object */ = {}) {
    try {
      let buffer$$1 = await this._readFile(filepath, options);
      return buffer$$1;
    } catch (err) {
      return null;
    }
  }
  /**
   * Write a file (creating missing directories if need be) without throwing errors.
   */
  async write(filepath /*: string */
  , contents /*: string|Buffer */
  , options /*: Object */ = {}) {
    try {
      await this._writeFile(filepath, contents, options);
      return;
    } catch (err) {
      // Hmm. Let's try mkdirp and try again.
      await this.mkdir(path.dirname(filepath));
      await this._writeFile(filepath, contents, options);
    }
  }
  /**
   * Make a directory (or series of nested directories) without throwing an error if it already exists.
   */
  async mkdir(filepath /*: string */) {
    try {
      await this._mkdir(filepath);
      return;
    } catch (err) {
      // If err is null then operation succeeded!
      if (err === null) return;
      // If the directory already exists, that's OK!
      if (err.code === 'EEXIST') return;
      // If we got a "no such file or directory error" backup and try again.
      if (err.code === 'ENOENT') {
        let parent = path.dirname(filepath);
        // Check to see if we've gone too far
        if (parent === '.' || parent === '/' || parent === filepath) throw err;
        // Infinite recursion, what could go wrong?
        await this.mkdir(parent);
        await this._mkdir(filepath);
      }
    }
  }
  /**
   * Delete a file without throwing an error if it is already deleted.
   */
  async rm(filepath) {
    try {
      await this._unlink(filepath);
    } catch (err) {
      if (err.code !== 'ENOENT') throw err;
    }
  }
  /**
   * Read a directory without throwing an error is the directory doesn't exist
   */
  async readdir(filepath) {
    try {
      return await this._readdir(filepath);
    } catch (err) {
      return [];
    }
  }
  /**
   * Return a flast list of all the files nested inside a directory
   *
   * Based on an elegant concurrent recursive solution from SO
   * https://stackoverflow.com/a/45130990/2168416
   */
  async readdirDeep(dir) {
    const subdirs = await this._readdir(dir);
    const files = await _Promise.all(subdirs.map(async subdir => {
      const res = dir + '/' + subdir;
      return (await this._stat(res)).isDirectory() ? this.readdirDeep(res) : res;
    }));
    return files.reduce((a, f) => a.concat(f), []);
  }

  async lock(filename, triesLeft = 3) {
    // check to see if we still have it
    if (delayedReleases.has(filename)) {
      clearTimeout(delayedReleases.get(filename));
      delayedReleases.delete(filename);
      return;
    }
    if (triesLeft === 0) {
      throw new Error(`Unable to acquire lockfile '${filename}'. Exhausted tries.`);
    }
    try {
      await this.mkdir(`${filename}.lock`);
    } catch (err) {
      if (err.code === 'EEXIST') {
        await sleep(100);
        await this.lock(filename, triesLeft - 1);
      }
    }
  }

  async unlock(filename, delayRelease = 50) {
    if (delayedReleases.has(filename)) {
      throw new Error('Cannot double-release lockfile');
    }
    // Basically, we lie and say it was deleted ASAP.
    // But really we wait a bit to see if you want to acquire it again.
    delayedReleases.set(filename, setTimeout(async () => {
      delayedReleases.delete(filename);
      await this._rmdir(`${filename}.lock`);
    }, delayRelease));
  }
}

// @flow
function formatTimezoneOffset(minutes /*: number */) {
  let sign = _Math$sign(minutes) || 1;
  minutes = Math.abs(minutes);
  let hours = Math.floor(minutes / 60);
  minutes -= hours * 60;
  let strHours = String(hours);
  let strMinutes = String(minutes);
  if (strHours.length < 2) strHours = '0' + strHours;
  if (strMinutes.length < 2) strMinutes = '0' + strMinutes;
  return (sign === 1 ? '-' : '+') + strHours + strMinutes;
}

function parseTimezoneOffset(offset) {
  let [, sign, hours, minutes] = offset.match(/(\+|-)(\d\d)(\d\d)/);
  minutes = (sign === '-' ? 1 : -1) * Number(hours) * 60 + Number(minutes);
  return minutes;
}

function parseAuthor(author) {
  let [, name, email, timestamp, offset] = author.match(/^(.*) <(.*)> (.*) (.*)$/);
  return {
    name: name,
    email: email,
    timestamp: Number(timestamp),
    timezoneOffset: parseTimezoneOffset(offset)
  };
}

function normalize(str) {
  // remove all <CR>
  str = str.replace(/\r/g, '');
  // no extra newlines up front
  str = str.replace(/^\n+/, '');
  // and a single newline at the end
  str = str.replace(/\n+$/, '') + '\n';
  return str;
}

function indent(str) {
  return str.trim().split('\n').map(x => ' ' + x).join('\n') + '\n';
}

function outdent(str) {
  return str.split('\n').map(x => x.replace(/^ /, '')).join('\n');
}

// TODO: Make all functions have static async signature?

/** @ignore */
class GitCommit {
  /*::
  _commit : string
  */
  constructor(commit /*: string|Buffer|Object */) {
    if (typeof commit === 'string') {
      this._commit = commit;
    } else if (buffer.Buffer.isBuffer(commit)) {
      this._commit = commit.toString('utf8');
    } else if (typeof commit === 'object') {
      this._commit = GitCommit.render(commit);
    } else {
      throw new Error('invalid type passed to GitCommit constructor');
    }
  }

  static fromPayloadSignature({ payload, signature }) {
    let headers = GitCommit.justHeaders(payload);
    let message = GitCommit.justMessage(payload);
    let commit = normalize(headers + '\ngpgsig' + indent(signature) + '\n' + message);
    return new GitCommit(commit);
  }

  static from(commit) {
    return new GitCommit(commit);
  }

  toObject() {
    return buffer.Buffer.from(this._commit, 'utf8');
  }

  // Todo: allow setting the headers and message
  headers() {
    return this.parseHeaders();
  }

  // Todo: allow setting the headers and message
  message() {
    return GitCommit.justMessage(this._commit);
  }

  parse() {
    return _extends({ message: this.message() }, this.headers());
  }

  static justMessage(commit) {
    return normalize(commit.slice(commit.indexOf('\n\n') + 2));
  }

  static justHeaders(commit) {
    return commit.slice(0, commit.indexOf('\n\n'));
  }

  parseHeaders() {
    let headers = GitCommit.justHeaders(this._commit).split('\n');
    let hs = [];
    for (let h of headers) {
      if (h[0] === ' ') {
        // combine with previous header (without space indent)
        hs[hs.length - 1] += '\n' + h.slice(1);
      } else {
        hs.push(h);
      }
    }
    let obj = {};
    for (let h of hs) {
      let key = h.slice(0, h.indexOf(' '));
      let value = h.slice(h.indexOf(' ') + 1);
      obj[key] = value;
    }
    obj.parent = obj.parent ? obj.parent.split(' ') : [];
    if (obj.author) {
      obj.author = parseAuthor(obj.author);
    }
    if (obj.committer) {
      obj.committer = parseAuthor(obj.committer);
    }
    return obj;
  }

  static renderHeaders(obj) {
    let headers = '';
    if (obj.tree) {
      headers += `tree ${obj.tree}\n`;
    } else {
      headers += `tree 4b825dc642cb6eb9a060e54bf8d69288fbee4904\n`; // the null tree
    }
    if (obj.parent && obj.parent.length) {
      headers += 'parent';
      for (let p of obj.parent) {
        headers += ' ' + p;
      }
      headers += '\n';
    }
    let author = obj.author;
    headers += `author ${author.name} <${author.email}> ${author.timestamp} ${formatTimezoneOffset(author.timezoneOffset)}\n`;
    let committer = obj.committer || obj.author;
    headers += `committer ${committer.name} <${committer.email}> ${committer.timestamp} ${formatTimezoneOffset(committer.timezoneOffset)}\n`;
    if (obj.gpgsig) {
      headers += 'gpgsig' + indent(obj.gpgsig);
    }
    return headers;
  }

  static render(obj) {
    return GitCommit.renderHeaders(obj) + '\n' + normalize(obj.message);
  }

  render() {
    return this._commit;
  }

  withoutSignature() {
    let commit = normalize(this._commit);
    if (commit.indexOf('\ngpgsig') === -1) return commit;
    let headers = commit.slice(0, commit.indexOf('\ngpgsig'));
    let message = commit.slice(commit.indexOf('-----END PGP SIGNATURE-----\n') + '-----END PGP SIGNATURE-----\n'.length);
    return normalize(headers + '\n' + message);
  }

  isolateSignature() {
    let signature = this._commit.slice(this._commit.indexOf('-----BEGIN PGP SIGNATURE-----'), this._commit.indexOf('-----END PGP SIGNATURE-----') + '-----END PGP SIGNATURE-----'.length);
    return outdent(signature);
  }
}

function normalize$1(str) {
  // remove all <CR>
  str = str.replace(/\r/g, '');
  // no extra newlines up front
  str = str.replace(/^\n+/, '');
  // and a single newline at the end
  str = str.replace(/\n+$/, '') + '\n';
  return str;
}

function indent$1(str) {
  return str.trim().split('\n').map(x => ' ' + x).join('\n') + '\n';
}

/** @ignore */
class SignedGitCommit extends GitCommit {
  static from(commit) {
    return new SignedGitCommit(commit);
  }
  async sign(openpgp, privateKeys /*: string */) {
    let commit = this.withoutSignature();
    let headers = GitCommit.justHeaders(this._commit);
    let message = GitCommit.justMessage(this._commit);
    let privKeyObj = openpgp.key.readArmored(privateKeys).keys;
    let { signature } = await openpgp.sign({
      data: openpgp.util.str2Uint8Array(commit),
      privateKeys: privKeyObj,
      detached: true,
      armor: true
    });
    // renormalize the line endings to the one true line-ending
    signature = normalize$1(signature);
    let signedCommit = headers + '\n' + 'gpgsig' + indent$1(signature) + '\n' + message;
    // return a new commit object
    return GitCommit.from(signedCommit);
  }

  async listSigningKeys(openpgp) {
    let msg = openpgp.message.readSignedContent(this.withoutSignature(), this.isolateSignature());
    return msg.getSigningKeyIds().map(keyid => keyid.toHex());
  }

  async verify(openpgp, publicKeys /*: string */) {
    let pubKeyObj = openpgp.key.readArmored(publicKeys).keys;
    let msg = openpgp.message.readSignedContent(this.withoutSignature(), this.isolateSignature());
    let results = msg.verify(pubKeyObj);
    let validity = results.reduce((a, b) => a.valid && b.valid, { valid: true });
    return validity;
  }
}

const complexKeys = ['remote', 'branch'];

const isComplexKey = key => complexKeys.reduce((x, y) => x || key.startsWith(y), false);

const splitComplexKey = key => key.split('"').map(x => x.trim()).filter(x => x !== '');

// Note: there are a LOT of edge cases that aren't covered (e.g. keys in sections that also
// have subsections, [include] directives, etc.
/** @ignore */
class GitConfig {
  constructor(text) {
    this.ini = ini.decode(text);
    // Some mangling to make it easier to work with (honestly)
    for (let key of _Object$keys(this.ini)) {
      if (isComplexKey(key)) {
        let parts = splitComplexKey(key);
        if (parts.length === 2) {
          // just to be cautious
          set(this.ini, [parts[0], parts[1]], this.ini[key]);
          delete this.ini[key];
        }
      }
    }
  }
  static from(text) {
    return new GitConfig(text);
  }
  async get(path$$1) {
    return get(this.ini, path$$1);
  }
  async set(path$$1, value) {
    if (value === undefined) {
      unset(this.ini, path$$1);
    } else {
      set(this.ini, path$$1, value);
    }
  }
  toString() {
    // de-mangle complex keys
    for (let key of _Object$keys(this.ini)) {
      if (isComplexKey(key)) {
        for (let childkey of _Object$keys(this.ini[key])) {
          let complexkey = `${key} "${childkey}"`;
          this.ini[complexkey] = this.ini[key][childkey];
          delete this.ini[key][childkey];
        }
        delete this.ini[key];
      }
    }
    let text = ini.encode(this.ini, { whitespace: true });
    return text;
  }
}

/** @ignore */
class GitObject {
  static hash({ type, object }) /*: Promise<string> */{
    let buffer$$1 = buffer.Buffer.concat([buffer.Buffer.from(`${type} ${object.byteLength.toString()}\0`), buffer.Buffer.from(object)]);
    let oid = shasum(buffer$$1);
    return oid;
  }
  static wrap({ type, object }) {
    let buffer$$1 = buffer.Buffer.concat([buffer.Buffer.from(`${type} ${object.byteLength.toString()}\0`), object]);
    let oid = shasum(buffer$$1);
    return {
      oid,
      buffer: buffer$$1
    };
  }
  static unwrap({ oid, buffer: buffer$$1 }) {
    if (oid) {
      let sha = shasum(buffer$$1);
      if (sha !== oid) {
        throw new Error(`SHA check failed! Expected ${oid}, computed ${sha}`);
      }
    }
    let s = buffer$$1.indexOf(32); // first space
    let i = buffer$$1.indexOf(0); // first null value
    let type = buffer$$1.slice(0, s).toString('utf8'); // get type of object
    let length = buffer$$1.slice(s + 1, i).toString('utf8'); // get type of object
    let actualLength = buffer$$1.length - (i + 1);
    // verify length
    if (parseInt(length) !== actualLength) {
      throw new Error(`Length mismatch: expected ${length} bytes but got ${actualLength} instead.`);
    }
    return {
      type,
      object: buffer.Buffer.from(buffer$$1.slice(i + 1))
    };
  }
}

// @flow
/**
pkt-line Format
---------------

Much (but not all) of the payload is described around pkt-lines.

A pkt-line is a variable length binary string.  The first four bytes
of the line, the pkt-len, indicates the total length of the line,
in hexadecimal.  The pkt-len includes the 4 bytes used to contain
the length's hexadecimal representation.

A pkt-line MAY contain binary data, so implementors MUST ensure
pkt-line parsing/formatting routines are 8-bit clean.

A non-binary line SHOULD BE terminated by an LF, which if present
MUST be included in the total length. Receivers MUST treat pkt-lines
with non-binary data the same whether or not they contain the trailing
LF (stripping the LF if present, and not complaining when it is
missing).

The maximum length of a pkt-line's data component is 65516 bytes.
Implementations MUST NOT send pkt-line whose length exceeds 65520
(65516 bytes of payload + 4 bytes of length data).

Implementations SHOULD NOT send an empty pkt-line ("0004").

A pkt-line with a length field of 0 ("0000"), called a flush-pkt,
is a special case and MUST be handled differently than an empty
pkt-line ("0004").

----
  pkt-line     =  data-pkt / flush-pkt

  data-pkt     =  pkt-len pkt-payload
  pkt-len      =  4*(HEXDIG)
  pkt-payload  =  (pkt-len - 4)*(OCTET)

  flush-pkt    = "0000"
----

Examples (as C-style strings):

----
  pkt-line          actual value
  ---------------------------------
  "0006a\n"         "a\n"
  "0005a"           "a"
  "000bfoobar\n"    "foobar\n"
  "0004"            ""
----
*/
// I'm really using this more as a namespace.
// There's not a lot of "state" in a pkt-line

/** @ignore */
class GitPktLine {
  static flush() {
    return buffer.Buffer.from('0000', 'utf8');
  }

  static encode(line /*: string|Buffer */) /*: Buffer */{
    if (typeof line === 'string') {
      line = buffer.Buffer.from(line);
    }
    let length = line.length + 4;
    let hexlength = pad(4, length.toString(16), '0');
    return buffer.Buffer.concat([buffer.Buffer.from(hexlength, 'utf8'), line]);
  }

  static reader(buffer$$1 /*: Buffer */) {
    let buffercursor = new BufferCursor(buffer$$1);
    return function read() {
      if (buffercursor.eof()) return true;
      let length = parseInt(buffercursor.slice(4).toString('utf8'), 16);
      if (length === 0) return null;
      return buffercursor.slice(length - 4).buffer;
    };
  }
}

function buffer2stream(buffer$$1) {
  let stream$$1 = new stream.PassThrough();
  stream$$1.end(buffer$$1);
  return stream$$1;
}

function decodeVarInt(reader) {
  let bytes = [];
  let byte = 0;
  let multibyte = 0;
  do {
    byte = reader.readUInt8();
    // We keep bits 6543210
    const lastSeven = byte & 0b01111111;
    bytes.push(lastSeven);
    // Whether the next byte is part of the variable-length encoded number
    // is encoded in bit 7
    multibyte = byte & 0b10000000;
  } while (multibyte);
  // Now that all the bytes are in big-endian order,
  // alternate shifting the bits left by 7 and OR-ing the next byte.
  // And... do a weird increment-by-one thing that I don't quite understand.
  return bytes.reduce((a, b) => a + 1 << 7 | b, -1);
}

// I'm pretty much copying this one from the git C source code,
// because it makes no sense.
function otherVarIntDecode(reader, startWith) {
  let result = startWith;
  let shift = 4;
  let byte = null;
  do {
    byte = reader.readUInt8();
    result |= (byte & 0b01111111) << shift;
    shift += 7;
  } while (byte & 0b10000000);
  return result;
}

/** @ignore */
class GitPackIndex {
  constructor(stuff) {
    _Object$assign(this, stuff);
    this.offsetCache = {};
  }
  static async fromIdx({ idx, getExternalRefDelta }) {
    let reader = new BufferCursor(idx);
    let magic = reader.slice(4).toString('hex');
    // Check for IDX v2 magic number
    if (magic !== 'ff744f63') {
      return; // undefined
    }
    let version = reader.readUInt32BE();
    if (version !== 2) {
      throw new Error(`Unable to read version ${version} packfile IDX. (Only version 2 supported)`);
    }
    // Verify checksums
    let shaComputed = shasum(idx.slice(0, -20));
    let shaClaimed = idx.slice(-20).toString('hex');
    if (shaClaimed !== shaComputed) {
      throw new Error(`Invalid checksum in IDX buffer: expected ${shaClaimed} but saw ${shaComputed}`);
    }
    if (idx.byteLength > 2048 * 1024 * 1024) {
      throw new Error(`To keep implementation simple, I haven't implemented the layer 5 feature needed to support packfiles > 2GB in size.`);
    }
    let fanout = [];
    for (let i = 0; i < 256; i++) {
      fanout.push(reader.readUInt32BE());
    }
    let size = fanout[255];
    // For now we'll parse the whole thing. We can optimize later if we need to.
    let hashes = [];
    for (let i = 0; i < size; i++) {
      hashes.push(reader.slice(20).toString('hex'));
    }
    let crcs = {};
    for (let i = 0; i < size; i++) {
      crcs[hashes[i]] = reader.readUInt32BE();
    }
    let offsets = {};
    for (let i = 0; i < size; i++) {
      offsets[hashes[i]] = reader.readUInt32BE();
    }
    let packfileSha = reader.slice(20).toString('hex');
    return new GitPackIndex({
      hashes,
      crcs,
      offsets,
      packfileSha,
      getExternalRefDelta
    });
  }
  static async fromPack({ pack, getExternalRefDelta }) {
    const listpackTypes = {
      1: 'commit',
      2: 'tree',
      3: 'blob',
      4: 'tag',
      6: 'ofs-delta',
      7: 'ref-delta'
    };
    let offsetToObject = {};

    // Older packfiles do NOT use the shasum of the pack itself,
    // so it is recommended to just use whatever bytes are in the trailer.
    // Source: https://github.com/git/git/commit/1190a1acf800acdcfd7569f87ac1560e2d077414
    // let packfileSha = shasum(pack.slice(0, -20))
    let packfileSha = pack.slice(-20).toString('hex');

    let hashes = [];
    let crcs = {};
    let offsets = {};
    let totalObjectCount = null;
    let lastPercent = null;
    let times = {
      hash: 0,
      readSlice: 0,
      offsets: 0,
      crcs: 0,
      sort: 0
    };
    let histogram = {
      commit: 0,
      tree: 0,
      blob: 0,
      tag: 0,
      'ofs-delta': 0,
      'ref-delta': 0
    };
    let bytesProcessed = 0;

    log('Indexing objects');
    log(`percent\tmilliseconds\tbytesProcessed\tcommits\ttrees\tblobs\ttags\tofs-deltas\tref-deltas`);
    marky.mark('total');
    marky.mark('offsets');
    marky.mark('percent');
    await new _Promise((resolve, reject) => {
      buffer2stream(pack).pipe(listpack()).on('data', async ({ data, type, reference, offset, num }) => {
        if (totalObjectCount === null) totalObjectCount = num;
        let percent = Math.floor((totalObjectCount - num) * 100 / totalObjectCount);
        if (percent !== lastPercent) {
          log(`${percent}%\t${Math.floor(marky.stop('percent').duration)}\t${bytesProcessed}\t${histogram.commit}\t${histogram.tree}\t${histogram.blob}\t${histogram.tag}\t${histogram['ofs-delta']}\t${histogram['ref-delta']}`);

          histogram = {
            commit: 0,
            tree: 0,
            blob: 0,
            tag: 0,
            'ofs-delta': 0,
            'ref-delta': 0
          };
          bytesProcessed = 0;
          marky.mark('percent');
        }
        lastPercent = percent;
        // Change type from a number to a meaningful string
        type = listpackTypes[type];

        histogram[type]++;
        bytesProcessed += data.byteLength;

        if (['commit', 'tree', 'blob', 'tag'].includes(type)) {
          offsetToObject[offset] = {
            type,
            offset
          };
        } else if (type === 'ofs-delta') {
          offsetToObject[offset] = {
            type,
            offset
          };
        } else if (type === 'ref-delta') {
          offsetToObject[offset] = {
            type,
            offset
          };
        }
        if (num === 0) resolve();
      });
    });
    times['offsets'] = Math.floor(marky.stop('offsets').duration);

    log('Computing CRCs');
    marky.mark('crcs');
    // We need to know the lengths of the slices to compute the CRCs.
    let offsetArray = _Object$keys(offsetToObject).map(Number);
    for (let [i, start] of offsetArray.entries()) {
      let end = i + 1 === offsetArray.length ? pack.byteLength - 20 : offsetArray[i + 1];
      let o = offsetToObject[start];
      let crc = crc32(pack.slice(start, end));
      o.end = end;
      o.crc = crc;
    }
    times['crcs'] = Math.floor(marky.stop('crcs').duration);

    // We don't have the hashes yet. But we can generate them using the .readSlice function!
    const p = new GitPackIndex({
      pack,
      packfileSha,
      crcs,
      hashes,
      offsets,
      getExternalRefDelta
    });

    // Resolve deltas and compute the oids
    log('Resolving deltas');
    log(`percent2\tmilliseconds2\tcallsToReadSlice\tcallsToGetExternal`);
    marky.mark('percent');
    lastPercent = null;
    let count = 0;
    let callsToReadSlice = 0;
    let callsToGetExternal = 0;
    let timeByDepth = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
    let objectsByDepth = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
    for (let offset in offsetToObject) {
      offset = Number(offset);
      let percent = Math.floor(count++ * 100 / totalObjectCount);
      if (percent !== lastPercent) {
        log(`${percent}%\t${Math.floor(marky.stop('percent').duration)}\t${callsToReadSlice}\t${callsToGetExternal}`);
        marky.mark('percent');
        callsToReadSlice = 0;
        callsToGetExternal = 0;
      }
      lastPercent = percent;

      let o = offsetToObject[offset];
      if (o.oid) continue;
      try {
        p.readDepth = 0;
        p.externalReadDepth = 0;
        marky.mark('readSlice');
        let { type, object } = await p.readSlice({ start: offset });
        let time = marky.stop('readSlice').duration;
        times.readSlice += time;
        callsToReadSlice += p.readDepth;
        callsToGetExternal += p.externalReadDepth;
        timeByDepth[p.readDepth] += time;
        objectsByDepth[p.readDepth] += 1;
        marky.mark('hash');
        let oid = GitObject.hash({ type, object });
        times.hash += marky.stop('hash').duration;
        o.oid = oid;
        hashes.push(oid);
        offsets[oid] = offset;
        crcs[oid] = o.crc;
      } catch (err) {
        log('ERROR', err);
        continue;
      }
    }

    marky.mark('sort');
    hashes.sort();
    times['sort'] = Math.floor(marky.stop('sort').duration);
    let totalElapsedTime = marky.stop('total').duration;
    times.hash = Math.floor(times.hash);
    times.readSlice = Math.floor(times.readSlice);
    times.misc = Math.floor(_Object$values(times).reduce((a, b) => a - b, totalElapsedTime));
    log(_Object$keys(times).join('\t'));
    log(_Object$values(times).join('\t'));
    log('by depth:');
    log([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11].join('\t'));
    log(objectsByDepth.slice(0, 12).join('\t'));
    log(timeByDepth.map(Math.floor).slice(0, 12).join('\t'));
    return p;
  }
  toBuffer() {
    let buffers = [];
    let write = (str, encoding) => {
      buffers.push(buffer.Buffer.from(str, encoding));
    };
    // Write out IDX v2 magic number
    write('ff744f63', 'hex');
    // Write out version number 2
    write('00000002', 'hex');
    // Write fanout table
    let fanoutBuffer = new BufferCursor(buffer.Buffer.alloc(256 * 4));
    for (let i = 0; i < 256; i++) {
      let count = 0;
      for (let hash of this.hashes) {
        if (parseInt(hash.slice(0, 2), 16) <= i) count++;
      }
      fanoutBuffer.writeUInt32BE(count);
    }
    buffers.push(fanoutBuffer.buffer);
    // Write out hashes
    for (let hash of this.hashes) {
      write(hash, 'hex');
    }
    // Write out crcs
    let crcsBuffer = new BufferCursor(buffer.Buffer.alloc(this.hashes.length * 4));
    for (let hash of this.hashes) {
      crcsBuffer.writeUInt32BE(this.crcs[hash]);
    }
    buffers.push(crcsBuffer.buffer);
    // Write out offsets
    let offsetsBuffer = new BufferCursor(buffer.Buffer.alloc(this.hashes.length * 4));
    for (let hash of this.hashes) {
      offsetsBuffer.writeUInt32BE(this.offsets[hash]);
    }
    buffers.push(offsetsBuffer.buffer);
    // Write out packfile checksum
    write(this.packfileSha, 'hex');
    // Write out shasum
    let totalBuffer = buffer.Buffer.concat(buffers);
    let sha = shasum(totalBuffer);
    let shaBuffer = buffer.Buffer.alloc(20);
    shaBuffer.write(sha, 'hex');
    return buffer.Buffer.concat([totalBuffer, shaBuffer]);
  }
  async load({ pack }) {
    this.pack = pack;
  }
  async unload() {
    this.pack = null;
  }
  async read({ oid /*: {oid: string} */ }) {
    if (!this.offsets[oid]) {
      if (this.getExternalRefDelta) {
        this.externalReadDepth++;
        return this.getExternalRefDelta(oid);
      } else {
        throw new Error(`Could not read object ${oid} from packfile`);
      }
    }
    let start = this.offsets[oid];
    return this.readSlice({ start });
  }
  async readSlice({ start }) {
    if (this.offsetCache[start]) return this.offsetCache[start];
    this.readDepth++;
    const types = {
      0b0010000: 'commit',
      0b0100000: 'tree',
      0b0110000: 'blob',
      0b1000000: 'tag',
      0b1100000: 'ofs_delta',
      0b1110000: 'ref_delta'
    };
    if (!this.pack) {
      throw new Error('Tried to read from a GitPackIndex with no packfile loaded into memory');
    }
    let raw = this.pack.slice(start);
    let reader = new BufferCursor(raw);
    let byte = reader.readUInt8();
    // Object type is encoded in bits 654
    let btype = byte & 0b1110000;
    let type = types[btype];
    if (type === undefined) {
      throw new Error('Unrecognized type: 0b' + btype.toString(2));
    }
    // The length encoding get complicated.
    // Last four bits of length is encoded in bits 3210
    let lastFour = byte & 0b1111;
    let length = lastFour;
    // Whether the next byte is part of the variable-length encoded number
    // is encoded in bit 7
    let multibyte = byte & 0b10000000;
    if (multibyte) {
      length = otherVarIntDecode(reader, lastFour);
    }
    let base = null;
    let object = null;
    // Handle deltified objects
    if (type === 'ofs_delta') {
      let offset = decodeVarInt(reader);
      let baseOffset = start - offset;({ object: base, type } = await this.readSlice({ start: baseOffset }));
    }
    if (type === 'ref_delta') {
      let oid = reader.slice(20).toString('hex');({ object: base, type } = await this.read({ oid }));
    }
    // Handle undeltified objects
    let buffer$$1 = raw.slice(reader.tell());
    object = buffer.Buffer.from(pako.inflate(buffer$$1));
    // Assert that the object length is as expected.
    if (object.byteLength !== length) {
      throw new Error(`Packfile told us object would have length ${length} but it had length ${object.byteLength}`);
    }
    if (base) {
      object = buffer.Buffer.from(applyDelta(object, base));
    }
    // Cache the result based on depth.
    if (this.readDepth > 3) {
      // hand tuned for speed / memory usage tradeoff
      this.offsetCache[start] = { type, object };
    }
    return { type, format: 'content', object };
  }
}

// @flow
/*::
import type {Stats} from 'fs'

type CacheEntryFlags = {
  assumeValid: boolean,
  extended: boolean,
  stage: number,
  nameLength: number
}

type CacheEntry = {
  ctime: Date,
  ctimeNanoseconds?: number,
  mtime: Date,
  mtimeNanoseconds?: number,
  dev: number,
  ino: number,
  mode: number,
  uid: number,
  gid: number,
  size: number,
  oid: string,
  flags: CacheEntryFlags,
  path: string
}
*/

// Extract 1-bit assume-valid, 1-bit extended flag, 2-bit merge state flag, 12-bit path length flag
function parseCacheEntryFlags(bits /*: number */) /*: CacheEntryFlags */{
  return {
    assumeValid: Boolean(bits & 0b1000000000000000),
    extended: Boolean(bits & 0b0100000000000000),
    stage: (bits & 0b0011000000000000) >> 12,
    nameLength: bits & 0b0000111111111111
  };
}

function renderCacheEntryFlags(flags /*: CacheEntryFlags */) /*: number */{
  return (flags.assumeValid ? 0b1000000000000000 : 0) + (flags.extended ? 0b0100000000000000 : 0) + ((flags.stage & 0b11) << 12) + (flags.nameLength & 0b111111111111);
}

function parseBuffer(buffer$$1) {
  // Verify shasum
  let shaComputed = shasum(buffer$$1.slice(0, -20));
  let shaClaimed = buffer$$1.slice(-20).toString('hex');
  if (shaClaimed !== shaComputed) {
    throw new Error(`Invalid checksum in GitIndex buffer: expected ${shaClaimed} but saw ${shaComputed}`);
  }
  let reader = new BufferCursor(buffer$$1);
  let _entries /*: Map<string, CacheEntry> */ = new _Map();
  let magic = reader.toString('utf8', 4);
  if (magic !== 'DIRC') {
    throw new Error(`Inavlid dircache magic file number: ${magic}`);
  }
  let version = reader.readUInt32BE();
  if (version !== 2) throw new Error(`Unsupported dircache version: ${version}`);
  let numEntries = reader.readUInt32BE();
  let i = 0;
  while (!reader.eof() && i < numEntries) {
    let entry = {};
    let ctimeSeconds = reader.readUInt32BE();
    let ctimeNanoseconds = reader.readUInt32BE();
    entry.ctime = new Date(ctimeSeconds * 1000 + ctimeNanoseconds / 1000000);
    entry.ctimeNanoseconds = ctimeNanoseconds;
    let mtimeSeconds = reader.readUInt32BE();
    let mtimeNanoseconds = reader.readUInt32BE();
    entry.mtime = new Date(mtimeSeconds * 1000 + mtimeNanoseconds / 1000000);
    entry.mtimeNanoseconds = mtimeNanoseconds;
    entry.dev = reader.readUInt32BE();
    entry.ino = reader.readUInt32BE();
    entry.mode = reader.readUInt32BE();
    entry.uid = reader.readUInt32BE();
    entry.gid = reader.readUInt32BE();
    entry.size = reader.readUInt32BE();
    entry.oid = reader.slice(20).toString('hex');
    let flags = reader.readUInt16BE();
    entry.flags = parseCacheEntryFlags(flags);
    // TODO: handle if (version === 3 && entry.flags.extended)
    let pathlength = buffer$$1.indexOf(0, reader.tell() + 1) - reader.tell();
    if (pathlength < 1) throw new Error(`Got a path length of: ${pathlength}`);
    entry.path = reader.toString('utf8', pathlength);
    // The next bit is awkward. We expect 1 to 8 null characters
    let tmp = reader.readUInt8();
    if (tmp !== 0) {
      throw new Error(`Expected 1-8 null characters but got '${tmp}'`);
    }
    let numnull = 1;
    while (!reader.eof() && reader.readUInt8() === 0 && numnull < 9) numnull++;
    reader.seek(reader.tell() - 1);
    // end of awkward part
    _entries.set(entry.path, entry);
    i++;
  }
  return _entries;
}

/** @ignore */
class GitIndex {
  /*::
   _entries: Map<string, CacheEntry>
   _dirty: boolean // Used to determine if index needs to be saved to filesystem
   */
  constructor(index /*: any */) {
    this._dirty = false;
    if (buffer.Buffer.isBuffer(index)) {
      this._entries = parseBuffer(index);
    } else if (index === null) {
      this._entries = new _Map();
    } else {
      throw new Error('invalid type passed to GitIndex constructor');
    }
  }
  static from(buffer$$1) {
    return new GitIndex(buffer$$1);
  }
  get entries() /*: Array<CacheEntry> */{
    return sortby([...this._entries.values()], 'path');
  }
  *[_Symbol$iterator]() {
    for (let entry of this.entries) {
      yield entry;
    }
  }
  insert({
    filepath,
    stats,
    oid
  }) /*: {filepath: string, stats: Stats, oid: string } */{
    let entry = {
      ctime: stats.ctime,
      mtime: stats.mtime,
      dev: stats.dev,
      ino: stats.ino,
      mode: stats.mode,
      uid: stats.uid,
      gid: stats.gid,
      size: stats.size,
      path: filepath,
      oid: oid,
      flags: {
        assumeValid: false,
        extended: false,
        stage: 0,
        nameLength: filepath.length < 0xfff ? filepath.length : 0xfff
      }
    };
    this._entries.set(entry.path, entry);
    this._dirty = true;
  }
  delete({ filepath /*: {filepath: string} */ }) {
    if (this._entries.has(filepath)) {
      this._entries.delete(filepath);
    } else {
      for (let key of this._entries.keys()) {
        if (key.startsWith(filepath + '/')) {
          this._entries.delete(key);
        }
      }
    }
    this._dirty = true;
  }
  clear() {
    this._entries.clear();
    this._dirty = true;
  }
  render() {
    return this.entries.map(entry => `${entry.mode.toString(8)} ${entry.oid}    ${entry.path}`).join('\n');
  }
  toObject() {
    let header = buffer.Buffer.alloc(12);
    let writer = new BufferCursor(header);
    writer.write('DIRC', 4, 'utf8');
    writer.writeUInt32BE(2);
    writer.writeUInt32BE(this.entries.length);
    let body = buffer.Buffer.concat(this.entries.map(entry => {
      // the fixed length + the filename + at least one null char => align by 8
      let length = Math.ceil((62 + entry.path.length + 1) / 8) * 8;
      let written = buffer.Buffer.alloc(length);
      let writer = new BufferCursor(written);
      let ctimeMilliseconds = entry.ctime.valueOf();
      let ctimeSeconds = Math.floor(ctimeMilliseconds / 1000);
      let ctimeNanoseconds = entry.ctimeNanoseconds || ctimeMilliseconds * 1000000 - ctimeSeconds * 1000000 * 1000;
      let mtimeMilliseconds = entry.mtime.valueOf();
      let mtimeSeconds = Math.floor(mtimeMilliseconds / 1000);
      let mtimeNanoseconds = entry.mtimeNanoseconds || mtimeMilliseconds * 1000000 - mtimeSeconds * 1000000 * 1000;
      writer.writeUInt32BE(ctimeSeconds);
      writer.writeUInt32BE(ctimeNanoseconds);
      writer.writeUInt32BE(mtimeSeconds);
      writer.writeUInt32BE(mtimeNanoseconds);
      writer.writeUInt32BE(entry.dev);
      writer.writeUInt32BE(entry.ino);
      writer.writeUInt32BE(entry.mode);
      writer.writeUInt32BE(entry.uid);
      writer.writeUInt32BE(entry.gid);
      writer.writeUInt32BE(entry.size);
      writer.write(entry.oid, 20, 'hex');
      writer.writeUInt16BE(renderCacheEntryFlags(entry.flags));
      writer.write(entry.path, entry.path.length, 'utf8');
      return written;
    }));
    let main = buffer.Buffer.concat([header, body]);
    let sum = shasum(main);
    return buffer.Buffer.concat([main, buffer.Buffer.from(sum, 'hex')]);
  }
}

// @flow
/*::
type TreeEntry = {
  mode: string,
  path: string,
  oid: string,
  type?: string
}
*/

function parseBuffer$1(buffer$$1) /*: Array<TreeEntry> */{
  let _entries = [];
  let cursor = 0;
  while (cursor < buffer$$1.length) {
    let space = buffer$$1.indexOf(32, cursor);
    if (space === -1) {
      throw new Error(`GitTree: Error parsing buffer at byte location ${cursor}: Could not find the next space character.`);
    }
    let nullchar = buffer$$1.indexOf(0, cursor);
    if (nullchar === -1) {
      throw new Error(`GitTree: Error parsing buffer at byte location ${cursor}: Could not find the next null character.`);
    }
    let mode = buffer$$1.slice(cursor, space).toString('utf8');
    if (mode === '40000') mode = '040000'; // makes it line up neater in printed output
    let type = mode === '040000' ? 'tree' : 'blob';
    let path$$1 = buffer$$1.slice(space + 1, nullchar).toString('utf8');
    let oid = buffer$$1.slice(nullchar + 1, nullchar + 21).toString('hex');
    cursor = nullchar + 21;
    _entries.push({ mode, path: path$$1, oid, type });
  }
  return _entries;
}

function nudgeIntoShape(entry) {
  if (!entry.oid && entry.sha) {
    entry.oid = entry.sha; // Github
  }
  if (typeof entry.mode === 'number') {
    entry.mode = entry.mode.toString(8); // index
  }
  if (!entry.type) {
    entry.type = 'blob'; // index
  }
  return entry;
}

/** @ignore */
class GitTree {
  /*::
  _entries: Array<TreeEntry>
  */
  constructor(entries /*: any */) {
    if (buffer.Buffer.isBuffer(entries)) {
      this._entries = parseBuffer$1(entries);
    } else if (Array.isArray(entries)) {
      this._entries = entries.map(nudgeIntoShape);
    } else {
      throw new Error('invalid type passed to GitTree constructor');
    }
  }
  static from(tree) {
    return new GitTree(tree);
  }
  render() {
    return this._entries.map(entry => `${entry.mode} ${entry.type} ${entry.oid}    ${entry.path}`).join('\n');
  }
  toObject() {
    return buffer.Buffer.concat(this._entries.map(entry => {
      let mode = buffer.Buffer.from(entry.mode.replace(/^0/, ''));
      let space = buffer.Buffer.from(' ');
      let path$$1 = buffer.Buffer.from(entry.path, { encoding: 'utf8' });
      let nullchar = buffer.Buffer.from([0]);
      let oid = buffer.Buffer.from(entry.oid.match(/../g).map(n => parseInt(n, 16)));
      return buffer.Buffer.concat([mode, space, path$$1, nullchar, oid]);
    }));
  }
  entries() {
    return this._entries;
  }
  *[_Symbol$iterator]() {
    for (let entry of this._entries) {
      yield entry;
    }
  }
}

// @flow
/** @ignore */
class GitConfigManager {
  static async get({ fs: _fs, gitdir }) {
    const fs = new FileSystem(_fs);
    // We can improve efficiency later if needed.
    // TODO: read from full list of git config files
    let text = await fs.read(`${gitdir}/config`, { encoding: 'utf8' });
    return GitConfig.from(text);
  }
  static async save({ fs: _fs, gitdir, config }) {
    const fs = new FileSystem(_fs);
    // We can improve efficiency later if needed.
    // TODO: handle saving to the correct global/user/repo location
    await fs.write(`${gitdir}/config`, config.toString(), {
      encoding: 'utf8'
    });
  }
}

// @flow
// TODO: Add file locks.

/** @ignore */
class GitShallowManager {
  static async read({ fs, gitdir }) {
    let oids = new _Set();
    let text = await fs.read(path.join(gitdir, 'shallow'), { encoding: 'utf8' });
    if (text === null) return oids;
    text.trim().split('\n').map(oid => oids.add(oid));
    return oids;
  }
  static async write({ fs, gitdir, oids }) {
    let text = '';
    for (let oid of oids) {
      text += `${oid}\n`;
    }
    await fs.write(path.join(gitdir, 'shallow'), text, {
      encoding: 'utf8'
    });
  }
}

// @flow
// import LockManager from 'travix-lock-manager'
// import Lock from '../utils'

// TODO: replace with an LRU cache?
const map /*: Map<string, GitIndex> */ = new _Map();
// const lm = new LockManager()
const lock = new AsyncLock();

/** @ignore */
class GitIndexManager {
  static async acquire({ fs: _fs, filepath }, closure) {
    const fs = new FileSystem(_fs);
    await lock.acquire(filepath, async function () {
      let index = map.get(filepath);
      if (index === undefined) {
        // Acquire a file lock while we're reading the index
        // to make sure other processes aren't writing to it
        // simultaneously, which could result in a corrupted index.
        // const fileLock = await Lock(filepath)
        const rawIndexFile = await fs.read(filepath);
        index = GitIndex.from(rawIndexFile);
        // cache the GitIndex object so we don't need to re-read it
        // every time.
        // TODO: save the stat data for the index so we know whether
        // the cached file is stale (modified by an outside process).
        map.set(filepath, index);
        // await fileLock.cancel()
      }
      await closure(index);
      if (index._dirty) {
        // Acquire a file lock while we're writing the index file
        // let fileLock = await Lock(filepath)
        const buffer$$1 = index.toObject();
        await fs.write(filepath, buffer$$1);
        index._dirty = false;
      }
      // For now, discard our cached object so that external index
      // manipulation is picked up. TODO: use lstat and compare
      // file times to determine if our cached object should be
      // discarded.
      map.delete(filepath);
    });
  }
}

// @flow
// I'm putting this in a Manager because I reckon it could benefit
// from a LOT of cacheing.

// TODO: Implement .git/info/exclude

/** @ignore */
class GitIgnoreManager {
  static async isIgnored({
    fs: _fs,
    dir,
    gitdir = path.join(dir, '.git'),
    filepath
  }) /*: Promise<boolean> */{
    const fs = new FileSystem(_fs);
    let pairs = [{
      gitignore: path.join(dir, '.gitignore'),
      filepath
    }];
    let pieces = filepath.split('/');
    for (let i = 1; i < pieces.length; i++) {
      let folder = pieces.slice(0, i).join('/');
      let file = pieces.slice(i).join('/');
      pairs.push({
        gitignore: path.join(dir, folder, '.gitignore'),
        filepath: file
      });
    }
    let ignoredStatus = false;
    for (let p of pairs) {
      let file;
      try {
        file = await fs.read(p.gitignore, 'utf8');
      } catch (err) {
        if (err.code === 'NOENT') continue;
      }
      let ign = ignore().add(file);
      let unign = ignore().add(`**\n${file}`);
      // If the parent directory is excluded, we are done.
      // "It is not possible to re-include a file if a parent directory of that file is excluded. Git doesn’t list excluded directories for performance reasons, so any patterns on contained files have no effect, no matter where they are defined."
      // source: https://git-scm.com/docs/gitignore
      let parentdir = path.dirname(p.filepath);
      if (ign.ignores(parentdir)) return true;
      // If the file is currently ignored, test for UNignoring.
      if (ignoredStatus) {
        ignoredStatus = unign.ignores(p.filepath);
      } else {
        ignoredStatus = ign.ignores(p.filepath);
      }
    }
    return ignoredStatus;
  }
}

const PackfileCache = new _Map();

/** @ignore */
class GitObjectManager {
  static async read({ fs: _fs, gitdir, oid, format = 'content' }) {
    const fs = new FileSystem(_fs);
    // Look for it in the loose object directory.
    let file = await fs.read(`${gitdir}/objects/${oid.slice(0, 2)}/${oid.slice(2)}`);
    let source = `./objects/${oid.slice(0, 2)}/${oid.slice(2)}`;
    // Check to see if it's in a packfile.
    if (!file) {
      // Curry the current read method so that the packfile un-deltification
      // process can acquire external ref-deltas.
      const getExternalRefDelta = oid => GitObjectManager.read({ fs: _fs, gitdir, oid });
      // Iterate through all the .pack files
      let list = await fs.readdir(path.join(gitdir, '/objects/pack'));
      list = list.filter(x => x.endsWith('.pack'));
      for (let filename of list) {
        // Try to get the packfile from the in-memory cache
        let p = PackfileCache.get(filename);
        if (!p) {
          // If not there, load it from a .idx file
          const idxName = filename.replace(/pack$/, 'idx');
          if (await fs.exists(`${gitdir}/objects/pack/${idxName}`)) {
            const idx = await fs.read(`${gitdir}/objects/pack/${idxName}`);
            p = await GitPackIndex.fromIdx({ idx, getExternalRefDelta });
          } else {
            // If the .idx file isn't available, generate one.
            const pack = await fs.read(`${gitdir}/objects/pack/${filename}`);
            p = await GitPackIndex.fromPack({ pack, getExternalRefDelta });
            // Save .idx file
            await fs.write(`${gitdir}/objects/pack/${idxName}`, p.toBuffer());
          }
          PackfileCache.set(filename, p);
        }
        // console.log(p)
        // If the packfile DOES have the oid we're looking for...
        if (p.hashes.includes(oid)) {
          // Make sure the packfile is loaded in memory
          if (!p.pack) {
            const pack = await fs.read(`${gitdir}/objects/pack/${filename}`);
            await p.load({ pack });
          }
          // Get the resolved git object from the packfile
          let result = await p.read({ oid, getExternalRefDelta });
          result.source = `./objects/pack/${filename}`;
          return result;
        }
      }
    }
    // Check to see if it's in shallow commits.
    if (!file) {
      let text = await fs.read(`${gitdir}/shallow`, { encoding: 'utf8' });
      if (text !== null && text.includes(oid)) {
        throw new Error(`Failed to read git object with oid ${oid} because it is a shallow commit`);
      }
    }
    // Finally
    if (!file) {
      throw new Error(`Failed to read git object with oid ${oid}`);
    }
    if (format === 'deflated') {
      return { format: 'deflated', object: file, source };
    }
    let buffer$$1 = buffer.Buffer.from(pako.inflate(file));
    if (format === 'wrapped') {
      return { format: 'wrapped', object: buffer$$1, source };
    }
    let { type, object } = GitObject.unwrap({ oid, buffer: buffer$$1 });
    if (format === 'content') return { type, format: 'content', object, source };
  }

  static async hash({ gitdir, type, object }) /*: Promise<string> */{
    let buffer$$1 = buffer.Buffer.concat([buffer.Buffer.from(type + ' '), buffer.Buffer.from(object.byteLength.toString()), buffer.Buffer.from([0]), buffer.Buffer.from(object)]);
    let oid = shasum(buffer$$1);
    return oid;
  }

  static async write({ fs: _fs, gitdir, type, object }) /*: Promise<string> */{
    const fs = new FileSystem(_fs);
    let { buffer: buffer$$1, oid } = GitObject.wrap({ type, object });
    let file = buffer.Buffer.from(pako.deflate(buffer$$1));
    let filepath = `${gitdir}/objects/${oid.slice(0, 2)}/${oid.slice(2)}`;
    // Don't overwrite existing git objects - this helps avoid EPERM errors.
    // Although I don't know how we'd fix corrupted objects then. Perhaps delete them
    // on read?
    if (!(await fs.exists(filepath))) await fs.write(filepath, file);
    return oid;
  }
}

// This is a convenience wrapper for reading and writing files in the 'refs' directory.
/** @ignore */
class GitRefManager {
  /* ::
  updateRemoteRefs : ({
    gitdir: string,
    remote: string,
    refs: Map<string, string>,
    symrefs: Map<string, string>
  }) => Promise<void>
  */
  static async updateRemoteRefs({
    fs: _fs,
    gitdir,
    remote,
    refs,
    symrefs,
    tags
  }) {
    const fs = new FileSystem(_fs);
    // Validate input
    for (let value of refs.values()) {
      if (!value.match(/[0-9a-f]{40}/)) {
        throw new Error(`Unexpected ref contents: '${value}'`);
      }
    }
    // Combine refs and symrefs giving symrefs priority
    let actualRefsToWrite = new _Map();
    for (let [key, value] of refs) {
      actualRefsToWrite.set(key, value);
    }
    for (let [key, value] of symrefs) {
      let branch = value.replace(/^refs\/heads\//, '');
      actualRefsToWrite.set(key, `ref: refs/remotes/${remote}/${branch}`);
    }
    // Update files
    // TODO: For large repos with a history of thousands of pull requests
    // (i.e. gitlab-ce) it would be vastly more efficient to write them
    // to .git/packed-refs.
    // The trick is to make sure we a) don't write a packed ref that is
    // already shadowed by a loose ref and b) don't loose any refs already
    // in packed-refs. Doing this efficiently may be difficult. A
    // solution that might work is
    // a) load the current packed-refs file
    // b) add actualRefsToWrite, overriding the existing values if present
    // c) enumerate all the loose refs currently in .git/refs/remotes/${remote}
    // d) overwrite their value with the new value.
    // Examples of refs we need to avoid writing in loose format for efficieny's sake
    // are .git/refs/remotes/origin/refs/remotes/remote_mirror_3059
    // and .git/refs/remotes/origin/refs/merge-requests
    const normalizeValue = value => value.trim() + '\n';
    for (let [key, value] of actualRefsToWrite) {
      if (key.startsWith('refs/heads') || key === 'HEAD') {
        key = key.replace(/^refs\/heads\//, '');
        await fs.write(path.join(gitdir, 'refs', 'remotes', remote, key), normalizeValue(value), 'utf8');
      } else if (tags === true && key.startsWith('refs/tags') && !key.endsWith('^{}')) {
        key = key.replace(/^refs\/tags\//, '');
        const filename = path.join(gitdir, 'refs', 'tags', key);
        // Git's behavior is to only fetch tags that do not conflict with tags already present.
        if (!(await fs.exists(filename))) {
          await fs.write(filename, normalizeValue(value), 'utf8');
        }
      }
    }
  }
  static async resolve({ fs: _fs, gitdir, ref, depth }) {
    const fs = new FileSystem(_fs);
    if (depth !== undefined) {
      depth--;
      if (depth === -1) {
        return ref;
      }
    }
    let sha;
    // Is it a ref pointer?
    if (ref.startsWith('ref: ')) {
      ref = ref.slice('ref: '.length);
      return GitRefManager.resolve({ fs, gitdir, ref, depth });
    }
    // Is it a complete and valid SHA?
    if (ref.length === 40 && /[0-9a-f]{40}/.test(ref)) {
      return ref;
    }
    // Is it a special ref?
    if (ref === 'HEAD' || ref === 'MERGE_HEAD') {
      sha = await fs.read(`${gitdir}/${ref}`, { encoding: 'utf8' });
      if (sha) {
        return GitRefManager.resolve({ fs, gitdir, ref: sha.trim(), depth });
      }
    }
    // Is it a full ref?
    if (ref.startsWith('refs/')) {
      sha = await fs.read(`${gitdir}/${ref}`, { encoding: 'utf8' });
      if (sha) {
        return GitRefManager.resolve({ fs, gitdir, ref: sha.trim(), depth });
      }
    }
    // Is it a (local) branch?
    sha = await fs.read(`${gitdir}/refs/heads/${ref}`, { encoding: 'utf8' });
    if (sha) {
      return GitRefManager.resolve({ fs, gitdir, ref: sha.trim(), depth });
    }
    // Is it a lightweight tag?
    sha = await fs.read(`${gitdir}/refs/tags/${ref}`, { encoding: 'utf8' });
    if (sha) {
      return GitRefManager.resolve({ fs, gitdir, ref: sha.trim(), depth });
    }
    // Is it remote branch?
    sha = await fs.read(`${gitdir}/refs/remotes/${ref}`, { encoding: 'utf8' });
    if (sha) {
      return GitRefManager.resolve({ fs, gitdir, ref: sha.trim(), depth });
    }
    // Is it a packed ref? (This must be last because refs in heads have priority)
    let text = await fs.read(`${gitdir}/packed-refs`, { encoding: 'utf8' });
    if (text && text.includes(ref)) {
      let candidates = text.trim().split('\n').filter(x => x.endsWith(ref)).filter(x => !x.startsWith('#'));
      if (candidates.length > 1) {
        throw new Error(`Could not resolve ambiguous reference ${ref}`);
      } else if (candidates.length === 1) {
        sha = candidates[0].split(' ')[0];
        return GitRefManager.resolve({ fs, gitdir, ref: sha.trim(), depth });
      }
    }
    // Do we give up?
    throw new Error(`Could not resolve reference ${ref}`);
  }
  static async packedRefs({ fs: _fs, gitdir }) {
    const refs = new _Map();
    const fs = new FileSystem(_fs);
    const text = await fs.read(`${gitdir}/packed-refs`, { encoding: 'utf8' });
    if (!text) return refs;
    const lines = text.trim().split('\n').filter(line => !/^\s*#/.test(line));
    let key = null;
    for (let line of lines) {
      const i = line.indexOf(' ');
      if (line.startsWith('^')) {
        // This is a oid for the commit associated with the annotated tag immediately preceding this line.
        // Trim off the '^'
        const value = line.slice(1, i);
        // The tagname^{} syntax is based on the output of `git show-ref --tags -d`
        refs.set(key + '^{}', value);
      } else {
        // This is an oid followed by the ref name
        const value = line.slice(0, i);
        key = line.slice(i + 1);
        refs.set(key, value);
      }
    }
    return refs;
  }
  // List all the refs that match the `filepath` prefix
  static async listRefs({ fs: _fs, gitdir, filepath }) {
    const fs = new FileSystem(_fs);
    let packedMap = GitRefManager.packedRefs({ fs, gitdir });
    let files = null;
    try {
      files = await fs.readdirDeep(`${gitdir}/${filepath}`);
      files = files.map(x => x.replace(`${gitdir}/${filepath}/`, ''));
    } catch (err) {
      files = [];
    }

    for (let key of (await packedMap).keys()) {
      // filter by prefix
      if (key.startsWith(filepath)) {
        // remove prefix
        key = key.replace(filepath + '/', '');
        // Don't include duplicates; the loose files have precedence anyway
        if (!files.includes(key)) {
          files.push(key);
        }
      }
    }
    return files;
  }
  static async listBranches({ fs: _fs, gitdir, remote }) {
    const fs = new FileSystem(_fs);
    if (remote) {
      return GitRefManager.listRefs({
        fs,
        gitdir,
        filepath: `refs/remotes/${remote}`
      });
    } else {
      return GitRefManager.listRefs({ fs, gitdir, filepath: `refs/heads` });
    }
  }
  static async listTags({ fs: _fs, gitdir }) {
    const fs = new FileSystem(_fs);
    let tags = await GitRefManager.listRefs({
      fs,
      gitdir,
      filepath: `refs/tags`
    });
    tags = tags.filter(x => !x.endsWith('^{}'));
    return tags;
  }
}

// @flow
function basicAuth(auth) {
  return `Basic ${buffer.Buffer.from(auth.username + ':' + auth.password).toString('base64')}`;
}

/** @ignore */
class GitRemoteHTTP {
  /*::
  GIT_URL : string
  refs : Map<string, string>
  symrefs : Map<string, string>
  capabilities : Set<string>
  auth : { username : string, password : string }
  */
  constructor(url /*: string */) {
    // Auto-append the (necessary) .git if it's missing.
    if (!url.endsWith('.git')) url = url += '.git';
    this.GIT_URL = url;
  }
  async preparePull() {
    await this.discover('git-upload-pack');
  }
  async preparePush() {
    await this.discover('git-receive-pack');
  }
  async discover(service /*: string */) {
    this.capabilities = new _Set();
    this.refs = new _Map();
    this.symrefs = new _Map();
    let headers = {};
    // headers['Accept'] = `application/x-${service}-advertisement`
    if (this.auth) {
      headers['Authorization'] = basicAuth(this.auth);
    }
    let res = await pify(simpleGet)({
      method: 'GET',
      url: `${this.GIT_URL}/info/refs?service=${service}`,
      headers
    });
    if (res.statusCode !== 200) {
      throw new Error(`Bad status code from server: ${res.statusCode}`);
    }
    let data = await pify(concat)(res);
    // There is probably a better way to do this, but for now
    // let's just throw the result parser inline here.
    let read = GitPktLine.reader(data);
    let lineOne = read();
    // skip past any flushes
    while (lineOne === null) lineOne = read();
    if (lineOne === true) throw new Error('Bad response from git server.');
    if (lineOne.toString('utf8') !== `# service=${service}\n`) {
      throw new Error(`Expected '# service=${service}\\n' but got '${lineOne.toString('utf8')}'`);
    }
    let lineTwo = read();
    // skip past any flushes
    while (lineTwo === null) lineTwo = read();
    // In the edge case of a brand new repo, zero refs (and zero capabilities)
    // are returned.
    if (lineTwo === true) return;
    let [firstRef, capabilities] = lineTwo.toString('utf8').trim().split('\0');
    capabilities.split(' ').map(x => this.capabilities.add(x));
    let [ref, name] = firstRef.split(' ');
    this.refs.set(name, ref);
    while (true) {
      let line = read();
      if (line === true) break;
      if (line !== null) {
        let [ref, name] = line.toString('utf8').trim().split(' ');
        this.refs.set(name, ref);
      }
    }
    // Symrefs are thrown into the "capabilities" unfortunately.
    for (let cap of this.capabilities) {
      if (cap.startsWith('symref=')) {
        let m = cap.match(/symref=([^:]+):(.*)/);
        if (m.length === 3) {
          this.symrefs.set(m[1], m[2]);
        }
      }
    }
  }
  async push(stream$$1 /*: ReadableStream */) {
    const service = 'git-receive-pack';
    let { packetlines, packfile } = await this.stream({
      stream: stream$$1,
      service
    });
    // TODO: Someday, maybe we will make this a streaming parser.
    packfile = await pify(concat)(packfile);
    packetlines = await pify(concat)(packetlines);
    let result = {};
    // Parse the response!
    // I'm combining the side-band-64k and regular streams
    // because Github returns the first line in the sideband while
    // git-http-server returns it without the sideband.
    let response = '';
    let read = GitPktLine.reader(packfile);
    let line = await read();
    while (line !== null && line !== true) {
      response += line.toString('utf8') + '\n';
      line = await read();
    }
    response += packetlines.toString('utf8');

    let lines = response.toString('utf8').split('\n');
    // We're expecting "unpack {unpack-result}"
    line = lines.shift();
    if (!line.startsWith('unpack ')) {
      throw new Error(`Unparsable response from server! Expected 'unpack ok' or 'unpack [error message]' but got '${line}'`);
    }
    if (line === 'unpack ok') {
      result.ok = ['unpack'];
    } else {
      result.errors = [line.trim()];
    }
    for (let line of lines) {
      let status = line.slice(0, 2);
      let refAndMessage = line.slice(3);
      if (status === 'ok') {
        result.ok = result.ok || [];
        result.ok.push(refAndMessage);
      } else if (status === 'ng') {
        result.errors = result.errors || [];
        result.errors.push(refAndMessage);
      }
    }
    log(result);
    return result;
  }
  async pull(stream$$1 /*: ReadableStream */) {
    const service = 'git-upload-pack';
    let res = await this.stream({ stream: stream$$1, service });
    return res;
  }
  async stream({
    stream: stream$$1,
    service
  }) /*: Promise<{ packfile: ReadableStream, progress: ReadableStream }> */{
    let headers = {};
    headers['content-type'] = `application/x-${service}-request`;
    headers['accept'] = `application/x-${service}-result`;
    headers['user-agent'] = `git/${name}@${version}`;
    if (this.auth) {
      headers['authorization'] = basicAuth(this.auth);
    }
    let res = await pify(simpleGet)({
      method: 'POST',
      url: `${this.GIT_URL}/${service}`,
      body: stream$$1,
      headers
    });
    let data = await pify(concat)(res);
    // Parse the response!
    let read = GitPktLine.reader(data);
    // And now for the ridiculous side-band-64k protocol
    let packetlines = new stream.PassThrough();
    let packfile = new stream.PassThrough();
    let progress = new stream.PassThrough();
    // TODO: Use a proper through stream?
    const nextBit = async function () {
      let line = await read();
      // Skip over flush packets
      if (line === null) return nextBit();
      // A made up convention to signal there's no more to read.
      if (line === true) {
        packetlines.end();
        progress.end();
        packfile.end();
        return;
      }
      // Examine first byte to determine which output "stream" to use
      switch (line[0]) {
        case 1:
          // pack data
          packfile.write(line.slice(1));
          break;
        case 2:
          // progress message
          progress.write(line.slice(1));
          break;
        case 3:
          // fatal error message just before stream aborts
          let error = line.slice(1);
          progress.write(error);
          packfile.destroy(new Error(error.toString('utf8')));
          return;
        default:
          // Not part of the side-band-64k protocol
          packetlines.write(line.slice(0));
      }
      // Careful not to blow up the stack.
      // I think Promises in a tail-call position should be OK.
      nextBit();
    };
    nextBit();
    return {
      packetlines,
      packfile,
      progress
    };
  }
}

/**
 * @external {FSModule} http://ghub.io/browserfs
 */

/**
 * Add a file to the git index (aka staging area)
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.filepath - The path to the file to add to the index.
 * @returns {Promise<void>} - Resolves successfully once the git index has been updated.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * await new Promise((resolve, reject) => fs.writeFile(
 *   '<@README.md@>',
 *   `<<@# TEST@>>`,
 *   (err) => err ? reject(err) : resolve()
 * ))
 * await git.add({...repo, filepath: '<@README.md@>'})
 * console.log('done')
 */
async function add({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  filepath
}) {
  const fs = new FileSystem(_fs);
  const type = 'blob';
  const object = await fs.read(path.join(dir, filepath));
  if (object === null) throw new Error(`Could not read file '${filepath}'`);
  const oid = await GitObjectManager.write({ fs, gitdir, type, object });
  await GitIndexManager.acquire({ fs, filepath: `${gitdir}/index` }, async function (index) {
    let stats = await fs._lstat(path.join(dir, filepath));
    index.insert({ filepath, stats, oid });
  });
  // TODO: return oid?
}

/**
 * Initialize a new repository
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @returns {Promise<void>} - Resolves successfully when filesystem operations are complete.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * await git.init(repo)
 * console.log('done')
 */
async function init({ dir, gitdir = path.join(dir, '.git'), fs: _fs }) {
  const fs = new FileSystem(_fs);
  let folders = ['hooks', 'info', 'objects/info', 'objects/pack', 'refs/heads', 'refs/tags'];
  folders = folders.map(dir => gitdir + '/' + dir);
  for (let folder of folders) {
    await fs.mkdir(folder);
  }
  await fs.write(gitdir + '/config', '[core]\n' + '\trepositoryformatversion = 0\n' + '\tfilemode = false\n' + '\tbare = false\n' + '\tlogallrefupdates = true\n' + '\tsymlinks = false\n' + '\tignorecase = true\n');
  await fs.write(gitdir + '/HEAD', 'ref: refs/heads/master\n');
}

/**
 * Read and/or write to the git config file(s)
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.path -  The key of the git config entry.
 * @param {string} [args.value] - A value to store at that path.
 * @returns {Promise<any>} - Resolves with the config value
 *
 * If no `value` is provided, it does a read.
 * If a `value` is provided, it does a write.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 *
 * // Write config value
 * await git.config({
 *   ...repo,
 *   path: '<@user.name@>',
 *   value: '<@Mr. Test@>'
 * })
 *
 * // Read config value
 * let value = await git.config({
 *   ...repo,
 *   path: '<@user.name@>'
 * })
 * console.log(value)
 */
async function config(_ref) {
  let {
    dir,
    gitdir = path.join(dir, '.git'),
    fs: _fs
  } = _ref,
      args = _objectWithoutProperties(_ref, ['dir', 'gitdir', 'fs']);

  const fs = new FileSystem(_fs);
  let { path: path$$1, value } = args;
  const config = await GitConfigManager.get({ fs, gitdir });
  // This carefully distinguishes between
  // 1) there is no 'value' argument (do a "get")
  // 2) there is a 'value' argument with a value of undefined (do a "set")
  // Because setting a key to undefined is how we delete entries from the ini.
  if (value === undefined && !args.hasOwnProperty('value')) {
    const value = await config.get(path$$1);
    return value;
  } else {
    await config.set(path$$1, value);
    await GitConfigManager.save({ fs, gitdir, config });
  }
}

/**
 * Fetch commits
 *
 * @link https://isomorphic-git.github.io/docs/fetch.html
 */
async function fetch({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  emitter,
  ref = 'HEAD',
  remote,
  url,
  authUsername,
  authPassword,
  depth,
  since,
  exclude,
  relative,
  tags,
  onprogress // deprecated
}) {
  if (onprogress !== undefined) {
    console.warn('The `onprogress` callback has been deprecated. Please use the more generic `emitter` EventEmitter argument instead.');
  }
  const fs = new FileSystem(_fs);
  let response = await fetchPackfile({
    gitdir,
    fs,
    ref,
    remote,
    url,
    authUsername,
    authPassword,
    depth,
    since,
    exclude,
    relative,
    tags
  });
  // Note: progress messages are designed to be written directly to the terminal,
  // so they are often sent with just a carriage return to overwrite the last line of output.
  // But there are also messages delimited with newlines.
  // I also include CRLF just in case.
  response.progress.pipe(split2(/(\r\n)|\r|\n/)).on('data', line => {
    if (emitter) {
      emitter.emit('message', line.trim());
    }
    let matches = line.match(/\((\d+?)\/(\d+?)\)/);
    if (matches && emitter) {
      emitter.emit('progress', {
        loaded: parseInt(matches[1], 10),
        total: parseInt(matches[2], 10),
        lengthComputable: true
      });
    }
  });
  let packfile = await pify(concat)(response.packfile);
  let packfileSha = packfile.slice(-20).toString('hex');
  await fs.write(path.join(gitdir, `objects/pack/pack-${packfileSha}.pack`), packfile);
}

async function fetchPackfile({
  gitdir,
  fs: _fs,
  ref,
  remote,
  url,
  authUsername,
  authPassword,
  depth = null,
  since = null,
  exclude = [],
  relative = false,
  tags = false
}) {
  const fs = new FileSystem(_fs);
  if (depth !== null) {
    if (_Number$isNaN(parseInt(depth))) {
      throw new Error(`Invalid value for depth argument: ${depth}`);
    }
    depth = parseInt(depth);
  }
  remote = remote || 'origin';
  if (url === undefined) {
    url = await config({
      fs,
      gitdir,
      path: `remote.${remote}.url`
    });
  }
  let remoteHTTP = new GitRemoteHTTP(url);
  if (authUsername !== undefined && authPassword !== undefined) {
    remoteHTTP.auth = {
      username: authUsername,
      password: authPassword
    };
  }
  await remoteHTTP.preparePull();
  // Check server supports shallow cloning
  if (depth !== null && !remoteHTTP.capabilities.has('shallow')) {
    throw new Error(`Remote does not support shallow fetches`);
  }
  if (since !== null && !remoteHTTP.capabilities.has('deepen-since')) {
    throw new Error(`Remote does not support shallow fetches by date`);
  }
  if (exclude.length > 0 && !remoteHTTP.capabilities.has('deepen-not')) {
    throw new Error(`Remote does not support shallow fetches excluding commits reachable by refs`);
  }
  if (relative === true && !remoteHTTP.capabilities.has('deepen-relative')) {
    throw new Error(`Remote does not support shallow fetches relative to the current shallow depth`);
  }
  await GitRefManager.updateRemoteRefs({
    fs,
    gitdir,
    remote,
    refs: remoteHTTP.refs,
    symrefs: remoteHTTP.symrefs,
    tags
  });
  let want = await GitRefManager.resolve({
    fs,
    gitdir,
    ref: `refs/remotes/${remote}/${ref}`
  });
  // Note: I removed "ofs-delta" from the capabilities list and now
  // Github uses all ref-deltas when I fetch packfiles instead of all ofs-deltas. Nice!
  const capabilities = `multi_ack_detailed no-done side-band-64k thin-pack ofs-delta agent=git/${name}@${version}${relative ? ' deepen-relative' : ''}`;
  let packstream = new stream.PassThrough();
  packstream.write(GitPktLine.encode(`want ${want} ${capabilities}\n`));
  let oids = await GitShallowManager.read({ fs, gitdir });
  if (oids.size > 0 && remoteHTTP.capabilities.has('shallow')) {
    for (let oid of oids) {
      packstream.write(GitPktLine.encode(`shallow ${oid}\n`));
    }
  }
  if (depth !== null) {
    packstream.write(GitPktLine.encode(`deepen ${depth}\n`));
  }
  if (since !== null) {
    packstream.write(GitPktLine.encode(`deepen-since ${Math.floor(since.valueOf() / 1000)}\n`));
  }
  for (let x of exclude) {
    packstream.write(GitPktLine.encode(`deepen-not ${x}\n`));
  }
  packstream.write(GitPktLine.flush());
  let have = null;
  try {
    have = await GitRefManager.resolve({ fs, gitdir, ref });
  } catch (err) {}
  if (have) {
    packstream.write(GitPktLine.encode(`have ${have}\n`));
    packstream.write(GitPktLine.flush());
  }
  packstream.end(GitPktLine.encode(`done\n`));
  let response = await remoteHTTP.pull(packstream);
  response.packetlines.pipe(through2(async (data, enc, next) => {
    let line = data.toString('utf8');
    if (line.startsWith('shallow')) {
      let oid = line.slice(-41).trim();
      if (oid.length !== 40) {
        throw new Error(`non-40 character 'shallow' oid: ${oid}`);
      }
      oids.add(oid);
      await GitShallowManager.write({ fs, gitdir, oids });
    } else if (line.startsWith('unshallow')) {
      let oid = line.slice(-41).trim();
      if (oid.length !== 40) {
        throw new Error(`non-40 character 'shallow' oid: ${oid}`);
      }
      oids.delete(oid);
      await GitShallowManager.write({ fs, gitdir, oids });
    }
    next(null, data);
  }));
  return response;
}

/**
 * @ignore
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {ReadableStream} args.inputStream
 * @param {Function} args.onprogress
 */

async function writeTreeToDisk({ fs: _fs, dir, gitdir, index, prefix, tree }) {
  const fs = new FileSystem(_fs);
  for (let entry of tree) {
    let { type, object } = await GitObjectManager.read({
      fs,
      gitdir,
      oid: entry.oid
    });
    let entrypath = prefix === '' ? entry.path : `${prefix}/${entry.path}`;
    let filepath = path.join(dir, prefix, entry.path);
    switch (type) {
      case 'blob':
        await fs.write(filepath, object);
        let stats = await fs._lstat(filepath);
        index.insert({
          filepath: entrypath,
          stats,
          oid: entry.oid
        });
        break;
      case 'tree':
        let tree = GitTree.from(object);
        await writeTreeToDisk({
          fs,
          dir,
          gitdir,
          index,
          prefix: entrypath,
          tree
        });
        break;
      default:
        throw new Error(`Unexpected object type ${type} found in tree for '${entrypath}'`);
    }
  }
}

/**
 * Checkout a branch
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} [args.remote='origin'] - What to name the remote that is created. The default is 'origin'.
 * @param {string} [args.ref=undefined] - Which branch to clone. By default this is the designated "main branch" of the repository.
 * @returns {Promise<void>} - Resolves successfully when filesystem operations are complete.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * await git.checkout({...repo, ref: '<@master@>'})
 * console.log('done')
 */
async function checkout({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  remote,
  ref
}) {
  const fs = new FileSystem(_fs);
  // Get tree oid
  let oid;
  if (remote) {
    let remoteRef;
    if (ref === undefined) {
      remoteRef = await GitRefManager.resolve({
        fs,
        gitdir,
        ref: `${remote}/HEAD`,
        depth: 2
      });
      ref = path.basename(remoteRef);
    } else {
      remoteRef = `${remote}/${ref}`;
    }
    oid = await GitRefManager.resolve({ fs, gitdir, ref: remoteRef });
    // Make the remote ref our own!
    await fs.write(`${gitdir}/refs/heads/${ref}`, oid + '\n');
  } else {
    if (ref === undefined) {
      throw new Error('Cannot checkout ref "undefined"');
    }
    oid = await GitRefManager.resolve({ fs, gitdir, ref });
  }
  let commit = await GitObjectManager.read({ fs, gitdir, oid });
  if (commit.type !== 'commit') {
    throw new Error(`Unexpected type: ${commit.type}`);
  }
  let comm = GitCommit.from(commit.object.toString('utf8'));
  let sha = comm.headers().tree;
  // Get top-level tree
  let { type, object } = await GitObjectManager.read({ fs, gitdir, oid: sha });
  if (type !== 'tree') throw new Error(`Unexpected type: ${type}`);
  let tree = GitTree.from(object);
  // Acquire a lock on the index
  await GitIndexManager.acquire({ fs, filepath: `${gitdir}/index` }, async function (index) {
    // TODO: Big optimization possible here.
    // Instead of deleting and rewriting everything, only delete files
    // that are not present in the new branch, and only write files that
    // are not in the index or are in the index but have the wrong SHA.
    for (let entry of index) {
      try {
        await fs.rm(path.join(dir, entry.path));
      } catch (err) {}
    }
    index.clear();
    // Write files. TODO: Write them atomically
    await writeTreeToDisk({ fs, gitdir, dir, index, prefix: '', tree });
    // Update HEAD TODO: Handle non-branch cases
    await fs.write(`${gitdir}/HEAD`, `ref: refs/heads/${ref}`);
  });
}

/**
 * Clone a repository
 *
 * @link https://isomorphic-git.github.io/docs/clone.html
 */
async function clone({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  emitter,
  url,
  ref,
  remote,
  authUsername,
  authPassword,
  depth,
  since,
  exclude,
  relative,
  onprogress
}) {
  if (onprogress !== undefined) {
    console.warn('The `onprogress` callback has been deprecated. Please use the more generic `emitter` EventEmitter argument instead.');
  }
  const fs = new FileSystem(_fs);
  remote = remote || 'origin';
  await init({ gitdir, fs });
  // Add remote
  await config({
    gitdir,
    fs,
    path: `remote.${remote}.url`,
    value: url
  });
  // Fetch commits
  await fetch({
    gitdir,
    fs,
    emitter,
    ref,
    remote,
    authUsername,
    authPassword,
    depth,
    since,
    exclude,
    relative
  });
  // Checkout branch
  await checkout({
    dir,
    gitdir,
    fs,
    ref,
    remote
  });
}

async function constructTree({ fs, gitdir, inode }) /*: string */{
  // use depth first traversal
  let children = inode.children;
  for (let inode of children) {
    if (inode.type === 'tree') {
      inode.metadata.mode = '040000';
      inode.metadata.oid = await constructTree({ fs, gitdir, inode });
    }
  }
  let entries = children.map(inode => ({
    mode: inode.metadata.mode,
    path: inode.basename,
    oid: inode.metadata.oid,
    type: inode.type
  }));
  const tree = GitTree.from(entries);
  let oid = await GitObjectManager.write({
    fs,
    gitdir,
    type: 'tree',
    object: tree.toObject()
  });
  return oid;
}

/**
 * Create a new commit
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.message - The commit message to use.
 * @param {Object} [args.author] - The details about the commit author.
 * @param {string} [args.author.name=undefined] - Default is `user.name` config.
 * @param {string} [args.author.email=undefined] - Default is `user.email` config.
 * @param {Date} [args.author.date=new Date()] - Set the author timestamp field. Default is the current date.
 * @param {number} [args.author.timestamp=undefined] - Set the author timestamp field. This is an alternative to using `date` using an integer number of seconds since the Unix epoch instead of a JavaScript date object.
 * @param {Object} [args.committer=author] - The details about the commit committer, in the same format as the author parameter. If not specified, the author details are used.
 * @returns {Promise<string>} - The object ID of the newly created commit.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * let sha = await git.commit({
 *   ...repo,
 *   author: {
 *     name: '<@Mr. Test@>',
 *     email: '<@mrtest@example.com@>'
 *   },
 *   message: '<@Added the a.txt file@>'
 * })
 * console.log(sha)
 */
async function commit({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  message,
  author,
  committer
}) {
  const fs = new FileSystem(_fs);
  // Fill in missing arguments with default values
  if (author === undefined) author = {};
  if (author.name === undefined) {
    author.name = await config({ fs, gitdir, path: 'user.name' });
  }
  if (author.email === undefined) {
    author.email = await config({ fs, gitdir, path: 'user.email' });
  }
  committer = committer || author;
  let authorDateTime = author.date || new Date();
  let committerDateTime = committer.date || authorDateTime;
  let oid;
  await GitIndexManager.acquire({ fs, filepath: `${gitdir}/index` }, async function (index) {
    const inode = flatFileListToDirectoryStructure(index.entries);
    const treeRef = await constructTree({ fs, gitdir, inode });
    let parents;
    try {
      let parent = await GitRefManager.resolve({ fs, gitdir, ref: 'HEAD' });
      parents = [parent];
    } catch (err) {
      // Probably an initial commit
      parents = [];
    }
    let comm = GitCommit.from({
      tree: treeRef,
      parent: parents,
      author: {
        name: author.name,
        email: author.email,
        timestamp: author.timestamp || Math.floor(authorDateTime.valueOf() / 1000),
        timezoneOffset: author.timezoneOffset || 0
      },
      committer: {
        name: committer.name,
        email: committer.email,
        timestamp: committer.timestamp || Math.floor(committerDateTime.valueOf() / 1000),
        timezoneOffset: committer.timezoneOffset || 0
      },
      message
    });
    oid = await GitObjectManager.write({
      fs,
      gitdir,
      type: 'commit',
      object: comm.toObject()
    });
    // Update branch pointer
    const branch = await GitRefManager.resolve({
      fs,
      gitdir,
      ref: 'HEAD',
      depth: 2
    });
    await fs.write(path.join(gitdir, branch), oid + '\n');
  });
  return oid;
}

/**
 * List all the tracked files in a repo
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @returns {Promise<string[]>} - Resolves successfully with an array of file paths.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * let files = await git.listFiles(repo)
 * console.log(files)
 */
async function listFiles({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs
}) {
  const fs = new FileSystem(_fs);
  let filenames;
  await GitIndexManager.acquire({ fs, filepath: `${gitdir}/index` }, async function (index) {
    filenames = index.entries.map(x => x.path);
  });
  return filenames;
}

/**
 * List branches
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} [remote=undefined] - If specified, lists the branches for that remote. Otherwise lists local branches.
 * @returns {Promise<string[]>} - Resolves successfully with an array of branch names.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * let branches = await git.listBranches(repo)
 * console.log(branches)
 */
async function listBranches({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  remote = undefined
}) {
  const fs = new FileSystem(_fs);
  return GitRefManager.listBranches({ fs, gitdir, remote });
}

/**
 * List all local tags
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @returns {Promise<string[]>} - Resolves successfully with an array of branch names.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * let tags = await git.listTags(repo)
 * console.log(tags)
 */
async function listTags({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs
}) {
  const fs = new FileSystem(_fs);
  return GitRefManager.listTags({ fs, gitdir });
}

/**
 * @typedef {Object} CommitDescription
 * @property {string} oid - SHA1 object id of this commit
 * @property {string} message - Commit message
 * @property {string} tree - SHA1 object id of corresponding file tree
 * @property {string[]} parent - an array of zero or more SHA1 oids
 * @property {Object} author
 * @property {string} author.name
 * @property {string} author.email
 * @property {number} author.timestamp - UTC Unix timestamp in seconds
 * @property {number} author.timezoneOffset - Timezone difference from UTC in minutes
 * @property {Object} committer
 * @property {string} committer.name
 * @property {string} committer.email
 * @property {number} committer.timestamp - UTC Unix timestamp in seconds
 * @property {number} committer.timezoneOffset - Timezone difference from UTC in minutes
 * @property {string} [gpgsig] - PGP signature (if present)
 */

/**
 * Get commit descriptions from the git history
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {number} [args.depth=undefined] - Limit the number of commits returned. No limit by default.
 * @param {Date} [args.since=undefined] - Return history newer than the given date. Can be combined with `depth` to get whichever is shorter.
 * @param {string} [args.ref=HEAD] - The commit to begin walking backwards through the history from.
 * @returns {Promise<CommitDescription[]>} - Resolves to an array of {@link CommitDescription} objects
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * let commits = await git.log({...repo, depth: 5, ref: '<@master@>'})
 * console.log(commits)
 */
async function log$1({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  ref = 'HEAD',
  depth,
  since // Date
}) {
  const fs = new FileSystem(_fs);
  let sinceTimestamp = since === undefined ? undefined : Math.floor(since.valueOf() / 1000);
  // TODO: In the future, we may want to have an API where we return a
  // async iterator that emits commits.
  let commits = [];
  let start = await GitRefManager.resolve({ fs, gitdir, ref });
  let { type, object } = await GitObjectManager.read({ fs, gitdir, oid: start });
  if (type !== 'commit') {
    throw new Error(`The given ref ${ref} did not resolve to a commit but to a ${type}`);
  }
  let currentCommit = _extends({ oid: start }, GitCommit.from(object).parse());
  commits.push(currentCommit);
  while (true) {
    if (depth !== undefined && commits.length === depth) break;
    if (currentCommit.parent.length === 0) break;
    let oid = currentCommit.parent[0];
    let gitobject;
    try {
      gitobject = await GitObjectManager.read({ fs, gitdir, oid });
    } catch (err) {
      commits.push({
        oid,
        error: err
      });
      break;
    }
    let { type, object } = gitobject;
    if (type !== 'commit') {
      commits.push({
        oid,
        error: new Error(`Invalid commit parent ${oid} is of type ${type}`)
      });
      break;
    }
    currentCommit = _extends({ oid }, GitCommit.from(object).parse());
    if (sinceTimestamp !== undefined && currentCommit.author.timestamp <= sinceTimestamp) {
      break;
    }
    commits.push(currentCommit);
  }
  return commits;
}

const types$1 = {
  commit: 0b0010000,
  tree: 0b0100000,
  blob: 0b0110000,
  tag: 0b1000000,
  ofs_delta: 0b1100000,
  ref_delta: 0b1110000

  /**
   *
   * If there were no errors, then there will be no `errors` property.
   * There can be a mix of `ok` messages and `errors` messages.
   *
   * @typedef {Object} PushResponse
   * @property {Array<string>} [ok] - The first item is "unpack" if the overall operation was successful. The remaining items are the names of refs that were updated successfully.
   * @property {Array<string>} [errors] - If the overall operation threw and error, the first item will be "unpack {Overall error message}". The remaining items are individual refs that failed to be updated in the format "{ref name} {error message}".
   */

  /**
   * Push a branch
   *
   * @param {Object} args - Arguments object
   * @param {FSModule} args.fs - The filesystem holding the git repo
   * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
   * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
   * @param {string} [args.ref=undefined] - Which branch to push. By default this is the currently checked out branch of the repository.
   * @param {string} [args.remote='origin'] - If URL is not specified, determines which remote to use.
   * @param {string} [args.url=undefined] - The URL of the remote git server. The default is the value set in the git config for that remote.
   * @param {string} [args.authUsername=undefined] - The username to use with Basic Auth
   * @param {string} [args.authPassword=undefined] - The password to use with Basic Auth
   * @returns {Promise<PushResponse>} - Resolves successfully when push completes with a detailed description of the operation from the server.
   *
   * @example
   * let repo = {fs, dir: '<@.@>'}
   * let pushResponse = await git.push({
   *   ...repo,
   *   remote: '<@origin@>',
   *   ref: '<@master@>',
   *   authUsername: <@process.env.GITHUB_TOKEN@>,
   *   authPassword: <@process.env.GITHUB_TOKEN@>
   * })
   * console.log(pushResponse)
   */
};async function push({
  fs: _fs,
  dir,
  gitdir = path.join(dir, '.git'),
  ref,
  remote = 'origin',
  url,
  authUsername,
  authPassword
}) {
  const fs = new FileSystem(_fs);
  // TODO: Figure out how pushing tags works. (This only works for branches.)
  if (url === undefined) {
    url = await config({ fs, gitdir, path: `remote.${remote}.url` });
  }
  let fullRef = ref.startsWith('refs/') ? ref : `refs/heads/${ref}`;
  let oid = await GitRefManager.resolve({ fs, gitdir, ref });
  let httpRemote = new GitRemoteHTTP(url);
  if (authUsername !== undefined && authPassword !== undefined) {
    httpRemote.auth = {
      username: authUsername,
      password: authPassword
    };
  }
  await httpRemote.preparePush();
  let commits = await listCommits({
    fs,
    gitdir,
    start: [oid],
    finish: httpRemote.refs.values()
  });
  let objects = await listObjects({ fs, gitdir, oids: commits });
  let packstream = new stream.PassThrough();
  let oldoid = httpRemote.refs.get(fullRef) || '0000000000000000000000000000000000000000';
  packstream.write(GitPktLine.encode(`${oldoid} ${oid} ${fullRef}\0 report-status\n`));
  packstream.write(GitPktLine.flush());
  pack({
    fs,
    gitdir,
    oids: [...objects],
    outputStream: packstream
  });
  let response = await httpRemote.push(packstream);
  return response;
}

/** @ignore */
async function listCommits({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  start,
  finish
}) {
  const fs = new FileSystem(_fs);
  let startingSet = new _Set();
  let finishingSet = new _Set();
  for (let ref of start) {
    startingSet.add((await GitRefManager.resolve({ fs, gitdir, ref })));
  }
  for (let ref of finish) {
    // We may not have these refs locally so we must try/catch
    try {
      let oid = await GitRefManager.resolve({ fs, gitdir, ref });
      finishingSet.add(oid);
    } catch (err) {}
  }
  let visited /*: Set<string> */ = new _Set();

  // Because git commits are named by their hash, there is no
  // way to construct a cycle. Therefore we won't worry about
  // setting a default recursion limit.
  async function walk(oid) {
    visited.add(oid);
    let { type, object } = await GitObjectManager.read({ fs, gitdir, oid });
    if (type !== 'commit') {
      throw new Error(`Expected type commit but type is ${type}`);
    }
    let commit = GitCommit.from(object);
    let parents = commit.headers().parent;
    for (oid of parents) {
      if (!finishingSet.has(oid) && !visited.has(oid)) {
        await walk(oid);
      }
    }
  }

  // Let's go walking!
  for (let oid of startingSet) {
    await walk(oid);
  }
  return visited;
}

/** @ignore */
async function listObjects({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  oids
}) {
  const fs = new FileSystem(_fs);
  let visited /*: Set<string> */ = new _Set();

  // We don't do the purest simplest recursion, because we can
  // avoid reading Blob objects entirely since the Tree objects
  // tell us which oids are Blobs and which are Trees.
  async function walk(oid) {
    visited.add(oid);
    let { type, object } = await GitObjectManager.read({ fs, gitdir, oid });
    if (type === 'commit') {
      let commit = GitCommit.from(object);
      let tree = commit.headers().tree;
      await walk(tree);
    } else if (type === 'tree') {
      let tree = GitTree.from(object);
      for (let entry /*: TreeEntry */ of tree) {
        visited.add(entry.oid);
        // only recurse for trees
        if (entry.type === 'tree') {
          await walk(entry.oid);
        }
      }
    }
  }

  // Let's go walking!
  for (let oid of oids) {
    await walk(oid);
  }
  return visited;
}

/** @ignore */
async function pack({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  oids,
  outputStream
}) {
  const fs = new FileSystem(_fs);
  let hash = createHash('sha1');
  function write(chunk, enc) {
    outputStream.write(chunk, enc);
    hash.update(chunk, enc);
  }
  function writeObject({ stype, object }) {
    let lastFour, multibyte, length;
    // Object type is encoded in bits 654
    let type = types$1[stype];
    if (type === undefined) throw new Error('Unrecognized type: ' + stype);
    // The length encoding get complicated.
    length = object.length;
    // Whether the next byte is part of the variable-length encoded number
    // is encoded in bit 7
    multibyte = length > 0b1111 ? 0b10000000 : 0b0;
    // Last four bits of length is encoded in bits 3210
    lastFour = length & 0b1111;
    // Discard those bits
    length = length >>> 4;
    // The first byte is then (1-bit multibyte?), (3-bit type), (4-bit least sig 4-bits of length)
    let byte = (multibyte | type | lastFour).toString(16);
    write(byte, 'hex');
    // Now we keep chopping away at length 7-bits at a time until its zero,
    // writing out the bytes in what amounts to little-endian order.
    while (multibyte) {
      multibyte = length > 0b01111111 ? 0b10000000 : 0b0;
      byte = multibyte | length & 0b01111111;
      write(pad(2, byte.toString(16), '0'), 'hex');
      length = length >>> 7;
    }
    // Lastly, we can compress and write the object.
    write(buffer.Buffer.from(pako.deflate(object)));
  }

  write('PACK');
  write('00000002', 'hex');
  // Write a 4 byte (32-bit) int
  write(pad(8, oids.length.toString(16), '0'), 'hex');
  for (let oid of oids) {
    let { type, object } = await GitObjectManager.read({ fs, gitdir, oid });
    writeObject({ write, object, stype: type });
  }
  // Write SHA1 checksum
  let digest = hash.digest();
  outputStream.end(digest);
  return outputStream;
}

/**
 * Remove a file from the git index (aka staging area)
 *
 * Note that this does NOT delete the file in the working directory.
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.filepath - The path to the file to remove to the index.
 * @returns {Promise<void>} - Resolves successfully once the git index has been updated.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * await git.remove({...repo, filepath: '<@README.md@>'})
 * console.log('done')
 */
async function remove({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  filepath
}) {
  const fs = new FileSystem(_fs);
  await GitIndexManager.acquire({ fs, filepath: `${gitdir}/index` }, async function (index) {
    index.delete({ filepath });
  });
  // TODO: return oid?
}

/**
 * Verify a signed commit
 *
 * It is up to you to figure out what the commit's public key *should* be.
 * I would use the "author" or "committer" name and email, and look up
 * that person's public key from a trusted source such as the Github API.
 *
 * The function returns false if any of the signatures on a signed git commit are invalid.
 * Otherwise, it returns an array of the key ids that were used to sign it.
 *
 * The {@link publicKeys} argument is a single string in ASCII armor format. However, it is plural "keys" because
 * you can technically have multiple public keys in a single ASCII armor string. While I haven't tested it, it
 * should support verifying a single commit signed with multiple keys. Hence why the returned result is an array of key ids.
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.ref - A reference to the commit to verify
 * @param {string} args.publicKeys - A PGP public key in ASCII armor format.
 * @returns {Promise<false|Array<string>>} - The key ids used to sign the commit, in hex format.
 *
 * @example
 * let repo = {fs, dir: '.'}
 * let keyids = await git.verify({
 *   ...repo,
 *   ref: '<@HEAD@>',
 *   publicKeys: `<<@
 * -----BEGIN PGP PUBLIC KEY BLOCK-----
 * ...
 * @>>`
 * })
 * console.log(keyids)
 */
async function verify({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  ref,
  publicKeys,
  openpgp
}) {
  const fs = new FileSystem(_fs);
  const oid = await GitRefManager.resolve({ fs, gitdir, ref });
  const { type, object } = await GitObjectManager.read({ fs, gitdir, oid });
  if (type !== 'commit') {
    throw new Error(`'ref' is not pointing to a 'commit' object but a '${type}' object`);
  }
  let commit = SignedGitCommit.from(object);
  let keys = await commit.listSigningKeys(openpgp);
  let validity = await commit.verify(openpgp, publicKeys);
  if (!validity) return false;
  return keys;
}

/*::
import type { Stats } from 'fs'
import type { CacheEntry } from '../models/GitIndex'
*/

function cacheIsStale({ entry, stats /*: {
                                     entry: CacheEntry,
                                     stats: Stats
                                     } */
}) {
  // Comparison based on the description in Paragraph 4 of
  // https://www.kernel.org/pub/software/scm/git/docs/technical/racy-git.txt
  return entry.mode !== stats.mode || entry.mtime.valueOf() !== stats.mtime.valueOf() || entry.ctime.valueOf() !== stats.ctime.valueOf() || entry.uid !== stats.uid || entry.gid !== stats.gid || entry.ino !== stats.ino >> 0 || entry.size !== stats.size;
}

async function getOidAtPath({ fs, gitdir, tree, path: path$$1 }) {
  if (typeof path$$1 === 'string') path$$1 = path$$1.split('/');
  let dirname = path$$1.shift();
  for (let entry of tree) {
    if (entry.path === dirname) {
      if (path$$1.length === 0) {
        return entry.oid;
      }
      let { type, object } = await GitObjectManager.read({
        fs,
        gitdir,
        oid: entry.oid
      });
      if (type === 'tree') {
        let tree = GitTree.from(object);
        return getOidAtPath({ fs, gitdir, tree, path: path$$1 });
      }
      if (type === 'blob') {
        throw new Error(`Blob found where tree expected.`);
      }
    }
  }
  return null;
}

async function getHeadTree({ fs, gitdir }) {
  // Get the tree from the HEAD commit.
  let oid = await GitRefManager.resolve({ fs, gitdir, ref: 'HEAD' });
  let { object: cobject } = await GitObjectManager.read({ fs, gitdir, oid });
  let commit = GitCommit.from(cobject);
  let { object: tobject } = await GitObjectManager.read({
    fs,
    gitdir,
    oid: commit.parseHeaders().tree
  });
  let tree = GitTree.from(tobject).entries();
  return tree;
}

/**
 * Tell whether a file has been changed
 *
 * The possible resolve values are:
 *
 * - `"ignored"` file ignored by a .gitignore rule
 * - `"unmodified"` file unchanged from HEAD commit
 * - `"*modified"` file has modifications, not yet staged
 * - `"*deleted"` file has been removed, but the removal is not yet staged
 * - `"*added"` file is untracked, not yet staged
 * - `"absent"` file not present in HEAD commit, staging area, or working dir
 * - `"modified"` file has modifications, staged
 * - `"deleted"` file has been removed, staged
 * - `"added"` previously untracked file, staged
 * - `"*unmodified"` working dir and HEAD commit match, but index differs
 * - `"*absent"` file not present in working dir or HEAD commit, but present in the index
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.filepath - The path to the file to query.
 * @returns {Promise<string>} - Resolves successfully with the file's git status.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * let status = await git.status({...repo, filepath: '<@README.md@>'})
 * console.log(status)
 */
async function status({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  filepath
}) {
  const fs = new FileSystem(_fs);
  let ignored = await GitIgnoreManager.isIgnored({
    gitdir,
    dir,
    filepath,
    fs
  });
  if (ignored) {
    return 'ignored';
  }
  let headTree = await getHeadTree({ fs, gitdir });
  let treeOid = await getOidAtPath({
    fs,
    gitdir,
    tree: headTree,
    path: filepath
  });
  let indexEntry = null;
  // Acquire a lock on the index
  await GitIndexManager.acquire({ fs, filepath: `${gitdir}/index` }, async function (index) {
    for (let entry of index) {
      if (entry.path === filepath) {
        indexEntry = entry;
        break;
      }
    }
  });
  let stats = null;
  try {
    stats = await fs._lstat(path.join(dir, filepath));
  } catch (err) {
    if (err.code !== 'ENOENT') {
      throw err;
    }
  }

  let H = treeOid !== null; // head
  let I = indexEntry !== null; // index
  let W = stats !== null; // working dir

  const getWorkdirOid = async () => {
    if (I && !cacheIsStale({ entry: indexEntry, stats })) {
      return indexEntry.oid;
    } else {
      let object = await fs.read(path.join(dir, filepath));
      let workdirOid = await GitObjectManager.hash({
        gitdir,
        type: 'blob',
        object
      });
      return workdirOid;
    }
  };

  if (!H && !W && !I) return 'absent'; // ---
  if (!H && !W && I) return '*absent'; // -A-
  if (!H && W && !I) return '*added'; // --A
  if (!H && W && I) {
    let workdirOid = await getWorkdirOid();
    return workdirOid === indexEntry.oid ? 'added' : '*added'; // -AA : -AB
  }
  if (H && !W && !I) return 'deleted'; // A--
  if (H && !W && I) {
    return treeOid === indexEntry.oid ? '*deleted' : '*deleted'; // AA- : AB-
  }
  if (H && W && !I) {
    let workdirOid = await getWorkdirOid();
    return workdirOid === treeOid ? '*undeleted' : '*undeletemodified'; // A-A : A-B
  }
  if (H && W && I) {
    let workdirOid = await getWorkdirOid();
    if (workdirOid === treeOid) {
      return workdirOid === indexEntry.oid ? 'unmodified' : '*unmodified'; // AAA : ABA
    } else {
      return workdirOid === indexEntry.oid ? 'modified' : '*modified'; // ABB : AAB
    }
  }
  /*
  ---
  -A-
  --A
  -AA
  -AB
  A--
  AA-
  AB-
  A-A
  A-B
  AAA
  ABA
  ABB
  AAB
  */
}

/**
 * Find the root git directory
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.filepath - The file directory to start searching in.
 * @returns {Promise<string>} - a directory name
 * @throws {Error} - Error('Unable to find git root')
 *
 * Starting at `filepath`, will walk upwards until it finds a directory that contains a directory called '.git'.
 *
 * @example
 * let gitroot = await git.findRoot({
 *   fs,
 *   filepath: '<@/path/to/some/gitrepo/path/to/some/file.txt@>'
 * })
 * console.log(gitroot) // '/path/to/some/gitrepo'
 */
async function findRoot({ fs: _fs, filepath }) {
  const fs = new FileSystem(_fs);
  return _findRoot(fs, filepath);
}

async function _findRoot(fs, filepath) {
  if (await fs.exists(path.join(filepath, '.git'))) {
    return filepath;
  } else {
    let parent = path.dirname(filepath);
    if (parent === filepath) throw new Error('Unable to find git root');
    return _findRoot(fs, parent);
  }
}

/**
 * Return the version number of 'isomorphic-git'
 *
 * I don't know why you might need this. I added it just so I could check that I was getting
 * the correct version of the library and not a cached version.
 *
 * TODO: Semantic-release broke this, now it always says '0.0.0-development'. Need to add a
 * prepublishOnly script to find & replace that with the actual version number.
 *
 * @returns {string} version - the version string taken from package.json at publication time
 * @example
 * console.log(git.version())
 */
function version$1() {
  return version;
}

/**
 * Create the .idx file for a given .pack file
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.filepath - The path to the .pack file to index.
 * @returns {Promise<void>} - Resolves successfully once the .idx file been written.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * await git.indexPack({...repo, filepath: '<@pack-9cbd243a1caa4cb4bef976062434a958d82721a9.pack@>'})
 */
async function indexPack({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  filepath
}) {
  const fs = new FileSystem(_fs);
  const pack = await fs.read(path.join(dir, filepath));
  const idx = await GitPackIndex.fromPack({ pack });
  await fs.write(filepath.replace(/\.pack$/, '.idx'), idx.toBuffer());
}

/**
 * Create a signed commit
 *
 * OpenPGP.js is a huge library and if you don't need to create or verify signed commits
 * you shouldn't be forced to include that weighty feature in your bundle. That's why this
 * is its own function.
 *
 * It creates a signed version of whatever commit HEAD currently points to, and then updates the current branch,
 * leaving the original commit dangling.
 *
 * The {@link privateKeys} argument is a single string in ASCII armor format. However, it is plural "keys" because
 * you can technically have multiple private keys in a single ASCII armor string. The openpgp.sign() function accepts
 * multiple keys, so while I haven't tested it, it should support signing a single commit with multiple keys.
 *
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.privateKeys - A PGP private key in ASCII armor format.
 * @returns {Promise<string>} - The object ID of the newly created commit.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * let sha = await git.sign({
 *   ...repo,
 *   privateKeys: `<<@
 * -----BEGIN PGP PRIVATE KEY BLOCK-----
 * ...
 * @>>`
 * })
 * console.log(sha)
 */
async function sign({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  privateKeys,
  openpgp
}) {
  const fs = new FileSystem(_fs);
  const oid = await GitRefManager.resolve({ fs, gitdir, ref: 'HEAD' });
  const { type, object } = await GitObjectManager.read({ fs, gitdir, oid });
  if (type !== 'commit') {
    throw new Error(`HEAD is not pointing to a 'commit' object but a '${type}' object`);
  }
  let commit = SignedGitCommit.from(object);
  commit = await commit.sign(openpgp, privateKeys);
  const newOid = await GitObjectManager.write({
    fs,
    gitdir,
    type: 'commit',
    object: commit.toObject()
  });
  // Update branch pointer
  // TODO: Use an updateBranch function instead of this.
  const branch = await GitRefManager.resolve({
    fs,
    gitdir,
    ref: 'HEAD',
    depth: 2
  });
  await fs.write(path.join(gitdir, branch), newOid + '\n');
}

/**
 * Get the value of a symbolic ref or resolve a ref to its object id.
 * @param {Object} args - Arguments object
 * @param {FSModule} args.fs - The filesystem holding the git repo
 * @param {string} args.dir - The path to the [working tree](index.html#dir-vs-gitdir) directory
 * @param {string} [args.gitdir=path.join(dir, '.git')] - The path to the [git directory](index.html#dir-vs-gitdir)
 * @param {string} args.ref - Which ref to resolve.
 * @param {number} [args.depth=undefined] - How many symbolic references to follow before returning.
 * @returns {Promise<string>} - Resolves successfully with the SHA, or the value of another symbolic ref.
 *
 * @example
 * let repo = {fs, dir: '<@.@>'}
 * let currentCommit = await git.resolveRef({...repo, ref: '<@HEAD@>'})
 * console.log(currentCommit)
 * let currentBranch = await git.resolveRef({...repo, ref: '<@HEAD@>', depth: 1})
 * console.log(currentBranch)
 */
async function resolveRef({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  ref,
  depth
}) {
  const fs = new FileSystem(_fs);
  return GitRefManager.resolve({
    fs,
    gitdir,
    ref,
    depth
  });
}

async function readObject({
  dir,
  gitdir = path.join(dir, '.git'),
  fs: _fs,
  oid,
  format = 'parsed'
}) {
  const fs = new FileSystem(_fs);
  // GitObjectManager does not know how to parse content, so we tweak that parameter before passing it.
  const _format = format === 'parsed' ? 'content' : format;
  let result = await GitObjectManager.read({ fs, gitdir, oid, format: _format });
  if (format === 'parsed') {
    switch (result.type) {
      case 'commit':
        result.object = GitCommit.from(result.object).parse();
        break;
      case 'tree':
        result.object = { entries: GitTree.from(result.object).entries() };
        break;
      case 'blob':
        break;
      case 'tag':
        throw new Error('TODO: Parsing annotated tag objects still needs to be implemented!!');
      default:
        throw new Error(`Unrecognized git object type: '${result.type}'`);
    }
    result.format = 'parsed';
  }
  return result;
}

/** @ignore */
const utils = { auth, oauth2 };

exports.utils = utils;
exports.add = add;
exports.clone = clone;
exports.checkout = checkout;
exports.commit = commit;
exports.fetch = fetch;
exports.init = init;
exports.listFiles = listFiles;
exports.listBranches = listBranches;
exports.listTags = listTags;
exports.log = log$1;
exports.push = push;
exports.remove = remove;
exports.config = config;
exports.verify = verify;
exports.status = status;
exports.findRoot = findRoot;
exports.version = version$1;
exports.indexPack = indexPack;
exports.sign = sign;
exports.resolveRef = resolveRef;
exports.readObject = readObject;
//# sourceMappingURL=index.js.map
