var ElementsRenderer, NodeElement, html_escape, node_element, util;

node_element = require('./node_element');

NodeElement = node_element.NodeElement;

util = require('./util');

html_escape = util.html_escape;

ElementsRenderer = (function() {
  function ElementsRenderer(tree_widget) {
    this.tree_widget = tree_widget;
    this.opened_icon_element = this.createButtonElement(tree_widget.options.openedIcon);
    this.closed_icon_element = this.createButtonElement(tree_widget.options.closedIcon);
  }

  ElementsRenderer.prototype.render = function(from_node) {
    if (from_node && from_node.parent) {
      return this.renderFromNode(from_node);
    } else {
      return this.renderFromRoot();
    }
  };

  ElementsRenderer.prototype.renderFromRoot = function() {
    var $element;
    $element = this.tree_widget.element;
    $element.empty();
    return this.createDomElements($element[0], this.tree_widget.tree.children, true, true);
  };

  ElementsRenderer.prototype.renderFromNode = function(node) {
    var $previous_li, li;
    $previous_li = $(node.element);
    li = this.createLi(node);
    this.attachNodeData(node, li);
    $previous_li.after(li);
    $previous_li.remove();
    if (node.children) {
      return this.createDomElements(li, node.children, false, false);
    }
  };

  ElementsRenderer.prototype.createDomElements = function(element, children, is_root_node, is_open) {
    var child, i, len, li, ul;
    ul = this.createUl(is_root_node);
    element.appendChild(ul);
    for (i = 0, len = children.length; i < len; i++) {
      child = children[i];
      li = this.createLi(child);
      ul.appendChild(li);
      this.attachNodeData(child, li);
      if (child.hasChildren()) {
        this.createDomElements(li, child.children, false, child.is_open);
      }
    }
    return null;
  };

  ElementsRenderer.prototype.attachNodeData = function(node, li) {
    node.element = li;
    return $(li).data('node', node);
  };

  ElementsRenderer.prototype.createUl = function(is_root_node) {
    var class_string, ul;
    if (is_root_node) {
      class_string = 'jqtree-tree';
    } else {
      class_string = '';
    }
    ul = document.createElement('ul');
    ul.className = "jqtree_common " + class_string;
    return ul;
  };

  ElementsRenderer.prototype.createLi = function(node) {
    var li;
    if (node.isFolder()) {
      li = this.createFolderLi(node);
    } else {
      li = this.createNodeLi(node);
    }
    if (this.tree_widget.options.onCreateLi) {
      this.tree_widget.options.onCreateLi(node, $(li));
    }
    return li;
  };

  ElementsRenderer.prototype.createFolderLi = function(node) {
    var button_classes, button_link, div, escaped_name, folder_classes, icon_element, li, title_span;
    button_classes = this.getButtonClasses(node);
    folder_classes = this.getFolderClasses(node);
    escaped_name = this.escapeIfNecessary(node.name);
    if (node.is_open) {
      icon_element = this.opened_icon_element;
    } else {
      icon_element = this.closed_icon_element;
    }
    li = document.createElement('li');
    li.className = "jqtree_common " + folder_classes;
    div = document.createElement('div');
    div.className = "jqtree-element jqtree_common";
    li.appendChild(div);
    button_link = document.createElement('a');
    button_link.className = "jqtree_common " + button_classes;
    button_link.appendChild(icon_element.cloneNode(false));
    div.appendChild(button_link);
    title_span = document.createElement('span');
    title_span.className = "jqtree_common jqtree-title jqtree-title-folder";
    div.appendChild(title_span);
    title_span.innerHTML = escaped_name;
    return li;
  };

  ElementsRenderer.prototype.createNodeLi = function(node) {
    var class_string, div, escaped_name, li, li_classes, title_span;
    li_classes = ['jqtree_common'];
    if (this.tree_widget.select_node_handler && this.tree_widget.select_node_handler.isNodeSelected(node)) {
      li_classes.push('jqtree-selected');
    }
    class_string = li_classes.join(' ');
    escaped_name = this.escapeIfNecessary(node.name);
    li = document.createElement('li');
    li.className = class_string;
    div = document.createElement('div');
    div.className = "jqtree-element jqtree_common";
    li.appendChild(div);
    title_span = document.createElement('span');
    title_span.className = "jqtree-title jqtree_common";
    title_span.innerHTML = escaped_name;
    div.appendChild(title_span);
    return li;
  };

  ElementsRenderer.prototype.getButtonClasses = function(node) {
    var classes;
    classes = ['jqtree-toggler'];
    if (!node.is_open) {
      classes.push('jqtree-closed');
    }
    return classes.join(' ');
  };

  ElementsRenderer.prototype.getFolderClasses = function(node) {
    var classes;
    classes = ['jqtree-folder'];
    if (!node.is_open) {
      classes.push('jqtree-closed');
    }
    if (this.tree_widget.select_node_handler && this.tree_widget.select_node_handler.isNodeSelected(node)) {
      classes.push('jqtree-selected');
    }
    if (node.is_loading) {
      classes.push('jqtree-loading');
    }
    return classes.join(' ');
  };

  ElementsRenderer.prototype.escapeIfNecessary = function(value) {
    if (this.tree_widget.options.autoEscape) {
      return html_escape(value);
    } else {
      return value;
    }
  };

  ElementsRenderer.prototype.createButtonElement = function(value) {
    var div;
    if (typeof value === 'string') {
      div = document.createElement('div');
      div.innerHTML = value;
      return document.createTextNode(div.innerHTML);
    } else {
      return $(value)[0];
    }
  };

  return ElementsRenderer;

})();

module.exports = ElementsRenderer;
