/**
 * x509.js
 */
import { AsnFormat, X509EncodingOptions, DER, PEM, SignatureType } from './typedef';
/**
 * Convert public key in JWK format to X.509 PEM or DER object.
 * @param {JsonWebKey} publicJwk - A public key object in JWK format to be encoded and signed to X.509
 * @param {JsonWebKey} privateJwk - A private key object in JWK format by which would be used to sign the public key.
 * @param {AsnFormat} [format='pem'] - Output format of X.509 certificate, 'pem' or 'der'
 * @param {X509EncodingOptions} [options={}] - X.509 encoding options
 * @return {Promise<DER|PEM>} - Generated X.509 public key certificate in intended format.
 * @throws {Error} - Throws if InvalidFormatSpecification or UnsupportedKeyType.
 */
export declare const fromJwk: (publicJwk: JsonWebKey, privateJwk: JsonWebKey, format?: AsnFormat, options?: X509EncodingOptions) => Promise<DER | PEM>;
/**
 * Convert X.509 certificate to a JWK object.
 * @param {PEM|DER} certX509 - X.509 public key certificate in DER or PEM format.
 * @param {AsnFormat} format - 'der' or 'pem'
 * @return {Promise<JsonWebKey>} - Extracted key object in JWK format.
 * @throws {Error} - Throws if InvalidFormatSpecification.
 */
export declare const toJwk: (certX509: PEM | DER, format?: AsnFormat) => Promise<JsonWebKey>;
/**
 * Parse X.509 certificate and return DER-encoded TBSCertificate and DER encoded signature
 * @param {DER|PEM} certX509 - X.509 public key certificate in DER or PEM format.
 * @param {AsnFormat} format - 'der' or 'pem'
 * @return {{tbsCertificate: Uint8Array, signatureValue: Uint8Array, signatureAlgorithm: String}} - Parsed object.
 * @throws {Error} - Throws if UnsupportedSignatureAlgorithm or InvalidFormatSpecification.
 */
export declare const parse: (certX509: PEM | DER, format?: AsnFormat) => {
    tbsCertificate: Uint8Array;
    signatureValue: Uint8Array;
    signatureAlgorithm: {
        algorithm: SignatureType;
        parameters?: any;
    };
};
/**
 * Parse X.509 certificate and return parsed info
 * @param {DER|PEM} certX509 - X.509 public key certificate in DER or PEM format.
 * @param {AsnFormat} format - 'der' or 'pem'
 * @return {{tbsCertificate: Uint8Array, signatureValue: Uint8Array, signatureAlgorithm: String}} - Parsed object.
 * @throws {Error} - Throws if UnsupportedSignatureAlgorithm or InvalidFormatSpecification.
 */
export declare const info: (certX509: PEM | DER, format?: AsnFormat) => any;
