const test = require('ava')

const { star } = require('./index')

const geom2 = require('../geometries/geom2')

const comparePoints = require('../../test/helpers/comparePoints')

test('star (defaults)', (t) => {
  const geometry = star()
  const pts = geom2.toPoints(geometry)
  const exp = [
    [1, 0],
    [0.30901699437494745, 0.2245139882897927],
    [0.30901699437494745, 0.9510565162951535],
    [-0.11803398874989482, 0.36327126400268045],
    [-0.8090169943749473, 0.5877852522924732],
    [-0.38196601125010515, 4.6777345306052316e-17],
    [-0.8090169943749475, -0.587785252292473],
    [-0.1180339887498949, -0.36327126400268045],
    [0.30901699437494723, -0.9510565162951536],
    [0.3090169943749474, -0.22451398828979277]
  ]

  t.notThrows(() => geom2.validate(geometry))
  t.deepEqual(pts.length, 10)
  t.true(comparePoints(pts, exp))
})

test('star (options)', (t) => {
  // test center
  let geometry = star({ outerRadius: 5, center: [5, 5] })
  let pts = geom2.toPoints(geometry)
  let exp = [
    [10, 5],
    [6.545084971874737, 6.122569941448964],
    [6.545084971874737, 9.755282581475768],
    [4.4098300562505255, 6.816356320013402],
    [0.9549150281252636, 7.938926261462367],
    [3.0901699437494745, 5],
    [0.9549150281252627, 2.061073738537635],
    [4.4098300562505255, 3.1836436799865977],
    [6.545084971874736, 0.2447174185242318],
    [6.545084971874736, 3.8774300585510364]
  ]

  t.notThrows(() => geom2.validate(geometry))
  t.deepEqual(pts.length, 10)
  t.true(comparePoints(pts, exp))

  // test vertices
  geometry = star({ outerRadius: 5, vertices: 8 })
  pts = geom2.toPoints(geometry)
  exp = [
    [5, 0],
    [3.5355339059327378, 1.4644660940672625],
    [3.5355339059327378, 3.5355339059327373],
    [1.4644660940672627, 3.5355339059327378],
    [3.061616997868383e-16, 5],
    [-1.4644660940672622, 3.5355339059327378],
    [-3.5355339059327373, 3.5355339059327378],
    [-3.5355339059327378, 1.464466094067263],
    [-5, 6.123233995736766e-16],
    [-3.535533905932738, -1.464466094067262],
    [-3.5355339059327386, -3.5355339059327373],
    [-1.4644660940672647, -3.535533905932737],
    [-9.184850993605148e-16, -5],
    [1.4644660940672634, -3.5355339059327373],
    [3.535533905932737, -3.5355339059327386],
    [3.535533905932737, -1.464466094067265]
  ]

  t.notThrows(() => geom2.validate(geometry))
  t.deepEqual(pts.length, 16)
  t.true(comparePoints(pts, exp))

  // test density
  geometry = star({ outerRadius: 5, vertices: 8, density: 3 })
  pts = geom2.toPoints(geometry)
  exp = [
    [5, 0],
    [2.5, 1.0355339059327378],
    [3.5355339059327378, 3.5355339059327373],
    [1.0355339059327378, 2.5],
    [3.061616997868383e-16, 5],
    [-1.0355339059327375, 2.5],
    [-3.5355339059327373, 3.5355339059327378],
    [-2.5, 1.035533905932738],
    [-5, 6.123233995736766e-16],
    [-2.5000000000000004, -1.0355339059327373],
    [-3.5355339059327386, -3.5355339059327373],
    [-1.035533905932739, -2.4999999999999996],
    [-9.184850993605148e-16, -5],
    [1.0355339059327382, -2.4999999999999996],
    [3.535533905932737, -3.5355339059327386],
    [2.4999999999999996, -1.0355339059327393]
  ]

  t.notThrows(() => geom2.validate(geometry))
  t.deepEqual(pts.length, 16)
  t.true(comparePoints(pts, exp))

  // test innerRadius
  geometry = star({ outerRadius: 5, vertices: 8, innerRadius: 1 })
  pts = geom2.toPoints(geometry)
  exp = [
    [5, 0],
    [0.9238795325112867, 0.3826834323650898],
    [3.5355339059327378, 3.5355339059327373],
    [0.38268343236508984, 0.9238795325112867],
    [3.061616997868383e-16, 5],
    [-0.3826834323650897, 0.9238795325112867],
    [-3.5355339059327373, 3.5355339059327378],
    [-0.9238795325112867, 0.3826834323650899],
    [-5, 6.123233995736766e-16],
    [-0.9238795325112868, -0.38268343236508967],
    [-3.5355339059327386, -3.5355339059327373],
    [-0.38268343236509034, -0.9238795325112865],
    [-9.184850993605148e-16, -5],
    [0.38268343236509, -0.9238795325112866],
    [3.535533905932737, -3.5355339059327386],
    [0.9238795325112865, -0.3826834323650904]
  ]

  t.notThrows(() => geom2.validate(geometry))
  t.deepEqual(pts.length, 16)
  t.true(comparePoints(pts, exp))

  // test start angle
  geometry = star({ outerRadius: 5, startAngle: (360 - 45) * 0.017453292519943295 })
  pts = geom2.toPoints(geometry)
  exp = [
    [3.535533905932737, -3.5355339059327386],
    [1.8863168790768001, -0.2987632431673055],
    [4.45503262094184, 2.269952498697733],
    [0.8670447016547834, 1.701671040210256],
    [-0.7821723252011483, 4.938441702975689],
    [-1.3504537836886306, 1.350453783688634],
    [-4.9384417029756875, 0.7821723252011604],
    [-1.701671040210257, -0.8670447016547808],
    [-2.26995249869774, -4.455032620941836],
    [0.2987632431673025, -1.8863168790768006]
  ]

  t.notThrows(() => geom2.validate(geometry))
  t.deepEqual(pts.length, 10)
  t.true(comparePoints(pts, exp))
})
