# JSON Patch utils
### Helpers for client/server applications using and implementing JSON Patch [RFC 6902]

[![Build Status](https://travis-ci.org/MarketsWorld/json-patch-utils.svg?branch=master)](https://travis-ci.org/MarketsWorld/json-patch-utils)
[![npm version](https://badge.fury.io/js/ramda-loader.svg)](https://badge.fury.io/js/json-patch-utils)
[![dependencies](https://david-dm.org/dumconstantin/ramda-loader.svg)](https://david-dm.org/marketsworld/json-patch-utils)

#### [See documentation](http://marketsworld.github.io/json-patch-utils/docs/)

## Install:

```bash
$ npm install --save-dev json-patch-utils
```

## Usage:

#### createPatch
Creates a JSON Patches given an operator, path and a value
```javascript

import { createPatch } from 'json-patch-utils'

createPatch('add', '/foo', 123) // { op: 'add', path: '/foo', value: 123 }
createPatch('move', 'foo', 'bar') // [Error: The path "foo" did not match the encoded path regexp: /^(\/[a-z0-9~\\\-%^|"\ ]*)*$/gi,The path "bar" did not match the encoded path regexp: /^(\/[a-z0-9~\\\-%^|"\ ]*)*$/gi]

```

#### isPatch
Tests if a given object is a valid JSON Patch object according to https://tools.ietf.org/html/rfc6902
```javascript

import { isPatch } from 'json-patch-utils'

let validPatch = { op: 'add', path: '/foo/bar', value: 'baz' }
let invalidPatch = { op: 'foo', path: 'bar' }

if(isPatch(validPatch) === true) {
  // Patch can be safely applied
}

isPatch(invalidPatch) // [
//    'The path "bar" did not match the encoded path regexp: /^(\\/[a-z0-9~\\\\\\-%^|"\\ ]*)*$/gi',
//    'The operation name is not among the valid names add, replace, test, remove, move, copy'
// ]

```

#### isPath
Tests if a given string is a valid JSON Pointer according to https://tools.ietf.org/html/rfc6901
```javascript

import { isPath } from 'json-patch-utils'

isPath('/foo/bar') // true
isPath('foo') // The path "foo" did not match the encoded path regexp: /^(\/[a-z0-9~\\\-%^|"\ ]*)*$/gi

```

#### isOperation
Tests if a given string is a valid JSON Patch operation name
```javascript

import { isOperation } from 'json-patch-utils'

isOperation('add') // true
isOperation('foo') // The operation name is not among the valid names add, replace, test, remove, move, copy

```

#### isValueForOperation
Tests if a given value corresponds to the given operation
```javascript

import { isValueForOperation } from 'json-patch-utils'

isValueForOperation('move', '/foo') // true
isValueForOperation('add', undefined) // The value or from path provided must be different than undefined
isValueForOperation('copy', 'foo') // The path "foo" did not match the encoded path regexp: /^(\/[a-z0-9~\\\-%^|"\ ]*)*$/gi

```

#### decodePath
Decodes a path according to https://tools.ietf.org/html/rfc6901 by replacing special symbols
```javascript

import { decodePath } from 'json-patch-utils'

decodePath('/foo/bar') // '/foo/bar'
decodePath('/foo~0bar~1') // '/foo/bar~'

```

#### listPath
Convert a path into a list of strings. Useful when traversing a path on a JSON Document
```javascript

import { listPath } from 'json-patch-utils'

listPath('/foo/bar/') // ['foo', 'bar']
listPath('/foo~0bar~1') // ['foo', 'bar~']

```

## Apply patch functions for various state/data structure libraries

#### [BaobabJS](https://github.com/Yomguithereal/baobab)
```javascript

import Baobab from 'baobab'
import { applyBaobabJsPatch, createPatch } from 'json-patch-utils'

let tree = new Baobab({
  foo: {
    bar: 123
  }
})
let patch = createPatch('replace', '/foo/bar', 1000)

applyBaobabJsPatch(tree, patch)

tree.select('foo', 'bar').get() // 1000

```

## Credits
https://github.com/Starcounter-Jack/JSON-Patch - for the tests battery in test/tests.json and test/spec_tests.json

## Contributing:

Feel free to open issues to propose stuff and participate. Pull requests are also welcome.

## Licence:

[MIT](http://en.wikipedia.org/wiki/MIT_License)

