'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _ramda = require('ramda');

var _isPatch = require('./isPatch');

var _isPatch2 = _interopRequireDefault(_isPatch);

var _listPath = require('./listPath');

var _listPath2 = _interopRequireDefault(_listPath);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var isLast = (0, _ramda.compose)((0, _ramda.equals)('-'), _ramda.last);

var applyCursorPatch = (0, _ramda.curry)(function (cursor, patch) {
  var updatePath = (0, _listPath2.default)(patch.path);
  var updateValue = patch.value;
  var updateType = void 0;

  var isLastEl = isLast(patch.path);

  if (isLastEl) {
    updatePath = (0, _ramda.dropLast)(1, updatePath);
  }

  switch (patch.op) {
    case 'add':
      updateType = isLastEl ? 'push' : 'set';

      cursor.update(updatePath, {
        type: updateType,
        value: updateValue
      });

      break;
    case 'replace':
      if (cursor.exists(updatePath)) {
        updateValue = isLastEl ? [-1, 1, updateValue] : updateValue;
        updateType = isLastEl ? 'splice' : 'set';

        cursor.update(updatePath, {
          type: updateType,
          value: updateValue
        });
      } else {
        throw 'Applying patch ' + patch + ' failed. The path ' + updatePath + ' does not exists.';
      }
      break;
    case 'test':
      if (cursor.exists(updatePath)) {
        var currentValue = cursor.select(updatePath).get();
        if (!(0, _ramda.equals)(currentValue, updateValue)) {
          throw 'Applying patch ' + patch + ' failed. The values ' + currentValue + ' and ' + updateValue + ' are not equal';
        }
      } else {
        throw 'Applying patch ' + patch + ' failed. The path ' + updatePath + ' does not exists.';
      }
      break;
    case 'move':
      console.error('Applying patch ' + patch + ' failed. Action "move" is not implemented');
      break;
    case 'copy':
      console.error('Applying patch ' + patch + ' failed. Action "copy" is not implemented');
      break;
    case 'merge':
      cursor.update(updatePath, {
        type: 'deepMerge',
        value: updateValue
      });
      break;

    case 'remove':
      cursor.update(updatePath, {
        type: 'unset'
      });

      break;

  }

  return cursor;
});

/**
 * Applies a JSON Patch to the given data cursor
 *
 * ** Sig: DataCursor -> Patch **
 *
 * @function
 * @implements curry
 * @param {object} dataCursor - The data cursor that will receive the patch
 * @param {object} patch - The JSON patch that implements https://tools.ietf.org/html/rfc6902
 * @return {object} cursor - The received cursor is returned after the applied patches
 * @example
 *
 * import { createPatch, applyBaobabJsPatch } from 'json-patch-utils'
 * import Baobab from 'baobab'
 *
 * let tree = new Baobab({
 *  foo: {
 *    bar: 123
 *  }
 * })
 * let patch = createPatch('add', '/foo/bar', 100)
 * applyPatch(tree, patch)
 * tree.get() // { foo: { bar: 100 } }
 *
 * @todo add atomic transactions
 */
var applyBaobabJsPatch = (0, _ramda.curry)(function (cursor, xs) {

  xs = (0, _ramda.isArrayLike)(xs) ? xs : [xs];

  xs.forEach(function (x) {
    if ((0, _isPatch2.default)(x)) {
      applyCursorPatch(cursor, x);
    } else {
      throw 'Patch not valid';
    }
  });

  return cursor;
});

exports.default = applyBaobabJsPatch;
//# sourceMappingURL=applyBaobabJsPatch.js.map
