import 'cross-fetch/polyfill';
import { IDocumentLoader } from "./IDocumentLoader";
import { IJsonLdContextNormalizedRaw, JsonLdContext } from "./JsonLdContext";
import { JsonLdContextNormalized } from "./JsonLdContextNormalized";
/**
 * Parses JSON-LD contexts.
 */
export declare class ContextParser {
    static readonly DEFAULT_PROCESSING_MODE: number;
    private readonly documentLoader;
    private readonly documentCache;
    private readonly validateContext;
    private readonly expandContentTypeToBase;
    private readonly remoteContextsDepthLimit;
    private readonly redirectSchemaOrgHttps;
    constructor(options?: IContextParserOptions);
    /**
     * Validate the given @language value.
     * An error will be thrown if it is invalid.
     * @param value An @language value.
     * @param {boolean} strictRange If the string value should be strictly checked against a regex.
     * @param {string} errorCode The error code to emit on errors.
     * @return {boolean} If validation passed.
     *                   Can only be false if strictRange is false and the string value did not pass the regex.
     */
    static validateLanguage(value: any, strictRange: boolean, errorCode: string): boolean;
    /**
     * Validate the given @direction value.
     * An error will be thrown if it is invalid.
     * @param value An @direction value.
     * @param {boolean} strictValues If the string value should be strictly checked against a regex.
     * @return {boolean} If validation passed.
     *                   Can only be false if strictRange is false and the string value did not pass the regex.
     */
    static validateDirection(value: any, strictValues: boolean): boolean;
    /**
     * Add an @id term for all @reverse terms.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @return {IJsonLdContextNormalizedRaw} The mutated input context.
     */
    idifyReverseTerms(context: IJsonLdContextNormalizedRaw): IJsonLdContextNormalizedRaw;
    /**
     * Expand all prefixed terms in the given context.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {boolean} expandContentTypeToBase If @type inside the context may be expanded
     *                                          via @base if @vocab is set to null.
     */
    expandPrefixedTerms(context: JsonLdContextNormalized, expandContentTypeToBase: boolean): void;
    /**
     * Normalize the @language entries in the given context to lowercase.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {IParseOptions} parseOptions The parsing options.
     */
    normalize(context: IJsonLdContextNormalizedRaw, { processingMode, normalizeLanguageTags }: IParseOptions): void;
    /**
     * Convert all @container strings and array values to hash-based values.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     */
    containersToHash(context: IJsonLdContextNormalizedRaw): void;
    /**
     * Normalize and apply context-levevl @protected terms onto each term separately.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {number} processingMode The processing mode.
     */
    applyScopedProtected(context: IJsonLdContextNormalizedRaw, { processingMode }: IParseOptions): void;
    /**
     * Check if the given context inheritance does not contain any overrides of protected terms.
     * @param {IJsonLdContextNormalizedRaw} contextBefore The context that may contain some protected terms.
     * @param {IJsonLdContextNormalizedRaw} contextAfter A new context that is being applied on the first one.
     * @param {IExpandOptions} expandOptions Options that are needed for any expansions during this validation.
     */
    validateKeywordRedefinitions(contextBefore: IJsonLdContextNormalizedRaw, contextAfter: IJsonLdContextNormalizedRaw, expandOptions: IExpandOptions): void;
    /**
     * Validate the entries of the given context.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {IParseOptions} options The parse options.
     */
    validate(context: IJsonLdContextNormalizedRaw, { processingMode }: IParseOptions): void;
    /**
     * Apply the @base context entry to the given context under certain circumstances.
     * @param context A context.
     * @param options Parsing options.
     * @param inheritFromParent If the @base value from the parent context can be inherited.
     * @return The given context.
     */
    applyBaseEntry(context: IJsonLdContextNormalizedRaw, options: IParseOptions, inheritFromParent: boolean): IJsonLdContextNormalizedRaw;
    /**
     * Resolve relative context IRIs, or return full IRIs as-is.
     * @param {string} contextIri A context IRI.
     * @param {string} baseIRI A base IRI.
     * @return {string} The normalized context IRI.
     */
    normalizeContextIri(contextIri: string, baseIRI?: string): string;
    /**
     * Parse scoped contexts in the given context.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {IParseOptions} options Parsing options.
     * @return {IJsonLdContextNormalizedRaw} The mutated input context.
     */
    parseInnerContexts(context: IJsonLdContextNormalizedRaw, options: IParseOptions): Promise<IJsonLdContextNormalizedRaw>;
    /**
     * Parse a JSON-LD context in any form.
     * @param {JsonLdContext} context A context, URL to a context, or an array of contexts/URLs.
     * @param {IParseOptions} options Optional parsing options.
     * @return {Promise<JsonLdContextNormalized>} A promise resolving to the context.
     */
    parse(context: JsonLdContext, options?: IParseOptions): Promise<JsonLdContextNormalized>;
    /**
     * Fetch the given URL as a raw JSON-LD context.
     * @param url An URL.
     * @return A promise resolving to a raw JSON-LD context.
     */
    load(url: string): Promise<JsonLdContext>;
    /**
     * Override the given context that may be loaded.
     *
     * This will check whether or not the url is recursively being loaded.
     * @param url An URL.
     * @param options Parsing options.
     * @return An overridden context, or null.
     *         Optionally an error can be thrown if a cyclic context is detected.
     */
    getOverriddenLoad(url: string, options: IParseOptions): IJsonLdContextNormalizedRaw | null;
    /**
     * Load an @import'ed context.
     * @param importContextIri The full URI of an @import value.
     */
    loadImportContext(importContextIri: string): Promise<IJsonLdContextNormalizedRaw>;
}
export interface IContextParserOptions {
    /**
     * An optional loader that should be used for fetching external JSON-LD contexts.
     */
    documentLoader?: IDocumentLoader;
    /**
     * By default, JSON-LD contexts will be validated.
     * This can be disabled by setting this option to true.
     * This will achieve slightly better performance for large contexts,
     * and may be useful if contexts are known to be valid.
     */
    skipValidation?: boolean;
    /**
     * If @type inside the context may be expanded via @base is @vocab is set to null.
     */
    expandContentTypeToBase?: boolean;
    /**
     * The maximum number of remote contexts that can be fetched recursively.
     *
     * Defaults to 32.
     */
    remoteContextsDepthLimit?: number;
    /**
     * If http-based schema.org contexts should internally be redirected to https.
     * WARNING: this option is a temporary workaround for https://github.com/schemaorg/schemaorg/issues/2578#issuecomment-652324465
     * and will be removed once that issue is fixed.
     * Defaults to true.
     */
    redirectSchemaOrgHttps?: boolean;
}
export interface IParseOptions {
    /**
     * An optional fallback base IRI to set.
     */
    baseIRI?: string;
    /**
     * The parent context.
     */
    parentContext?: IJsonLdContextNormalizedRaw;
    /**
     * If the parsing context is an external context.
     */
    external?: boolean;
    /**
     * The default JSON-LD version that the context should be parsed with.
     */
    processingMode?: number;
    /**
     * If language tags should be normalized to lowercase.
     * This is always true for JSON-LD 1.0,
     * but false by default for all following versions.
     */
    normalizeLanguageTags?: boolean;
    /**
     * If checks for validating term protection should be skipped.
     */
    ignoreProtection?: boolean;
    /**
     * If the context should only be parsed and validated,
     * without performing normalizations and other modifications.
     *
     * If true, this *will* dereference external contexts.
     *
     * This option is used internally when handling type-scoped and property-scoped contexts.
     */
    minimalProcessing?: boolean;
    /**
     * If true, a remote context that will be looked up,
     * and is already contained in `remoteContexts`,
     * will not emit an error but will produce an empty context.
     */
    ignoreRemoteScopedContexts?: boolean;
    /**
     * A hash containing all remote contexts that have been looked up before.
     *
     * This is used to avoid stack overflows on cyclic context references.
     */
    remoteContexts?: {
        [url: string]: boolean;
    };
    /**
     * If further processing of scoped contexts should be skipped.
     *
     * This is done to avoid combinatorial explosions when handling a scoped context if there are many scoped contexts.
     */
    ignoreScopedContexts?: boolean;
}
export interface IExpandOptions {
    /**
     * If compact IRI prefixes can end with any kind of character in simple term definitions,
     * instead of only the default gen-delim characters (:,/,?,#,[,],@).
     */
    allowPrefixNonGenDelims: boolean;
    /**
     * If compact IRI prefixes ending with a non-gen-delim character
     * can be forced as a prefix using @prefix: true.
     */
    allowPrefixForcing: boolean;
    /**
     * If @vocab values are allowed contain IRIs relative to @base.
     */
    allowVocabRelativeToBase: boolean;
}
export declare const defaultExpandOptions: IExpandOptions;
