"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultExpandOptions = exports.ContextParser = void 0;
require("cross-fetch/polyfill");
const relative_to_absolute_iri_1 = require("relative-to-absolute-iri");
const ErrorCoded_1 = require("./ErrorCoded");
const FetchDocumentLoader_1 = require("./FetchDocumentLoader");
const JsonLdContextNormalized_1 = require("./JsonLdContextNormalized");
const Util_1 = require("./Util");
// tslint:disable-next-line:no-var-requires
const canonicalizeJson = require('canonicalize');
/**
 * Parses JSON-LD contexts.
 */
class ContextParser {
    constructor(options) {
        options = options || {};
        this.documentLoader = options.documentLoader || new FetchDocumentLoader_1.FetchDocumentLoader();
        this.documentCache = {};
        this.validateContext = !options.skipValidation;
        this.expandContentTypeToBase = !!options.expandContentTypeToBase;
        this.remoteContextsDepthLimit = options.remoteContextsDepthLimit || 32;
        this.redirectSchemaOrgHttps = 'redirectSchemaOrgHttps' in options ? !!options.redirectSchemaOrgHttps : true;
    }
    /**
     * Validate the given @language value.
     * An error will be thrown if it is invalid.
     * @param value An @language value.
     * @param {boolean} strictRange If the string value should be strictly checked against a regex.
     * @param {string} errorCode The error code to emit on errors.
     * @return {boolean} If validation passed.
     *                   Can only be false if strictRange is false and the string value did not pass the regex.
     */
    static validateLanguage(value, strictRange, errorCode) {
        if (typeof value !== 'string') {
            throw new ErrorCoded_1.ErrorCoded(`The value of an '@language' must be a string, got '${JSON.stringify(value)}'`, errorCode);
        }
        if (!Util_1.Util.REGEX_LANGUAGE_TAG.test(value)) {
            if (strictRange) {
                throw new ErrorCoded_1.ErrorCoded(`The value of an '@language' must be a valid language tag, got '${JSON.stringify(value)}'`, errorCode);
            }
            else {
                return false;
            }
        }
        return true;
    }
    /**
     * Validate the given @direction value.
     * An error will be thrown if it is invalid.
     * @param value An @direction value.
     * @param {boolean} strictValues If the string value should be strictly checked against a regex.
     * @return {boolean} If validation passed.
     *                   Can only be false if strictRange is false and the string value did not pass the regex.
     */
    static validateDirection(value, strictValues) {
        if (typeof value !== 'string') {
            throw new ErrorCoded_1.ErrorCoded(`The value of an '@direction' must be a string, got '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_BASE_DIRECTION);
        }
        if (!Util_1.Util.REGEX_DIRECTION_TAG.test(value)) {
            if (strictValues) {
                throw new ErrorCoded_1.ErrorCoded(`The value of an '@direction' must be 'ltr' or 'rtl', got '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_BASE_DIRECTION);
            }
            else {
                return false;
            }
        }
        return true;
    }
    /**
     * Add an @id term for all @reverse terms.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @return {IJsonLdContextNormalizedRaw} The mutated input context.
     */
    idifyReverseTerms(context) {
        for (const key of Object.keys(context)) {
            const value = context[key];
            if (value && typeof value === 'object') {
                if (value['@reverse'] && !value['@id']) {
                    if (typeof value['@reverse'] !== 'string' || Util_1.Util.isValidKeyword(value['@reverse'])) {
                        throw new ErrorCoded_1.ErrorCoded(`Invalid @reverse value, must be absolute IRI or blank node: '${value['@reverse']}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
                    }
                    value['@id'] = value['@reverse'];
                    if (Util_1.Util.isPotentialKeyword(value['@reverse'])) {
                        delete value['@reverse'];
                    }
                    else {
                        value['@reverse'] = true;
                    }
                }
            }
        }
        return context;
    }
    /**
     * Expand all prefixed terms in the given context.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {boolean} expandContentTypeToBase If @type inside the context may be expanded
     *                                          via @base if @vocab is set to null.
     */
    expandPrefixedTerms(context, expandContentTypeToBase) {
        const contextRaw = context.getContextRaw();
        for (const key of Object.keys(contextRaw)) {
            // Only expand allowed keys
            if (Util_1.Util.EXPAND_KEYS_BLACKLIST.indexOf(key) < 0 && !Util_1.Util.isReservedInternalKeyword(key)) {
                // Error if we try to alias a keyword to something else.
                const keyValue = contextRaw[key];
                if (Util_1.Util.isPotentialKeyword(key) && Util_1.Util.ALIAS_DOMAIN_BLACKLIST.indexOf(key) >= 0) {
                    if (key !== '@type' || typeof contextRaw[key] === 'object'
                        && !(contextRaw[key]['@protected'] || contextRaw[key]['@container'] === '@set')) {
                        throw new ErrorCoded_1.ErrorCoded(`Keywords can not be aliased to something else.
Tried mapping ${key} to ${JSON.stringify(keyValue)}`, ErrorCoded_1.ERROR_CODES.KEYWORD_REDEFINITION);
                    }
                }
                // Error if we try to alias to an illegal keyword
                if (Util_1.Util.ALIAS_RANGE_BLACKLIST.indexOf(Util_1.Util.getContextValueId(keyValue)) >= 0) {
                    throw new ErrorCoded_1.ErrorCoded(`Aliasing to certain keywords is not allowed.
Tried mapping ${key} to ${JSON.stringify(keyValue)}`, ErrorCoded_1.ERROR_CODES.INVALID_KEYWORD_ALIAS);
                }
                // Error if this term was marked as prefix as well
                if (keyValue && Util_1.Util.isPotentialKeyword(Util_1.Util.getContextValueId(keyValue))
                    && keyValue['@prefix'] === true) {
                    throw new ErrorCoded_1.ErrorCoded(`Tried to use keyword aliases as prefix: '${key}': '${JSON.stringify(keyValue)}'`, ErrorCoded_1.ERROR_CODES.INVALID_TERM_DEFINITION);
                }
                // Loop because prefixes might be nested
                while (Util_1.Util.isPrefixValue(contextRaw[key])) {
                    const value = contextRaw[key];
                    let changed = false;
                    if (typeof value === 'string') {
                        contextRaw[key] = context.expandTerm(value, true);
                        changed = changed || value !== contextRaw[key];
                    }
                    else {
                        const id = value['@id'];
                        const type = value['@type'];
                        // If @id is missing, don't allow @id to be added if @prefix: true and key not being a valid IRI.
                        const canAddIdEntry = !('@prefix' in value) || Util_1.Util.isValidIri(key);
                        if ('@id' in value) {
                            // Use @id value for expansion
                            if (id !== undefined && id !== null && typeof id === 'string') {
                                contextRaw[key]['@id'] = context.expandTerm(id, true);
                                changed = changed || id !== contextRaw[key]['@id'];
                            }
                        }
                        else if (!Util_1.Util.isPotentialKeyword(key) && canAddIdEntry) {
                            // Add an explicit @id value based on the expanded key value
                            const newId = context.expandTerm(key, true);
                            if (newId !== key) {
                                // Don't set @id if expansion failed
                                contextRaw[key]['@id'] = newId;
                                changed = true;
                            }
                        }
                        if (type && typeof type === 'string' && type !== '@vocab'
                            && (!value['@container'] || !value['@container']['@type'])
                            && canAddIdEntry) {
                            // First check @vocab, then fallback to @base
                            contextRaw[key]['@type'] = context.expandTerm(type, true);
                            if (expandContentTypeToBase && type === contextRaw[key]['@type']) {
                                contextRaw[key]['@type'] = context.expandTerm(type, false);
                            }
                            changed = changed || type !== contextRaw[key]['@type'];
                        }
                    }
                    if (!changed) {
                        break;
                    }
                }
            }
        }
    }
    /**
     * Normalize the @language entries in the given context to lowercase.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {IParseOptions} parseOptions The parsing options.
     */
    normalize(context, { processingMode, normalizeLanguageTags }) {
        // Lowercase language keys in 1.0
        if (normalizeLanguageTags || processingMode === 1.0) {
            for (const key of Object.keys(context)) {
                if (key === '@language' && typeof context[key] === 'string') {
                    context[key] = context[key].toLowerCase();
                }
                else {
                    const value = context[key];
                    if (value && typeof value === 'object') {
                        if (typeof value['@language'] === 'string') {
                            value['@language'] = value['@language'].toLowerCase();
                        }
                    }
                }
            }
        }
    }
    /**
     * Convert all @container strings and array values to hash-based values.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     */
    containersToHash(context) {
        for (const key of Object.keys(context)) {
            const value = context[key];
            if (value && typeof value === 'object') {
                if (typeof value['@container'] === 'string') {
                    value['@container'] = { [value['@container']]: true };
                }
                else if (Array.isArray(value['@container'])) {
                    const newValue = {};
                    for (const containerValue of value['@container']) {
                        newValue[containerValue] = true;
                    }
                    value['@container'] = newValue;
                }
            }
        }
    }
    /**
     * Normalize and apply context-levevl @protected terms onto each term separately.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {number} processingMode The processing mode.
     */
    applyScopedProtected(context, { processingMode }) {
        if (processingMode && processingMode >= 1.1) {
            if (context['@protected']) {
                for (const key of Object.keys(context)) {
                    if (Util_1.Util.isReservedInternalKeyword(key)) {
                        continue;
                    }
                    if (!Util_1.Util.isPotentialKeyword(key) && !Util_1.Util.isTermProtected(context, key)) {
                        const value = context[key];
                        if (value && typeof value === 'object') {
                            if (!('@protected' in context[key])) {
                                // Mark terms with object values as protected if they don't have an @protected: false annotation
                                context[key]['@protected'] = true;
                            }
                        }
                        else {
                            // Convert string-based term values to object-based values with @protected: true
                            context[key] = {
                                '@id': value,
                                '@protected': true,
                            };
                        }
                    }
                }
                delete context['@protected'];
            }
        }
    }
    /**
     * Check if the given context inheritance does not contain any overrides of protected terms.
     * @param {IJsonLdContextNormalizedRaw} contextBefore The context that may contain some protected terms.
     * @param {IJsonLdContextNormalizedRaw} contextAfter A new context that is being applied on the first one.
     * @param {IExpandOptions} expandOptions Options that are needed for any expansions during this validation.
     */
    validateKeywordRedefinitions(contextBefore, contextAfter, expandOptions) {
        for (const key of Object.keys(contextAfter)) {
            if (Util_1.Util.isTermProtected(contextBefore, key)) {
                // The entry in the context before will always be in object-mode
                // If the new entry is in string-mode, convert it to object-mode
                // before checking if it is identical.
                if (typeof contextAfter[key] === 'string') {
                    const isPrefix = Util_1.Util.isSimpleTermDefinitionPrefix(contextAfter[key], expandOptions);
                    contextAfter[key] = { '@id': contextAfter[key] };
                    // If the simple term def was a prefix, explicitly mark the term as a prefix in the expanded term definition,
                    // because otherwise we loose this information due to JSON-LD interpreting prefixes differently
                    // in simple vs expanded term definitions.
                    if (isPrefix) {
                        contextAfter[key]['@prefix'] = true;
                        contextBefore[key]['@prefix'] = true; // Also on before, to make sure the next step still considers them ==
                    }
                }
                // Convert term values to strings for each comparison
                const valueBefore = canonicalizeJson(contextBefore[key]);
                // We modify this deliberately,
                // as we need it for the value comparison (they must be identical modulo '@protected')),
                // and for the fact that this new value will override the first one.
                contextAfter[key]['@protected'] = true;
                const valueAfter = canonicalizeJson(contextAfter[key]);
                // Error if they are not identical
                if (valueBefore !== valueAfter) {
                    throw new ErrorCoded_1.ErrorCoded(`Attempted to override the protected keyword ${key} from ${JSON.stringify(Util_1.Util.getContextValueId(contextBefore[key]))} to ${JSON.stringify(Util_1.Util.getContextValueId(contextAfter[key]))}`, ErrorCoded_1.ERROR_CODES.PROTECTED_TERM_REDEFINITION);
                }
            }
        }
    }
    /**
     * Validate the entries of the given context.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {IParseOptions} options The parse options.
     */
    validate(context, { processingMode }) {
        for (const key of Object.keys(context)) {
            // Ignore reserved internal keywords.
            if (Util_1.Util.isReservedInternalKeyword(key)) {
                continue;
            }
            // Do not allow empty term
            if (key === '') {
                throw new ErrorCoded_1.ErrorCoded(`The empty term is not allowed, got: '${key}': '${JSON.stringify(context[key])}'`, ErrorCoded_1.ERROR_CODES.INVALID_TERM_DEFINITION);
            }
            const value = context[key];
            const valueType = typeof value;
            // First check if the key is a keyword
            if (Util_1.Util.isPotentialKeyword(key)) {
                switch (key.substr(1)) {
                    case 'vocab':
                        if (value !== null && valueType !== 'string') {
                            throw new ErrorCoded_1.ErrorCoded(`Found an invalid @vocab IRI: ${value}`, ErrorCoded_1.ERROR_CODES.INVALID_VOCAB_MAPPING);
                        }
                        break;
                    case 'base':
                        if (value !== null && valueType !== 'string') {
                            throw new ErrorCoded_1.ErrorCoded(`Found an invalid @base IRI: ${context[key]}`, ErrorCoded_1.ERROR_CODES.INVALID_BASE_IRI);
                        }
                        break;
                    case 'language':
                        if (value !== null) {
                            ContextParser.validateLanguage(value, true, ErrorCoded_1.ERROR_CODES.INVALID_DEFAULT_LANGUAGE);
                        }
                        break;
                    case 'version':
                        if (value !== null && valueType !== 'number') {
                            throw new ErrorCoded_1.ErrorCoded(`Found an invalid @version number: ${value}`, ErrorCoded_1.ERROR_CODES.INVALID_VERSION_VALUE);
                        }
                        break;
                    case 'direction':
                        if (value !== null) {
                            ContextParser.validateDirection(value, true);
                        }
                        break;
                    case 'propagate':
                        if (processingMode === 1.0) {
                            throw new ErrorCoded_1.ErrorCoded(`Found an illegal @propagate keyword: ${value}`, ErrorCoded_1.ERROR_CODES.INVALID_CONTEXT_ENTRY);
                        }
                        if (value !== null && valueType !== 'boolean') {
                            throw new ErrorCoded_1.ErrorCoded(`Found an invalid @propagate value: ${value}`, ErrorCoded_1.ERROR_CODES.INVALID_PROPAGATE_VALUE);
                        }
                        break;
                }
                // Don't allow keywords to be overridden
                if (Util_1.Util.isValidKeyword(key) && Util_1.Util.isValidKeyword(Util_1.Util.getContextValueId(value))) {
                    throw new ErrorCoded_1.ErrorCoded(`Illegal keyword alias in term value, found: '${key}': '${Util_1.Util
                        .getContextValueId(value)}'`, ErrorCoded_1.ERROR_CODES.KEYWORD_REDEFINITION);
                }
                continue;
            }
            // Otherwise, consider the key a term
            if (value !== null) {
                switch (valueType) {
                    case 'string':
                        if (Util_1.Util.getPrefix(value, context) === key) {
                            throw new ErrorCoded_1.ErrorCoded(`Detected cyclical IRI mapping in context entry: '${key}': '${JSON
                                .stringify(value)}'`, ErrorCoded_1.ERROR_CODES.CYCLIC_IRI_MAPPING);
                        }
                        if (Util_1.Util.isValidIriWeak(key)) {
                            if (value === '@type') {
                                throw new ErrorCoded_1.ErrorCoded(`IRIs can not be mapped to @type, found: '${key}': '${value}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
                            }
                            else if (Util_1.Util.isValidIri(value) && value !== new JsonLdContextNormalized_1.JsonLdContextNormalized(context).expandTerm(key)) {
                                throw new ErrorCoded_1.ErrorCoded(`IRIs can not be mapped to other IRIs, found: '${key}': '${value}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
                            }
                        }
                        break;
                    case 'object':
                        if (!Util_1.Util.isCompactIri(key) && !('@id' in value)
                            && (value['@type'] === '@id' ? !context['@base'] : !context['@vocab'])) {
                            throw new ErrorCoded_1.ErrorCoded(`Missing @id in context entry: '${key}': '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
                        }
                        for (const objectKey of Object.keys(value)) {
                            const objectValue = value[objectKey];
                            if (!objectValue) {
                                continue;
                            }
                            switch (objectKey) {
                                case '@id':
                                    if (Util_1.Util.isValidKeyword(objectValue)
                                        && objectValue !== '@type' && objectValue !== '@id' && objectValue !== '@graph') {
                                        throw new ErrorCoded_1.ErrorCoded(`Illegal keyword alias in term value, found: '${key}': '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
                                    }
                                    if (Util_1.Util.isValidIriWeak(key)) {
                                        if (objectValue === '@type') {
                                            throw new ErrorCoded_1.ErrorCoded(`IRIs can not be mapped to @type, found: '${key}': '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
                                        }
                                        else if (Util_1.Util.isValidIri(objectValue)
                                            && objectValue !== new JsonLdContextNormalized_1.JsonLdContextNormalized(context).expandTerm(key)) {
                                            throw new ErrorCoded_1.ErrorCoded(`IRIs can not be mapped to other IRIs, found: '${key}': '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
                                        }
                                    }
                                    if (typeof objectValue !== 'string') {
                                        throw new ErrorCoded_1.ErrorCoded(`Detected non-string @id in context entry: '${key}': '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
                                    }
                                    if (Util_1.Util.getPrefix(objectValue, context) === key) {
                                        throw new ErrorCoded_1.ErrorCoded(`Detected cyclical IRI mapping in context entry: '${key}': '${JSON
                                            .stringify(value)}'`, ErrorCoded_1.ERROR_CODES.CYCLIC_IRI_MAPPING);
                                    }
                                    break;
                                case '@type':
                                    if (value['@container'] === '@type' && objectValue !== '@id' && objectValue !== '@vocab') {
                                        throw new ErrorCoded_1.ErrorCoded(`@container: @type only allows @type: @id or @vocab, but got: '${key}': '${objectValue}'`, ErrorCoded_1.ERROR_CODES.INVALID_TYPE_MAPPING);
                                    }
                                    if (typeof objectValue !== 'string') {
                                        throw new ErrorCoded_1.ErrorCoded(`The value of an '@type' must be a string, got '${JSON.stringify(valueType)}'`, ErrorCoded_1.ERROR_CODES.INVALID_TYPE_MAPPING);
                                    }
                                    if (objectValue !== '@id' && objectValue !== '@vocab'
                                        && (processingMode === 1.0 || objectValue !== '@json')
                                        && (processingMode === 1.0 || objectValue !== '@none')
                                        && (objectValue[0] === '_' || !Util_1.Util.isValidIri(objectValue))) {
                                        throw new ErrorCoded_1.ErrorCoded(`A context @type must be an absolute IRI, found: '${key}': '${objectValue}'`, ErrorCoded_1.ERROR_CODES.INVALID_TYPE_MAPPING);
                                    }
                                    break;
                                case '@reverse':
                                    if (typeof objectValue === 'string' && value['@id'] && value['@id'] !== objectValue) {
                                        throw new ErrorCoded_1.ErrorCoded(`Found non-matching @id and @reverse term values in '${key}':\
'${objectValue}' and '${value['@id']}'`, ErrorCoded_1.ERROR_CODES.INVALID_REVERSE_PROPERTY);
                                    }
                                    if ('@nest' in value) {
                                        throw new ErrorCoded_1.ErrorCoded(`@nest is not allowed in the reverse property '${key}'`, ErrorCoded_1.ERROR_CODES.INVALID_REVERSE_PROPERTY);
                                    }
                                    break;
                                case '@container':
                                    if (processingMode === 1.0) {
                                        if (Object.keys(objectValue).length > 1
                                            || Util_1.Util.CONTAINERS_1_0.indexOf(Object.keys(objectValue)[0]) < 0) {
                                            throw new ErrorCoded_1.ErrorCoded(`Invalid term @container for '${key}' ('${Object.keys(objectValue)}') in 1.0, \
must be only one of ${Util_1.Util.CONTAINERS_1_0.join(', ')}`, ErrorCoded_1.ERROR_CODES.INVALID_CONTAINER_MAPPING);
                                        }
                                    }
                                    for (const containerValue of Object.keys(objectValue)) {
                                        if (containerValue === '@list' && value['@reverse']) {
                                            throw new ErrorCoded_1.ErrorCoded(`Term value can not be @container: @list and @reverse at the same time on '${key}'`, ErrorCoded_1.ERROR_CODES.INVALID_REVERSE_PROPERTY);
                                        }
                                        if (Util_1.Util.CONTAINERS.indexOf(containerValue) < 0) {
                                            throw new ErrorCoded_1.ErrorCoded(`Invalid term @container for '${key}' ('${containerValue}'), \
must be one of ${Util_1.Util.CONTAINERS.join(', ')}`, ErrorCoded_1.ERROR_CODES.INVALID_CONTAINER_MAPPING);
                                        }
                                    }
                                    break;
                                case '@language':
                                    ContextParser.validateLanguage(objectValue, true, ErrorCoded_1.ERROR_CODES.INVALID_LANGUAGE_MAPPING);
                                    break;
                                case '@direction':
                                    ContextParser.validateDirection(objectValue, true);
                                    break;
                                case '@prefix':
                                    if (objectValue !== null && typeof objectValue !== 'boolean') {
                                        throw new ErrorCoded_1.ErrorCoded(`Found an invalid term @prefix boolean in: '${key}': '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_PREFIX_VALUE);
                                    }
                                    if (!('@id' in value) && !Util_1.Util.isValidIri(key)) {
                                        throw new ErrorCoded_1.ErrorCoded(`Invalid @prefix definition for '${key}' ('${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_TERM_DEFINITION);
                                    }
                                    break;
                                case '@index':
                                    if (processingMode === 1.0 || !value['@container'] || !value['@container']['@index']) {
                                        throw new ErrorCoded_1.ErrorCoded(`Attempt to add illegal key to value object: '${key}': '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_TERM_DEFINITION);
                                    }
                                    break;
                                case '@nest':
                                    if (Util_1.Util.isPotentialKeyword(objectValue) && objectValue !== '@nest') {
                                        throw new ErrorCoded_1.ErrorCoded(`Found an invalid term @nest value in: '${key}': '${JSON.stringify(value)}'`, ErrorCoded_1.ERROR_CODES.INVALID_NEST_VALUE);
                                    }
                            }
                        }
                        break;
                    default:
                        throw new ErrorCoded_1.ErrorCoded(`Found an invalid term value: '${key}': '${value}'`, ErrorCoded_1.ERROR_CODES.INVALID_TERM_DEFINITION);
                }
            }
        }
    }
    /**
     * Apply the @base context entry to the given context under certain circumstances.
     * @param context A context.
     * @param options Parsing options.
     * @param inheritFromParent If the @base value from the parent context can be inherited.
     * @return The given context.
     */
    applyBaseEntry(context, options, inheritFromParent) {
        // In some special cases, this can be a string, so ignore those.
        if (typeof context === 'string') {
            return context;
        }
        // Give priority to @base in the parent context
        if (inheritFromParent && !('@base' in context) && options.parentContext
            && typeof options.parentContext === 'object' && '@base' in options.parentContext) {
            context['@base'] = options.parentContext['@base'];
            if (options.parentContext['@__baseDocument']) {
                context['@__baseDocument'] = true;
            }
        }
        // Override the base IRI if provided.
        if (options.baseIRI && !options.external) {
            if (!('@base' in context)) {
                // The context base is the document base
                context['@base'] = options.baseIRI;
                context['@__baseDocument'] = true;
            }
            else if (context['@base'] !== null && typeof context['@base'] === 'string'
                && !Util_1.Util.isValidIri(context['@base'])) {
                // The context base is relative to the document base
                context['@base'] = (0, relative_to_absolute_iri_1.resolve)(context['@base'], options.parentContext && options.parentContext['@base'] || options.baseIRI);
            }
        }
        return context;
    }
    /**
     * Resolve relative context IRIs, or return full IRIs as-is.
     * @param {string} contextIri A context IRI.
     * @param {string} baseIRI A base IRI.
     * @return {string} The normalized context IRI.
     */
    normalizeContextIri(contextIri, baseIRI) {
        if (!Util_1.Util.isValidIri(contextIri)) {
            try {
                contextIri = (0, relative_to_absolute_iri_1.resolve)(contextIri, baseIRI);
            }
            catch (_a) {
                throw new Error(`Invalid context IRI: ${contextIri}`);
            }
        }
        // TODO: Temporary workaround for fixing schema.org CORS issues (https://github.com/schemaorg/schemaorg/issues/2578#issuecomment-652324465)
        if (this.redirectSchemaOrgHttps && contextIri.startsWith('http://schema.org')) {
            contextIri = 'https://schema.org/';
        }
        return contextIri;
    }
    /**
     * Parse scoped contexts in the given context.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {IParseOptions} options Parsing options.
     * @return {IJsonLdContextNormalizedRaw} The mutated input context.
     */
    async parseInnerContexts(context, options) {
        for (const key of Object.keys(context)) {
            const value = context[key];
            if (value && typeof value === 'object') {
                if ('@context' in value && value['@context'] !== null && !options.ignoreScopedContexts) {
                    // Simulate a processing based on the parent context to check if there are any (potential errors).
                    // Honestly, I find it a bit weird to do this here, as the context may be unused,
                    // and the final effective context may differ based on any other embedded/scoped contexts.
                    // But hey, it's part of the spec, so we have no choice...
                    // https://w3c.github.io/json-ld-api/#h-note-10
                    if (this.validateContext) {
                        try {
                            const parentContext = Object.assign({}, context);
                            parentContext[key] = Object.assign({}, parentContext[key]);
                            delete parentContext[key]['@context'];
                            await this.parse(value['@context'], Object.assign(Object.assign({}, options), { external: false, parentContext, ignoreProtection: true, ignoreRemoteScopedContexts: true, ignoreScopedContexts: true }));
                        }
                        catch (e) {
                            throw new ErrorCoded_1.ErrorCoded(e.message, ErrorCoded_1.ERROR_CODES.INVALID_SCOPED_CONTEXT);
                        }
                    }
                    value['@context'] = (await this.parse(value['@context'], Object.assign(Object.assign({}, options), { external: false, minimalProcessing: true, ignoreRemoteScopedContexts: true, parentContext: context })))
                        .getContextRaw();
                }
            }
        }
        return context;
    }
    /**
     * Parse a JSON-LD context in any form.
     * @param {JsonLdContext} context A context, URL to a context, or an array of contexts/URLs.
     * @param {IParseOptions} options Optional parsing options.
     * @return {Promise<JsonLdContextNormalized>} A promise resolving to the context.
     */
    async parse(context, options = {}) {
        const { baseIRI, parentContext: parentContextInitial, external, processingMode = ContextParser.DEFAULT_PROCESSING_MODE, normalizeLanguageTags, ignoreProtection, minimalProcessing, } = options;
        let parentContext = parentContextInitial;
        const remoteContexts = options.remoteContexts || {};
        // Avoid remote context overflows
        if (Object.keys(remoteContexts).length >= this.remoteContextsDepthLimit) {
            throw new ErrorCoded_1.ErrorCoded('Detected an overflow in remote context inclusions: ' + Object.keys(remoteContexts), ErrorCoded_1.ERROR_CODES.CONTEXT_OVERFLOW);
        }
        if (context === null || context === undefined) {
            // Don't allow context nullification and there are protected terms
            if (!ignoreProtection && parentContext && Util_1.Util.hasProtectedTerms(parentContext)) {
                throw new ErrorCoded_1.ErrorCoded('Illegal context nullification when terms are protected', ErrorCoded_1.ERROR_CODES.INVALID_CONTEXT_NULLIFICATION);
            }
            // Context that are explicitly set to null are empty.
            return new JsonLdContextNormalized_1.JsonLdContextNormalized(this.applyBaseEntry({}, options, false));
        }
        else if (typeof context === 'string') {
            const contextIri = this.normalizeContextIri(context, baseIRI);
            const overriddenLoad = this.getOverriddenLoad(contextIri, options);
            if (overriddenLoad) {
                return new JsonLdContextNormalized_1.JsonLdContextNormalized(overriddenLoad);
            }
            const parsedStringContext = await this.parse(await this.load(contextIri), Object.assign(Object.assign({}, options), { baseIRI: contextIri, external: true, remoteContexts: Object.assign(Object.assign({}, remoteContexts), { [contextIri]: true }) }));
            this.applyBaseEntry(parsedStringContext.getContextRaw(), options, true);
            return parsedStringContext;
        }
        else if (Array.isArray(context)) {
            // As a performance consideration, first load all external contexts in parallel.
            const contextIris = [];
            const contexts = await Promise.all(context.map((subContext, i) => {
                if (typeof subContext === 'string') {
                    const contextIri = this.normalizeContextIri(subContext, baseIRI);
                    contextIris[i] = contextIri;
                    const overriddenLoad = this.getOverriddenLoad(contextIri, options);
                    if (overriddenLoad) {
                        return overriddenLoad;
                    }
                    return this.load(contextIri);
                }
                else {
                    return subContext;
                }
            }));
            // Don't apply inheritance logic on minimal processing
            if (minimalProcessing) {
                return new JsonLdContextNormalized_1.JsonLdContextNormalized(contexts);
            }
            const reducedContexts = await contexts.reduce((accContextPromise, contextEntry, i) => accContextPromise
                .then((accContext) => this.parse(contextEntry, Object.assign(Object.assign({}, options), { baseIRI: contextIris[i] || options.baseIRI, external: !!contextIris[i] || options.external, parentContext: accContext.getContextRaw(), remoteContexts: contextIris[i] ? Object.assign(Object.assign({}, remoteContexts), { [contextIris[i]]: true }) : remoteContexts }))), Promise.resolve(new JsonLdContextNormalized_1.JsonLdContextNormalized(parentContext || {})));
            // Override the base IRI if provided.
            this.applyBaseEntry(reducedContexts.getContextRaw(), options, true);
            return reducedContexts;
        }
        else if (typeof context === 'object') {
            if ('@context' in context) {
                return await this.parse(context['@context'], options);
            }
            // Make a deep clone of the given context, to avoid modifying it.
            context = JSON.parse(JSON.stringify(context)); // No better way in JS at the moment.
            if (parentContext && !minimalProcessing) {
                parentContext = JSON.parse(JSON.stringify(parentContext));
            }
            // We have an actual context object.
            let newContext = {};
            // According to the JSON-LD spec, @base must be ignored from external contexts.
            if (external) {
                delete context['@base'];
            }
            // Override the base IRI if provided.
            this.applyBaseEntry(context, options, true);
            // Hashify container entries
            // Do this before protected term validation as that influences term format
            this.containersToHash(context);
            // Don't perform any other modifications if only minimal processing is needed.
            if (minimalProcessing) {
                return new JsonLdContextNormalized_1.JsonLdContextNormalized(context);
            }
            // In JSON-LD 1.1, load @import'ed context prior to processing.
            let importContext = {};
            if ('@import' in context) {
                if (processingMode >= 1.1) {
                    // Only accept string values
                    if (typeof context['@import'] !== 'string') {
                        throw new ErrorCoded_1.ErrorCoded('An @import value must be a string, but got ' + typeof context['@import'], ErrorCoded_1.ERROR_CODES.INVALID_IMPORT_VALUE);
                    }
                    // Load context
                    importContext = await this.loadImportContext(this.normalizeContextIri(context['@import'], baseIRI));
                    delete context['@import'];
                }
                else {
                    throw new ErrorCoded_1.ErrorCoded('Context importing is not supported in JSON-LD 1.0', ErrorCoded_1.ERROR_CODES.INVALID_CONTEXT_ENTRY);
                }
            }
            // Merge different parts of the final context in order
            newContext = Object.assign(Object.assign(Object.assign(Object.assign({}, newContext), (typeof parentContext === 'object' ? parentContext : {})), importContext), context);
            const newContextWrapped = new JsonLdContextNormalized_1.JsonLdContextNormalized(newContext);
            // Parse inner contexts with minimal processing
            await this.parseInnerContexts(newContext, options);
            // In JSON-LD 1.1, @vocab can be relative to @vocab in the parent context.
            if ((newContext && newContext['@version'] || ContextParser.DEFAULT_PROCESSING_MODE) >= 1.1
                && ((context['@vocab'] && typeof context['@vocab'] === 'string') || context['@vocab'] === '')
                && context['@vocab'].indexOf(':') < 0 && parentContext && '@vocab' in parentContext) {
                newContext['@vocab'] = parentContext['@vocab'] + context['@vocab'];
            }
            // Handle terms (before protection checks)
            this.idifyReverseTerms(newContext);
            this.expandPrefixedTerms(newContextWrapped, this.expandContentTypeToBase);
            // In JSON-LD 1.1, check if we are not redefining any protected keywords
            if (!ignoreProtection && parentContext && processingMode >= 1.1) {
                this.validateKeywordRedefinitions(parentContext, newContext, exports.defaultExpandOptions);
            }
            this.normalize(newContext, { processingMode, normalizeLanguageTags });
            this.applyScopedProtected(newContext, { processingMode });
            if (this.validateContext) {
                this.validate(newContext, { processingMode });
            }
            return newContextWrapped;
        }
        else {
            throw new ErrorCoded_1.ErrorCoded(`Tried parsing a context that is not a string, array or object, but got ${context}`, ErrorCoded_1.ERROR_CODES.INVALID_LOCAL_CONTEXT);
        }
    }
    /**
     * Fetch the given URL as a raw JSON-LD context.
     * @param url An URL.
     * @return A promise resolving to a raw JSON-LD context.
     */
    async load(url) {
        // First try to retrieve the context from cache
        const cached = this.documentCache[url];
        if (cached) {
            return typeof cached === 'string' ? cached : Array.isArray(cached) ? cached.slice() : Object.assign({}, cached);
        }
        // If not in cache, load it
        let document;
        try {
            document = await this.documentLoader.load(url);
        }
        catch (e) {
            throw new ErrorCoded_1.ErrorCoded(`Failed to load remote context ${url}: ${e.message}`, ErrorCoded_1.ERROR_CODES.LOADING_REMOTE_CONTEXT_FAILED);
        }
        // Validate the context
        if (!('@context' in document)) {
            throw new ErrorCoded_1.ErrorCoded(`Missing @context in remote context at ${url}`, ErrorCoded_1.ERROR_CODES.INVALID_REMOTE_CONTEXT);
        }
        return this.documentCache[url] = document['@context'];
    }
    /**
     * Override the given context that may be loaded.
     *
     * This will check whether or not the url is recursively being loaded.
     * @param url An URL.
     * @param options Parsing options.
     * @return An overridden context, or null.
     *         Optionally an error can be thrown if a cyclic context is detected.
     */
    getOverriddenLoad(url, options) {
        if (url in (options.remoteContexts || {})) {
            if (options.ignoreRemoteScopedContexts) {
                return url;
            }
            else {
                throw new ErrorCoded_1.ErrorCoded('Detected a cyclic context inclusion of ' + url, ErrorCoded_1.ERROR_CODES.RECURSIVE_CONTEXT_INCLUSION);
            }
        }
        return null;
    }
    /**
     * Load an @import'ed context.
     * @param importContextIri The full URI of an @import value.
     */
    async loadImportContext(importContextIri) {
        // Load the context
        const importContext = await this.load(importContextIri);
        // Require the context to be a non-array object
        if (typeof importContext !== 'object' || Array.isArray(importContext)) {
            throw new ErrorCoded_1.ErrorCoded('An imported context must be a single object: ' + importContextIri, ErrorCoded_1.ERROR_CODES.INVALID_REMOTE_CONTEXT);
        }
        // Error if the context contains another @import
        if ('@import' in importContext) {
            throw new ErrorCoded_1.ErrorCoded('An imported context can not import another context: ' + importContextIri, ErrorCoded_1.ERROR_CODES.INVALID_CONTEXT_ENTRY);
        }
        // Containers have to be converted into hash values the same way as for the importing context
        // Otherwise context validation will fail for container values
        this.containersToHash(importContext);
        return importContext;
    }
}
exports.ContextParser = ContextParser;
ContextParser.DEFAULT_PROCESSING_MODE = 1.1;
exports.defaultExpandOptions = {
    allowPrefixForcing: true,
    allowPrefixNonGenDelims: false,
    allowVocabRelativeToBase: true,
};
//# sourceMappingURL=ContextParser.js.map