"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FetchDocumentLoader = void 0;
require("cross-fetch/polyfill");
const ErrorCoded_1 = require("./ErrorCoded");
const http_link_header_1 = require("http-link-header");
const relative_to_absolute_iri_1 = require("relative-to-absolute-iri");
/**
 * Loads documents via the fetch API.
 */
class FetchDocumentLoader {
    constructor(fetcher) {
        this.fetcher = fetcher;
    }
    async load(url) {
        const response = await (this.fetcher || fetch)(url, { headers: new Headers({ accept: 'application/ld+json' }) });
        if (response.ok && response.headers) {
            let mediaType = response.headers.get('Content-Type');
            if (mediaType) {
                const colonPos = mediaType.indexOf(';');
                if (colonPos > 0) {
                    mediaType = mediaType.substr(0, colonPos);
                }
            }
            if (mediaType === 'application/ld+json') {
                // Return JSON-LD if proper content type was returned
                return (await response.json());
            }
            else {
                // Check for alternate link for a non-JSON-LD response
                if (response.headers.has('Link')) {
                    let alternateUrl;
                    response.headers.forEach((value, key) => {
                        if (key === 'link') {
                            const linkHeader = (0, http_link_header_1.parse)(value);
                            for (const link of linkHeader.get('type', 'application/ld+json')) {
                                if (link.rel === 'alternate') {
                                    if (alternateUrl) {
                                        throw new Error('Multiple JSON-LD alternate links were found on ' + url);
                                    }
                                    alternateUrl = (0, relative_to_absolute_iri_1.resolve)(link.uri, url);
                                }
                            }
                        }
                    });
                    if (alternateUrl) {
                        return this.load(alternateUrl);
                    }
                }
                throw new ErrorCoded_1.ErrorCoded(`Unsupported JSON-LD media type ${mediaType}`, ErrorCoded_1.ERROR_CODES.LOADING_DOCUMENT_FAILED);
            }
        }
        else {
            throw new Error(response.statusText || `Status code: ${response.status}`);
        }
    }
}
exports.FetchDocumentLoader = FetchDocumentLoader;
//# sourceMappingURL=FetchDocumentLoader.js.map