"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonLdContextNormalized = void 0;
const relative_to_absolute_iri_1 = require("relative-to-absolute-iri");
const ContextParser_1 = require("./ContextParser");
const ErrorCoded_1 = require("./ErrorCoded");
const Util_1 = require("./Util");
/**
 * A class exposing operations over a normalized JSON-LD context.
 */
class JsonLdContextNormalized {
    constructor(contextRaw) {
        this.contextRaw = contextRaw;
    }
    /**
     * @return The raw inner context.
     */
    getContextRaw() {
        return this.contextRaw;
    }
    /**
     * Expand the term or prefix of the given term if it has one,
     * otherwise return the term as-is.
     *
     * This will try to expand the IRI as much as possible.
     *
     * Iff in vocab-mode, then other references to other terms in the context can be used,
     * such as to `myTerm`:
     * ```
     * {
     *   "myTerm": "http://example.org/myLongTerm"
     * }
     * ```
     *
     * @param {string} term A term that is an URL or a prefixed URL.
     * @param {boolean} expandVocab If the term is a predicate or type and should be expanded based on @vocab,
     *                              otherwise it is considered a regular term that is expanded based on @base.
     * @param {IExpandOptions} options Options that define the way how expansion must be done.
     * @return {string} The expanded term, the term as-is, or null if it was explicitly disabled in the context.
     * @throws If the term is aliased to an invalid value (not a string, IRI or keyword).
     */
    expandTerm(term, expandVocab, options = ContextParser_1.defaultExpandOptions) {
        const contextValue = this.contextRaw[term];
        // Immediately return if the term was disabled in the context
        if (contextValue === null || (contextValue && contextValue['@id'] === null)) {
            return null;
        }
        // Check the @id
        let validIriMapping = true;
        if (contextValue && expandVocab) {
            const value = Util_1.Util.getContextValueId(contextValue);
            if (value && value !== term) {
                if (typeof value !== 'string' || (!Util_1.Util.isValidIri(value) && !Util_1.Util.isValidKeyword(value))) {
                    // Don't mark this mapping as invalid if we have an unknown keyword, but of the correct form.
                    if (!Util_1.Util.isPotentialKeyword(value)) {
                        validIriMapping = false;
                    }
                }
                else {
                    return value;
                }
            }
        }
        // Check if the term is prefixed
        const prefix = Util_1.Util.getPrefix(term, this.contextRaw);
        const vocab = this.contextRaw['@vocab'];
        const vocabRelative = (!!vocab || vocab === '') && vocab.indexOf(':') < 0;
        const base = this.contextRaw['@base'];
        const potentialKeyword = Util_1.Util.isPotentialKeyword(term);
        if (prefix) {
            const contextPrefixValue = this.contextRaw[prefix];
            const value = Util_1.Util.getContextValueId(contextPrefixValue);
            if (value) {
                if (typeof contextPrefixValue === 'string' || !options.allowPrefixForcing) {
                    // If we have a simple term definition,
                    // check the last character of the prefix to determine whether or not it is a prefix.
                    // Validate that prefix ends with gen-delim character, unless @prefix is true
                    if (!Util_1.Util.isSimpleTermDefinitionPrefix(value, options)) {
                        // Treat the term as an absolute IRI
                        return term;
                    }
                }
                else {
                    // If we have an expanded term definition, default to @prefix: false
                    if (value[0] !== '_' && !potentialKeyword && !contextPrefixValue['@prefix'] && !(term in this.contextRaw)) {
                        // Treat the term as an absolute IRI
                        return term;
                    }
                }
                return value + term.substr(prefix.length + 1);
            }
        }
        else if (expandVocab && ((vocab || vocab === '') || (options.allowVocabRelativeToBase && (base && vocabRelative)))
            && !potentialKeyword && !Util_1.Util.isCompactIri(term)) {
            if (vocabRelative) {
                if (options.allowVocabRelativeToBase) {
                    return ((vocab || base) ? (0, relative_to_absolute_iri_1.resolve)(vocab, base) : '') + term;
                }
                else {
                    throw new ErrorCoded_1.ErrorCoded(`Relative vocab expansion for term '${term}' with vocab '${vocab}' is not allowed.`, ErrorCoded_1.ERROR_CODES.INVALID_VOCAB_MAPPING);
                }
            }
            else {
                return vocab + term;
            }
        }
        else if (!expandVocab && base && !potentialKeyword && !Util_1.Util.isCompactIri(term)) {
            return (0, relative_to_absolute_iri_1.resolve)(term, base);
        }
        // Return the term as-is, unless we discovered an invalid IRI mapping for this term in the context earlier.
        if (validIriMapping) {
            return term;
        }
        else {
            throw new ErrorCoded_1.ErrorCoded(`Invalid IRI mapping found for context entry '${term}': '${JSON.stringify(contextValue)}'`, ErrorCoded_1.ERROR_CODES.INVALID_IRI_MAPPING);
        }
    }
    /**
     * Compact the given term using @base, @vocab, an aliased term, or a prefixed term.
     *
     * This will try to compact the IRI as much as possible.
     *
     * @param {string} iri An IRI to compact.
     * @param {boolean} vocab If the term is a predicate or type and should be compacted based on @vocab,
     *                        otherwise it is considered a regular term that is compacted based on @base.
     * @return {string} The compacted term or the IRI as-is.
     */
    compactIri(iri, vocab) {
        // Try @vocab compacting
        if (vocab && this.contextRaw['@vocab'] && iri.startsWith(this.contextRaw['@vocab'])) {
            return iri.substr(this.contextRaw['@vocab'].length);
        }
        // Try @base compacting
        if (!vocab && this.contextRaw['@base'] && iri.startsWith(this.contextRaw['@base'])) {
            return iri.substr(this.contextRaw['@base'].length);
        }
        // Loop over all terms in the context
        // This will try to prefix as short as possible.
        // Once a fully compacted alias is found, return immediately, as we can not go any shorter.
        const shortestPrefixing = { prefix: '', suffix: iri };
        for (const key in this.contextRaw) {
            const value = this.contextRaw[key];
            if (value && !Util_1.Util.isPotentialKeyword(key)) {
                const contextIri = Util_1.Util.getContextValueId(value);
                if (iri.startsWith(contextIri)) {
                    const suffix = iri.substr(contextIri.length);
                    if (!suffix) {
                        if (vocab) {
                            // Immediately return on compacted alias
                            return key;
                        }
                    }
                    else if (suffix.length < shortestPrefixing.suffix.length) {
                        // Overwrite the shortest prefix
                        shortestPrefixing.prefix = key;
                        shortestPrefixing.suffix = suffix;
                    }
                }
            }
        }
        // Return the shortest prefix
        if (shortestPrefixing.prefix) {
            return shortestPrefixing.prefix + ':' + shortestPrefixing.suffix;
        }
        return iri;
    }
}
exports.JsonLdContextNormalized = JsonLdContextNormalized;
//# sourceMappingURL=JsonLdContextNormalized.js.map