import { IExpandOptions } from "./ContextParser";
import { IJsonLdContextNormalizedRaw } from "./JsonLdContext";
export declare class Util {
    static readonly IRI_REGEX: RegExp;
    static readonly IRI_REGEX_WEAK: RegExp;
    static readonly KEYWORD_REGEX: RegExp;
    static readonly ENDS_WITH_GEN_DELIM: RegExp;
    static readonly REGEX_LANGUAGE_TAG: RegExp;
    static readonly REGEX_DIRECTION_TAG: RegExp;
    static readonly VALID_KEYWORDS: {
        [keyword: string]: boolean;
    };
    static readonly EXPAND_KEYS_BLACKLIST: string[];
    static readonly ALIAS_DOMAIN_BLACKLIST: string[];
    static readonly ALIAS_RANGE_BLACKLIST: string[];
    static readonly CONTAINERS: string[];
    static readonly CONTAINERS_1_0: string[];
    /**
     * Check if the given term is a valid compact IRI.
     * Otherwise, it may be an IRI.
     * @param {string} term A term.
     * @return {boolean} If it is a compact IRI.
     */
    static isCompactIri(term: string): boolean;
    /**
     * Get the prefix from the given term.
     * @see https://json-ld.org/spec/latest/json-ld/#compact-iris
     * @param {string} term A term that is an URL or a prefixed URL.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @return {string} The prefix or null.
     */
    static getPrefix(term: string, context: IJsonLdContextNormalizedRaw): string | null;
    /**
     * From a given context entry value, get the string value, or the @id field.
     * @param contextValue A value for a term in a context.
     * @return {string} The id value, or null.
     */
    static getContextValueId(contextValue: any): string;
    /**
     * Check if the given simple term definition (string-based value of a context term)
     * should be considered a prefix.
     * @param value A simple term definition value.
     * @param options Options that define the way how expansion must be done.
     */
    static isSimpleTermDefinitionPrefix(value: string, options: IExpandOptions): boolean;
    /**
     * Check if the given keyword is of the keyword format "@"1*ALPHA.
     * @param {string} keyword A potential keyword.
     * @return {boolean} If the given keyword is of the keyword format.
     */
    static isPotentialKeyword(keyword: any): boolean;
    /**
     * Check if the given prefix ends with a gen-delim character.
     * @param {string} prefixIri A prefix IRI.
     * @return {boolean} If the given prefix IRI is valid.
     */
    static isPrefixIriEndingWithGenDelim(prefixIri: string): boolean;
    /**
     * Check if the given context value can be a prefix value.
     * @param value A context value.
     * @return {boolean} If it can be a prefix value.
     */
    static isPrefixValue(value: any): boolean;
    /**
     * Check if the given IRI is valid.
     * @param {string} iri A potential IRI.
     * @return {boolean} If the given IRI is valid.
     */
    static isValidIri(iri: string | null): boolean;
    /**
     * Check if the given IRI is valid, this includes the possibility of being a relative IRI.
     * @param {string} iri A potential IRI.
     * @return {boolean} If the given IRI is valid.
     */
    static isValidIriWeak(iri: string | null): boolean;
    /**
     * Check if the given keyword is a defined according to the JSON-LD specification.
     * @param {string} keyword A potential keyword.
     * @return {boolean} If the given keyword is valid.
     */
    static isValidKeyword(keyword: any): boolean;
    /**
     * Check if the given term is protected in the context.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {string} key A context term.
     * @return {boolean} If the given term has an @protected flag.
     */
    static isTermProtected(context: IJsonLdContextNormalizedRaw, key: string): boolean;
    /**
     * Check if the given context has at least one protected term.
     * @param context A context.
     * @return If the context has a protected term.
     */
    static hasProtectedTerms(context: IJsonLdContextNormalizedRaw): boolean;
    /**
     * Check if the given key is an internal reserved keyword.
     * @param key A context key.
     */
    static isReservedInternalKeyword(key: string): boolean;
}
