"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Util = void 0;
class Util {
    /**
     * Check if the given term is a valid compact IRI.
     * Otherwise, it may be an IRI.
     * @param {string} term A term.
     * @return {boolean} If it is a compact IRI.
     */
    static isCompactIri(term) {
        return term.indexOf(':') > 0 && !(term && term[0] === '#');
    }
    /**
     * Get the prefix from the given term.
     * @see https://json-ld.org/spec/latest/json-ld/#compact-iris
     * @param {string} term A term that is an URL or a prefixed URL.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @return {string} The prefix or null.
     */
    static getPrefix(term, context) {
        // Do not consider relative IRIs starting with a hash as compact IRIs
        if (term && term[0] === '#') {
            return null;
        }
        const separatorPos = term.indexOf(':');
        if (separatorPos >= 0) {
            // Suffix can not begin with two slashes
            if (term.length > separatorPos + 1
                && term.charAt(separatorPos + 1) === '/'
                && term.charAt(separatorPos + 2) === '/') {
                return null;
            }
            const prefix = term.substr(0, separatorPos);
            // Prefix can not be an underscore (this is a blank node)
            if (prefix === '_') {
                return null;
            }
            // Prefix must match a term in the active context
            if (context[prefix]) {
                return prefix;
            }
        }
        return null;
    }
    /**
     * From a given context entry value, get the string value, or the @id field.
     * @param contextValue A value for a term in a context.
     * @return {string} The id value, or null.
     */
    static getContextValueId(contextValue) {
        if (contextValue === null || typeof contextValue === 'string') {
            return contextValue;
        }
        const id = contextValue['@id'];
        return id ? id : null;
    }
    /**
     * Check if the given simple term definition (string-based value of a context term)
     * should be considered a prefix.
     * @param value A simple term definition value.
     * @param options Options that define the way how expansion must be done.
     */
    static isSimpleTermDefinitionPrefix(value, options) {
        return !Util.isPotentialKeyword(value)
            && (value[0] === '_' || options.allowPrefixNonGenDelims || Util.isPrefixIriEndingWithGenDelim(value));
    }
    /**
     * Check if the given keyword is of the keyword format "@"1*ALPHA.
     * @param {string} keyword A potential keyword.
     * @return {boolean} If the given keyword is of the keyword format.
     */
    static isPotentialKeyword(keyword) {
        return typeof keyword === 'string' && Util.KEYWORD_REGEX.test(keyword);
    }
    /**
     * Check if the given prefix ends with a gen-delim character.
     * @param {string} prefixIri A prefix IRI.
     * @return {boolean} If the given prefix IRI is valid.
     */
    static isPrefixIriEndingWithGenDelim(prefixIri) {
        return Util.ENDS_WITH_GEN_DELIM.test(prefixIri);
    }
    /**
     * Check if the given context value can be a prefix value.
     * @param value A context value.
     * @return {boolean} If it can be a prefix value.
     */
    static isPrefixValue(value) {
        return value && (typeof value === 'string' || (value && typeof value === 'object'));
    }
    /**
     * Check if the given IRI is valid.
     * @param {string} iri A potential IRI.
     * @return {boolean} If the given IRI is valid.
     */
    static isValidIri(iri) {
        return Boolean(iri && Util.IRI_REGEX.test(iri));
    }
    /**
     * Check if the given IRI is valid, this includes the possibility of being a relative IRI.
     * @param {string} iri A potential IRI.
     * @return {boolean} If the given IRI is valid.
     */
    static isValidIriWeak(iri) {
        return !!iri && iri[0] !== ':' && Util.IRI_REGEX_WEAK.test(iri);
    }
    /**
     * Check if the given keyword is a defined according to the JSON-LD specification.
     * @param {string} keyword A potential keyword.
     * @return {boolean} If the given keyword is valid.
     */
    static isValidKeyword(keyword) {
        return Util.VALID_KEYWORDS[keyword];
    }
    /**
     * Check if the given term is protected in the context.
     * @param {IJsonLdContextNormalizedRaw} context A context.
     * @param {string} key A context term.
     * @return {boolean} If the given term has an @protected flag.
     */
    static isTermProtected(context, key) {
        const value = context[key];
        return !(typeof value === 'string') && value && value['@protected'];
    }
    /**
     * Check if the given context has at least one protected term.
     * @param context A context.
     * @return If the context has a protected term.
     */
    static hasProtectedTerms(context) {
        for (const key of Object.keys(context)) {
            if (Util.isTermProtected(context, key)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Check if the given key is an internal reserved keyword.
     * @param key A context key.
     */
    static isReservedInternalKeyword(key) {
        return key.startsWith('@__');
    }
}
exports.Util = Util;
// Regex for valid IRIs
Util.IRI_REGEX = /^([A-Za-z][A-Za-z0-9+-.]*|_):[^ "<>{}|\\\[\]`#]*(#[^#]*)?$/;
// Weaker regex for valid IRIs, this includes relative IRIs
Util.IRI_REGEX_WEAK = /(?::[^:])|\//;
// Regex for keyword form
Util.KEYWORD_REGEX = /^@[a-z]+$/i;
// Regex to see if an IRI ends with a gen-delim character (see RFC 3986)
Util.ENDS_WITH_GEN_DELIM = /[:/?#\[\]@]$/;
// Regex for language tags
Util.REGEX_LANGUAGE_TAG = /^[a-zA-Z]+(-[a-zA-Z0-9]+)*$/;
// Regex for base directions
Util.REGEX_DIRECTION_TAG = /^(ltr)|(rtl)$/;
// All known valid JSON-LD keywords
// @see https://www.w3.org/TR/json-ld11/#keywords
Util.VALID_KEYWORDS = {
    '@base': true,
    '@container': true,
    '@context': true,
    '@direction': true,
    '@graph': true,
    '@id': true,
    '@import': true,
    '@included': true,
    '@index': true,
    '@json': true,
    '@language': true,
    '@list': true,
    '@nest': true,
    '@none': true,
    '@prefix': true,
    '@propagate': true,
    '@protected': true,
    '@reverse': true,
    '@set': true,
    '@type': true,
    '@value': true,
    '@version': true,
    '@vocab': true,
};
// Keys in the contexts that will not be expanded based on the base IRI
Util.EXPAND_KEYS_BLACKLIST = [
    '@base',
    '@vocab',
    '@language',
    '@version',
    '@direction',
];
// Keys in the contexts that may not be aliased from
Util.ALIAS_DOMAIN_BLACKLIST = [
    '@container',
    '@graph',
    '@id',
    '@index',
    '@list',
    '@nest',
    '@none',
    '@prefix',
    '@reverse',
    '@set',
    '@type',
    '@value',
    '@version',
];
// Keys in the contexts that may not be aliased to
Util.ALIAS_RANGE_BLACKLIST = [
    '@context',
    '@preserve',
];
// All valid @container values
Util.CONTAINERS = [
    '@list',
    '@set',
    '@index',
    '@language',
    '@graph',
    '@id',
    '@type',
];
// All valid @container values under processing mode 1.0
Util.CONTAINERS_1_0 = [
    '@list',
    '@set',
    '@index',
];
//# sourceMappingURL=Util.js.map