/// <reference types="node" />
import * as RDF from "rdf-js";
import { IDocumentLoader, JsonLdContext } from "jsonld-context-parser";
import { Transform, TransformCallback } from "stream";
import { IEntryHandler } from "./entryhandler/IEntryHandler";
import EventEmitter = NodeJS.EventEmitter;
/**
 * A stream transformer that parses JSON-LD (text) streams to an {@link RDF.Stream}.
 */
export declare class JsonLdParser extends Transform {
    static readonly DEFAULT_PROCESSING_MODE: string;
    static readonly ENTRY_HANDLERS: IEntryHandler<any>[];
    private readonly options;
    private readonly parsingContext;
    private readonly util;
    private readonly jsonParser;
    private readonly contextAwaitingJobs;
    private readonly contextJobs;
    private lastDepth;
    private lastOnValueJob;
    constructor(options?: IJsonLdParserOptions);
    /**
     * Parses the given text stream into a quad stream.
     * @param {NodeJS.EventEmitter} stream A text stream.
     * @return {NodeJS.EventEmitter} A quad stream.
     */
    import(stream: EventEmitter): EventEmitter;
    _transform(chunk: any, encoding: string, callback: TransformCallback): void;
    /**
     * Start a new job for parsing the given value.
     *
     * This will let the first valid {@link IEntryHandler} handle the entry.
     *
     * @param {any[]} keys The stack of keys.
     * @param value The value to parse.
     * @param {number} depth The depth to parse at.
     * @return {Promise<void>} A promise resolving when the job is done.
     */
    newOnValueJob(keys: any[], value: any, depth: number): Promise<void>;
    /**
     * Check if at least one {@link IEntryHandler} validates the entry to true.
     * @param {any[]} keys A stack of keys.
     * @param {number} depth A depth.
     * @param {boolean} inProperty If the current depth is part of a valid property node.
     * @return {Promise<{ valid: boolean, property: boolean }>} A promise resolving to true or false.
     */
    protected validateKey(keys: any[], depth: number, inProperty: boolean): Promise<{
        valid: boolean;
        property: boolean;
    }>;
    /**
     * Attach all required listeners to the JSON parser.
     *
     * This should only be called once.
     */
    protected attachJsonParserListeners(): void;
    /**
     * Check if the parser is currently parsing an element that is part of an @context entry.
     * @param {number} depth A depth.
     * @return {boolean} A boolean.
     */
    protected isParsingContextInner(depth: number): boolean;
    /**
     * Execute all buffered jobs.
     * @return {Promise<void>} A promise resolving if all jobs are finished.
     */
    protected executeBufferedJobs(): Promise<void>;
    /**
     * Flush buffers for the given depth.
     *
     * This should be called after the last entry at a given depth was processed.
     *
     * @param {number} depth A depth.
     * @param {any[]} keys A stack of keys.
     * @return {Promise<void>} A promise resolving if flushing is done.
     */
    protected flushBuffer(depth: number, keys: any[]): Promise<void>;
}
/**
 * Constructor arguments for {@link JsonLdParser}
 */
export interface IJsonLdParserOptions {
    /**
     * A data factory.
     */
    dataFactory?: RDF.DataFactory;
    /**
     * The root context.
     */
    context?: JsonLdContext;
    /**
     * The base IRI.
     */
    baseIRI?: string;
    /**
     * If @context definitions should be allowed as non-first object entries.
     * When enabled, streaming results may not come as soon as possible,
     * and will be buffered until the end when no context is defined at all.
     * Defaults to false.
     *
     * Spec-compliance: to be fully spec-compliant,
     * this must be explicitly set to true.
     */
    allowOutOfOrderContext?: boolean;
    /**
     * Loader for remote contexts.
     */
    documentLoader?: IDocumentLoader;
    /**
     * If blank node predicates should be allowed,
     * they will be ignored otherwise.
     * Defaults to false.
     */
    produceGeneralizedRdf?: boolean;
    /**
     * The maximum JSON-LD version that should be processable by this parser.
     * Defaults to JsonLdParser.DEFAULT_PROCESSING_MODE.
     */
    processingMode?: string;
    /**
     * By default, JSON-LD requires that
     * all properties (or @id's) that are not URIs,
     * are unknown keywords,
     * and do not occur in the context
     * should be silently dropped.
     * When setting this value to true,
     * an error will be thrown when such properties occur.
     * Defaults to false.
     */
    errorOnInvalidIris?: boolean;
    /**
     * If RDF lists can appear in the subject position.
     * Defaults to false.
     */
    allowSubjectList?: boolean;
    /**
     * If @index inside array nodes should be validated.
     * I.e., nodes inside the same array with the same @id,
     * should have equal @index values.
     * This is not applicable to this parser as we don't do explicit flattening,
     * but it is required to be spec-compliant.
     * Defaults to false.
     *
     * Spec-compliance: to be fully spec-compliant,
     * this must be explicitly set to true.
     */
    validateValueIndexes?: boolean;
    /**
     * The graph to use as default graph when no explicit @graph is set.
     * Defaults to dataFactory.defaultGraph().
     */
    defaultGraph?: RDF.Term;
}
