"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tslint:disable-next-line:no-var-requires
const Parser = require('jsonparse');
const stream_1 = require("stream");
const EntryHandlerArrayValue_1 = require("./entryhandler/EntryHandlerArrayValue");
const EntryHandlerContainer_1 = require("./entryhandler/EntryHandlerContainer");
const EntryHandlerInvalidFallback_1 = require("./entryhandler/EntryHandlerInvalidFallback");
const EntryHandlerPredicate_1 = require("./entryhandler/EntryHandlerPredicate");
const EntryHandlerKeywordContext_1 = require("./entryhandler/keyword/EntryHandlerKeywordContext");
const EntryHandlerKeywordGraph_1 = require("./entryhandler/keyword/EntryHandlerKeywordGraph");
const EntryHandlerKeywordId_1 = require("./entryhandler/keyword/EntryHandlerKeywordId");
const EntryHandlerKeywordType_1 = require("./entryhandler/keyword/EntryHandlerKeywordType");
const EntryHandlerKeywordUnknownFallback_1 = require("./entryhandler/keyword/EntryHandlerKeywordUnknownFallback");
const EntryHandlerKeywordValue_1 = require("./entryhandler/keyword/EntryHandlerKeywordValue");
const ParsingContext_1 = require("./ParsingContext");
const Util_1 = require("./Util");
/**
 * A stream transformer that parses JSON-LD (text) streams to an {@link RDF.Stream}.
 */
class JsonLdParser extends stream_1.Transform {
    constructor(options) {
        super({ objectMode: true });
        options = options || {};
        this.options = options;
        this.parsingContext = new ParsingContext_1.ParsingContext(Object.assign({ parser: this }, options));
        this.util = new Util_1.Util({ dataFactory: options.dataFactory, parsingContext: this.parsingContext });
        this.jsonParser = new Parser();
        this.contextAwaitingJobs = [];
        this.contextJobs = [];
        this.lastDepth = 0;
        this.lastOnValueJob = Promise.resolve();
        this.attachJsonParserListeners();
    }
    /**
     * Parses the given text stream into a quad stream.
     * @param {NodeJS.EventEmitter} stream A text stream.
     * @return {NodeJS.EventEmitter} A quad stream.
     */
    import(stream) {
        const output = new stream_1.PassThrough({ objectMode: true });
        stream.on('error', (error) => parsed.emit('error', error));
        stream.on('data', (data) => output.write(data));
        stream.on('end', () => output.emit('end'));
        const parsed = output.pipe(new JsonLdParser(this.options));
        return parsed;
    }
    _transform(chunk, encoding, callback) {
        this.jsonParser.write(chunk);
        this.lastOnValueJob
            .then(() => callback(), (error) => callback(error));
    }
    /**
     * Start a new job for parsing the given value.
     *
     * This will let the first valid {@link IEntryHandler} handle the entry.
     *
     * @param {any[]} keys The stack of keys.
     * @param value The value to parse.
     * @param {number} depth The depth to parse at.
     * @return {Promise<void>} A promise resolving when the job is done.
     */
    async newOnValueJob(keys, value, depth) {
        // When we go up the stack, emit all unidentified values
        // We need to do this before the new job, because the new job may require determined values from the flushed jobs.
        if (depth < this.lastDepth) {
            // Check if we had any RDF lists that need to be terminated with an rdf:nil
            const listPointer = this.parsingContext.listPointerStack[this.lastDepth];
            if (listPointer) {
                if (listPointer.term) {
                    this.emit('data', this.util.dataFactory.quad(listPointer.term, this.util.rdfRest, this.util.rdfNil, this.util.getDefaultGraph()));
                }
                else {
                    this.parsingContext.getUnidentifiedValueBufferSafe(listPointer.listRootDepth)
                        .push({ predicate: listPointer.initialPredicate, object: this.util.rdfNil, reverse: false });
                }
                this.parsingContext.listPointerStack.splice(this.lastDepth, 1);
            }
            // Flush the buffer for lastDepth
            await this.flushBuffer(this.lastDepth, keys);
        }
        const key = await this.util.unaliasKeyword(keys[depth], keys, depth);
        const parentKey = await this.util.unaliasKeywordParent(keys, depth);
        this.parsingContext.emittedStack[depth] = true;
        let handleKey = true;
        // Keywords inside @reverse is not allowed
        if (Util_1.Util.isKeyword(key) && parentKey === '@reverse') {
            this.emit('error', new Error(`Found the @id '${value}' inside an @reverse property`));
        }
        // Skip further processing if one of the parent nodes are invalid.
        // We use the validationStack to reuse validation results that were produced before with common key stacks.
        let inProperty = false;
        if (this.parsingContext.validationStack.length > 1) {
            inProperty = this.parsingContext.validationStack[this.parsingContext.validationStack.length - 1].property;
        }
        for (let i = Math.max(1, this.parsingContext.validationStack.length - 1); i < keys.length - 1; i++) {
            const validationResult = this.parsingContext.validationStack[i]
                || (this.parsingContext.validationStack[i] = await this.validateKey(keys.slice(0, i + 1), i, inProperty));
            if (!validationResult.valid) {
                this.parsingContext.emittedStack[depth] = false;
                handleKey = false;
                break;
            }
            else if (!inProperty && validationResult.property) {
                inProperty = true;
            }
        }
        // Skip further processing if this node is part of a literal
        if (this.util.isLiteral(depth)) {
            handleKey = false;
        }
        // Get handler
        if (handleKey) {
            for (const entryHandler of JsonLdParser.ENTRY_HANDLERS) {
                const testResult = await entryHandler.test(this.parsingContext, this.util, key, keys, depth);
                if (testResult) {
                    // Pass processing over to the handler
                    await entryHandler.handle(this.parsingContext, this.util, key, keys, value, depth, testResult);
                    break;
                }
            }
            // Flag that this depth is processed
            this.parsingContext.processingStack[depth] = true;
        }
        // Validate value indexes on the root.
        if (depth === 0 && Array.isArray(value)) {
            await this.util.validateValueIndexes(value);
        }
        // When we go up the stack, flush the old stack
        if (depth < this.lastDepth) {
            // Reset our stack
            this.parsingContext.processingStack.splice(this.lastDepth, 1);
            this.parsingContext.emittedStack.splice(this.lastDepth, 1);
            this.parsingContext.idStack.splice(this.lastDepth, 1);
            this.parsingContext.graphStack.splice(this.lastDepth + 1, 1);
            this.parsingContext.literalStack.splice(this.lastDepth, 1);
            this.parsingContext.validationStack.splice(this.lastDepth - 1, 2);
        }
        this.lastDepth = depth;
        // Clear the keyword cache at this depth, and everything underneath.
        this.parsingContext.unaliasedKeywordCacheStack.splice(depth - 1);
    }
    /**
     * Check if at least one {@link IEntryHandler} validates the entry to true.
     * @param {any[]} keys A stack of keys.
     * @param {number} depth A depth.
     * @param {boolean} inProperty If the current depth is part of a valid property node.
     * @return {Promise<{ valid: boolean, property: boolean }>} A promise resolving to true or false.
     */
    async validateKey(keys, depth, inProperty) {
        for (const entryHandler of JsonLdParser.ENTRY_HANDLERS) {
            if (await entryHandler.validate(this.parsingContext, this.util, keys, depth, inProperty)) {
                return { valid: true, property: inProperty || entryHandler.isPropertyHandler() };
            }
        }
        return { valid: false, property: false };
    }
    /**
     * Attach all required listeners to the JSON parser.
     *
     * This should only be called once.
     */
    attachJsonParserListeners() {
        // Listen to json parser events
        this.jsonParser.onValue = (value) => {
            const depth = this.jsonParser.stack.length;
            const keys = (new Array(depth + 1).fill(0)).map((v, i) => {
                return i === depth ? this.jsonParser.key : this.jsonParser.stack[i].key;
            });
            if (!this.isParsingContextInner(depth)) { // Don't parse inner nodes inside @context
                const valueJobCb = () => this.newOnValueJob(keys, value, depth);
                if (this.parsingContext.allowOutOfOrderContext
                    && !this.parsingContext.contextTree.getContext(keys.slice(0, -1))) {
                    // If an out-of-order context is allowed,
                    // we have to buffer everything.
                    // We store jobs for @context's separately,
                    // because at the end, we have to process them first.
                    if (keys[depth] === '@context') {
                        let jobs = this.contextJobs[depth];
                        if (!jobs) {
                            jobs = this.contextJobs[depth] = [];
                        }
                        jobs.push(valueJobCb);
                    }
                    else {
                        this.contextAwaitingJobs.push(valueJobCb);
                    }
                }
                else {
                    // Make sure that our value jobs are chained synchronously
                    this.lastOnValueJob = this.lastOnValueJob.then(valueJobCb);
                }
                // Execute all buffered jobs on deeper levels
                if (this.parsingContext.allowOutOfOrderContext && depth === 0) {
                    this.lastOnValueJob = this.lastOnValueJob
                        .then(() => this.executeBufferedJobs());
                }
            }
        };
        this.jsonParser.onError = (error) => {
            this.emit('error', error);
        };
    }
    /**
     * Check if the parser is currently parsing an element that is part of an @context entry.
     * @param {number} depth A depth.
     * @return {boolean} A boolean.
     */
    isParsingContextInner(depth) {
        for (let i = depth; i > 0; i--) {
            if (this.jsonParser.stack[i - 1].key === '@context') {
                return true;
            }
        }
        return false;
    }
    /**
     * Execute all buffered jobs.
     * @return {Promise<void>} A promise resolving if all jobs are finished.
     */
    async executeBufferedJobs() {
        // Handle context jobs
        for (const jobs of this.contextJobs) {
            if (jobs) {
                for (const job of jobs) {
                    await job();
                }
            }
        }
        // Handle non-context jobs
        for (const job of this.contextAwaitingJobs) {
            await job();
        }
    }
    /**
     * Flush buffers for the given depth.
     *
     * This should be called after the last entry at a given depth was processed.
     *
     * @param {number} depth A depth.
     * @param {any[]} keys A stack of keys.
     * @return {Promise<void>} A promise resolving if flushing is done.
     */
    async flushBuffer(depth, keys) {
        let subject = this.parsingContext.idStack[depth];
        if (subject === undefined) {
            subject = this.parsingContext.idStack[depth] = this.util.dataFactory.blankNode();
        }
        // Flush values at this level
        const valueBuffer = this.parsingContext.unidentifiedValuesBuffer[depth];
        if (valueBuffer) {
            if (subject) {
                const depthOffsetGraph = await this.util.getDepthOffsetGraph(depth, keys);
                const graph = this.parsingContext.graphStack[depth] || depthOffsetGraph >= 0
                    ? this.parsingContext.idStack[depth - depthOffsetGraph - 1] : this.util.getDefaultGraph();
                if (graph) {
                    // Flush values to stream if the graph @id is known
                    this.parsingContext.emittedStack[depth] = true;
                    for (const bufferedValue of valueBuffer) {
                        if (bufferedValue.reverse) {
                            this.parsingContext.emitQuad(depth, this.util.dataFactory.quad(bufferedValue.object, bufferedValue.predicate, subject, graph));
                        }
                        else {
                            this.parsingContext.emitQuad(depth, this.util.dataFactory.quad(subject, bufferedValue.predicate, bufferedValue.object, graph));
                        }
                    }
                }
                else {
                    // Place the values in the graphs buffer if the graph @id is not yet known
                    const subGraphBuffer = this.parsingContext.getUnidentifiedGraphBufferSafe(depth - await this.util.getDepthOffsetGraph(depth, keys) - 1);
                    for (const bufferedValue of valueBuffer) {
                        if (bufferedValue.reverse) {
                            subGraphBuffer.push({
                                object: subject,
                                predicate: bufferedValue.predicate,
                                subject: bufferedValue.object,
                            });
                        }
                        else {
                            subGraphBuffer.push({
                                object: bufferedValue.object,
                                predicate: bufferedValue.predicate,
                                subject,
                            });
                        }
                    }
                }
            }
            this.parsingContext.unidentifiedValuesBuffer.splice(depth, 1);
            this.parsingContext.literalStack.splice(depth, 1);
        }
        // Flush graphs at this level
        const graphBuffer = this.parsingContext.unidentifiedGraphsBuffer[depth];
        if (graphBuffer) {
            if (subject) {
                // A @graph statement at the root without @id relates to the default graph,
                // unless there are top-level properties,
                // others relate to blank nodes.
                const graph = depth === 1 && subject.termType === 'BlankNode'
                    && !this.parsingContext.topLevelProperties ? this.util.getDefaultGraph() : subject;
                this.parsingContext.emittedStack[depth] = true;
                for (const bufferedValue of graphBuffer) {
                    this.parsingContext.emitQuad(depth, this.util.dataFactory.quad(bufferedValue.subject, bufferedValue.predicate, bufferedValue.object, graph));
                }
            }
            this.parsingContext.unidentifiedGraphsBuffer.splice(depth, 1);
        }
    }
}
JsonLdParser.DEFAULT_PROCESSING_MODE = '1.0';
JsonLdParser.ENTRY_HANDLERS = [
    new EntryHandlerArrayValue_1.EntryHandlerArrayValue(),
    new EntryHandlerKeywordContext_1.EntryHandlerKeywordContext(),
    new EntryHandlerKeywordId_1.EntryHandlerKeywordId(),
    new EntryHandlerKeywordGraph_1.EntryHandlerKeywordGraph(),
    new EntryHandlerKeywordType_1.EntryHandlerKeywordType(),
    new EntryHandlerKeywordValue_1.EntryHandlerKeywordValue(),
    new EntryHandlerKeywordUnknownFallback_1.EntryHandlerKeywordUnknownFallback(),
    new EntryHandlerContainer_1.EntryHandlerContainer(),
    new EntryHandlerPredicate_1.EntryHandlerPredicate(),
    new EntryHandlerInvalidFallback_1.EntryHandlerInvalidFallback(),
];
exports.JsonLdParser = JsonLdParser;
//# sourceMappingURL=JsonLdParser.js.map