import { ContextParser, IJsonLdContextNormalized } from "jsonld-context-parser";
import * as RDF from "rdf-js";
import { ContextTree } from "./ContextTree";
import { IJsonLdParserOptions, JsonLdParser } from "./JsonLdParser";
import { JsonLdContext } from "jsonld-context-parser/lib/JsonLdContext";
/**
 * Data holder for parsing information.
 */
export declare class ParsingContext {
    readonly contextParser: ContextParser;
    readonly allowOutOfOrderContext: boolean;
    readonly baseIRI: string;
    readonly produceGeneralizedRdf: boolean;
    readonly allowSubjectList: boolean;
    readonly processingMode: string;
    readonly errorOnInvalidProperties: boolean;
    readonly validateValueIndexes: boolean;
    readonly rootContext: Promise<IJsonLdContextNormalized>;
    readonly defaultGraph?: RDF.Term;
    readonly processingStack: boolean[];
    readonly emittedStack: boolean[];
    readonly idStack: RDF.Term[];
    readonly graphStack: boolean[];
    readonly listPointerStack: {
        term: RDF.Term;
        initialPredicate: RDF.Term;
        listRootDepth: number;
    }[];
    readonly contextTree: ContextTree;
    readonly literalStack: boolean[];
    readonly validationStack: {
        valid: boolean;
        property: boolean;
    }[];
    readonly unaliasedKeywordCacheStack: any[];
    readonly unidentifiedValuesBuffer: {
        predicate: RDF.Term;
        object: RDF.Term;
        reverse: boolean;
    }[][];
    readonly unidentifiedGraphsBuffer: {
        subject: RDF.Term;
        predicate: RDF.Term;
        object: RDF.Term;
    }[][];
    topLevelProperties: boolean;
    private readonly parser;
    constructor(options: IParsingContextOptions);
    /**
     * Check if the given context is valid.
     * If not, an error will be thrown.
     * @param {IJsonLdContextNormalized} context A context.
     */
    validateContext(context: IJsonLdContextNormalized): void;
    /**
     * Get the context at the given path.
     * @param {keys} keys The path of keys to get the context at.
     * @param {number} offset The path offset, defaults to 1.
     * @return {Promise<IJsonLdContextNormalized>} A promise resolving to a context.
     */
    getContext(keys: string[], offset?: number): Promise<IJsonLdContextNormalized>;
    /**
     * Start a new job for parsing the given value.
     * @param {any[]} keys The stack of keys.
     * @param value The value to parse.
     * @param {number} depth The depth to parse at.
     * @return {Promise<void>} A promise resolving when the job is done.
     */
    newOnValueJob(keys: any[], value: any, depth: number): Promise<void>;
    /**
     * Emit the given quad into the output stream.
     * @param {number} depth The depth the quad was generated at.
     * @param {Quad} quad A quad to emit.
     */
    emitQuad(depth: number, quad: RDF.BaseQuad): void;
    /**
     * Emit the given error into the output stream.
     * @param {Error} error An error to emit.
     */
    emitError(error: Error): void;
    /**
     * Emit the given context into the output stream under the 'context' event.
     * @param {JsonLdContext} context A context to emit.
     */
    emitContext(context: JsonLdContext): void;
    /**
     * Safely get or create the depth value of {@link ParsingContext.unidentifiedValuesBuffer}.
     * @param {number} depth A depth.
     * @return {{predicate: Term; object: Term; reverse: boolean}[]} An element of
     *                                                               {@link ParsingContext.unidentifiedValuesBuffer}.
     */
    getUnidentifiedValueBufferSafe(depth: number): {
        predicate: RDF.Term;
        object: RDF.Term;
        reverse: boolean;
    }[];
    /**
     * Safely get or create the depth value of {@link ParsingContext.unidentifiedGraphsBuffer}.
     * @param {number} depth A depth.
     * @return {{predicate: Term; object: Term; reverse: boolean}[]} An element of
     *                                                               {@link ParsingContext.unidentifiedGraphsBuffer}.
     */
    getUnidentifiedGraphBufferSafe(depth: number): {
        subject: RDF.Term;
        predicate: RDF.Term;
        object: RDF.Term;
    }[];
}
/**
 * Constructor arguments for {@link ParsingContext}
 */
export interface IParsingContextOptions extends IJsonLdParserOptions {
    /**
     * The parser instance.
     */
    parser: JsonLdParser;
}
