"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const jsonld_context_parser_1 = require("jsonld-context-parser");
const ContextTree_1 = require("./ContextTree");
const JsonLdParser_1 = require("./JsonLdParser");
/**
 * Data holder for parsing information.
 */
class ParsingContext {
    constructor(options) {
        // Initialize settings
        this.contextParser = new jsonld_context_parser_1.ContextParser({ documentLoader: options.documentLoader });
        this.allowOutOfOrderContext = options.allowOutOfOrderContext;
        this.baseIRI = options.baseIRI;
        this.produceGeneralizedRdf = options.produceGeneralizedRdf;
        this.allowSubjectList = options.allowSubjectList;
        this.processingMode = options.processingMode || JsonLdParser_1.JsonLdParser.DEFAULT_PROCESSING_MODE;
        this.errorOnInvalidProperties = options.errorOnInvalidIris;
        this.validateValueIndexes = options.validateValueIndexes;
        this.defaultGraph = options.defaultGraph;
        // Initialize stacks
        this.processingStack = [];
        this.emittedStack = [];
        this.idStack = [];
        this.graphStack = [];
        this.listPointerStack = [];
        this.contextTree = new ContextTree_1.ContextTree();
        this.literalStack = [];
        this.validationStack = [];
        this.unaliasedKeywordCacheStack = [];
        this.unidentifiedValuesBuffer = [];
        this.unidentifiedGraphsBuffer = [];
        this.parser = options.parser;
        if (options.context) {
            this.rootContext = this.contextParser.parse(options.context, { baseIri: options.baseIRI });
            this.rootContext.then((context) => this.validateContext(context));
        }
        else {
            this.rootContext = Promise.resolve(this.baseIRI ? { '@base': this.baseIRI } : {});
        }
        this.topLevelProperties = false;
    }
    /**
     * Check if the given context is valid.
     * If not, an error will be thrown.
     * @param {IJsonLdContextNormalized} context A context.
     */
    validateContext(context) {
        const activeVersion = context['@version'];
        if (activeVersion && parseFloat(activeVersion) > parseFloat(this.processingMode)) {
            throw new Error(`Unsupported JSON-LD processing mode: ${activeVersion}`);
        }
    }
    /**
     * Get the context at the given path.
     * @param {keys} keys The path of keys to get the context at.
     * @param {number} offset The path offset, defaults to 1.
     * @return {Promise<IJsonLdContextNormalized>} A promise resolving to a context.
     */
    getContext(keys, offset = 1) {
        if (offset) {
            keys = keys.slice(0, -offset);
        }
        return this.contextTree.getContext(keys) || this.rootContext;
    }
    /**
     * Start a new job for parsing the given value.
     * @param {any[]} keys The stack of keys.
     * @param value The value to parse.
     * @param {number} depth The depth to parse at.
     * @return {Promise<void>} A promise resolving when the job is done.
     */
    async newOnValueJob(keys, value, depth) {
        await this.parser.newOnValueJob(keys, value, depth);
    }
    /**
     * Emit the given quad into the output stream.
     * @param {number} depth The depth the quad was generated at.
     * @param {Quad} quad A quad to emit.
     */
    emitQuad(depth, quad) {
        if (depth === 1) {
            this.topLevelProperties = true;
        }
        this.parser.push(quad);
    }
    /**
     * Emit the given error into the output stream.
     * @param {Error} error An error to emit.
     */
    emitError(error) {
        this.parser.emit('error', error);
    }
    /**
     * Emit the given context into the output stream under the 'context' event.
     * @param {JsonLdContext} context A context to emit.
     */
    emitContext(context) {
        this.parser.emit('context', context);
    }
    /**
     * Safely get or create the depth value of {@link ParsingContext.unidentifiedValuesBuffer}.
     * @param {number} depth A depth.
     * @return {{predicate: Term; object: Term; reverse: boolean}[]} An element of
     *                                                               {@link ParsingContext.unidentifiedValuesBuffer}.
     */
    getUnidentifiedValueBufferSafe(depth) {
        let buffer = this.unidentifiedValuesBuffer[depth];
        if (!buffer) {
            buffer = [];
            this.unidentifiedValuesBuffer[depth] = buffer;
        }
        return buffer;
    }
    /**
     * Safely get or create the depth value of {@link ParsingContext.unidentifiedGraphsBuffer}.
     * @param {number} depth A depth.
     * @return {{predicate: Term; object: Term; reverse: boolean}[]} An element of
     *                                                               {@link ParsingContext.unidentifiedGraphsBuffer}.
     */
    getUnidentifiedGraphBufferSafe(depth) {
        let buffer = this.unidentifiedGraphsBuffer[depth];
        if (!buffer) {
            buffer = [];
            this.unidentifiedGraphsBuffer[depth] = buffer;
        }
        return buffer;
    }
}
exports.ParsingContext = ParsingContext;
//# sourceMappingURL=ParsingContext.js.map