import { IJsonLdContextNormalized } from "jsonld-context-parser";
import * as RDF from "rdf-js";
import { ParsingContext } from "./ParsingContext";
/**
 * Utility functions and methods.
 */
export declare class Util {
    static readonly XSD: string;
    static readonly XSD_BOOLEAN: string;
    static readonly XSD_INTEGER: string;
    static readonly XSD_DOUBLE: string;
    static readonly RDF: string;
    readonly dataFactory: RDF.DataFactory;
    readonly rdfFirst: RDF.NamedNode;
    readonly rdfRest: RDF.NamedNode;
    readonly rdfNil: RDF.NamedNode;
    readonly rdfType: RDF.NamedNode;
    private readonly parsingContext;
    constructor(options: {
        parsingContext: ParsingContext;
        dataFactory?: RDF.DataFactory;
    });
    /**
     * Helper function to get the value of a context entry,
     * or fallback to a certain value.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param {string} contextKey A pre-defined JSON-LD key in context entries.
     * @param {string} key A context entry key.
     * @param {string} fallback A fallback value for when the given contextKey
     *                          could not be found in the value with the given key.
     * @return {string} The value of the given contextKey in the entry behind key in the given context,
     *                  or the given fallback value.
     */
    static getContextValue(context: IJsonLdContextNormalized, contextKey: string, key: string, fallback: string): string;
    /**
     * Get the container type of the given key in the context.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param {string} key A context entry key.
     * @return {string} The container type.
     */
    static getContextValueContainer(context: IJsonLdContextNormalized, key: string): string;
    /**
     * Get the node type of the given key in the context.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param {string} key A context entry key.
     * @return {string} The node type.
     */
    static getContextValueType(context: IJsonLdContextNormalized, key: string): string;
    /**
     * Get the node type of the given key in the context.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param {string} key A context entry key.
     * @return {string} The node type.
     */
    static getContextValueLanguage(context: IJsonLdContextNormalized, key: string): string;
    /**
     * Check if the given key in the context is a reversed property.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param {string} key A context entry key.
     * @return {boolean} If the context value has a @reverse key.
     */
    static isContextValueReverse(context: IJsonLdContextNormalized, key: string): boolean;
    /**
     * Check if the given key refers to a reversed property.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param {string} key The property key.
     * @param {string} parentKey The parent key.
     * @return {boolean} If the property must be reversed.
     */
    static isPropertyReverse(context: IJsonLdContextNormalized, key: string, parentKey: string): boolean;
    /**
     * Check if the given key is a keyword.
     * @param {string} key A key, can be falsy.
     * @return {boolean} If the given key starts with an @.
     */
    static isKeyword(key: any): boolean;
    /**
     * Check if the given IRI is valid.
     * @param {string} iri A potential IRI.
     * @return {boolean} If the given IRI is valid.
     */
    static isValidIri(iri: string): boolean;
    /**
     * Make sure that @id-@index pairs are equal over all array values.
     * Reject otherwise.
     * @param {any[]} value An array value.
     * @return {Promise<void>} A promise rejecting if conflicts are present.
     */
    validateValueIndexes(value: any[]): Promise<void>;
    /**
     * Convert a given JSON value to an RDF term.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param {string} key The current JSON key.
     * @param value A JSON value.
     * @param {number} depth The depth the value is at.
     * @param {string[]} keys The path of keys.
     * @return {RDF.Term} An RDF term.
     */
    valueToTerm(context: IJsonLdContextNormalized, key: string, value: any, depth: number, keys: string[]): Promise<RDF.Term>;
    /**
     * Convert a given JSON key to an RDF predicate term,
     * based on @vocab.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param key A JSON key.
     * @return {RDF.NamedNode} An RDF named node.
     */
    predicateToTerm(context: IJsonLdContextNormalized, key: string): RDF.Term;
    /**
     * Convert a given JSON key to an RDF resource term or blank node,
     * based on @base.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param key A JSON key.
     * @return {RDF.NamedNode} An RDF named node or null.
     */
    resourceToTerm(context: IJsonLdContextNormalized, key: string): RDF.Term;
    /**
     * Convert a given JSON key to an RDF resource term.
     * It will do this based on the @vocab,
     * and fallback to @base.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param key A JSON key.
     * @return {RDF.NamedNode} An RDF named node or null.
     */
    createVocabOrBaseTerm(context: IJsonLdContextNormalized, key: string): RDF.Term;
    /**
     * Ensure that the given value becomes a string.
     * @param {string | number} value A string or number.
     * @param {NamedNode} datatype The intended datatype.
     * @return {string} The returned string.
     */
    intToString(value: string | number, datatype: RDF.NamedNode): string;
    /**
     * Convert a given JSON string value to an RDF term.
     * @param {IJsonLdContextNormalized} context A JSON-LD context.
     * @param {string} key The current JSON key.
     * @param {string} value A JSON value.
     * @param {NamedNode} defaultDatatype The default datatype for the given value.
     * @return {RDF.Term} An RDF term or null.
     */
    stringValueToTerm(context: IJsonLdContextNormalized, key: string, value: string | number, defaultDatatype: RDF.NamedNode): RDF.Term;
    /**
     * If the key is not a keyword, try to check if it is an alias for a keyword,
     * and if so, un-alias it.
     * @param {string} key A key, can be falsy.
     * @param {string[]} keys The path of keys.
     * @param {number} depth The depth to
     * @param {boolean} disableCache If the cache should be disabled
     * @return {Promise<string>} A promise resolving to the key itself, or another key.
     */
    unaliasKeyword(key: any, keys: string[], depth: number, disableCache?: boolean): Promise<any>;
    /**
     * Unalias the keyword of the parent.
     * This adds a safety check if no parent exist.
     * @param {any[]} keys A stack of keys.
     * @param {number} depth The current depth.
     * @return {Promise<any>} A promise resolving to the parent key, or another key.
     */
    unaliasKeywordParent(keys: any[], depth: number): Promise<any>;
    /**
     * Un-alias all keywords in the given hash.
     * @param {{[p: string]: any}} hash A hash object.
     * @param {string[]} keys The path of keys.
     * @param {number} depth The depth.
     * @return {Promise<{[p: string]: any}>} A promise resolving to the new hash.
     */
    unaliasKeywords(hash: {
        [id: string]: any;
    }, keys: string[], depth: number): Promise<{
        [id: string]: any;
    }>;
    /**
     * Check if we are processing a literal at the given depth.
     * This will also check higher levels,
     * because if a parent is a literal,
     * then the deeper levels are definitely a literal as well.
     * @param {number} depth The depth.
     * @return {boolean} If we are processing a literal.
     */
    isLiteral(depth: number): boolean;
    /**
     * Check how many parents should be skipped for checking the @graph for the given node.
     *
     * @param {number} depth The depth of the node.
     * @param {any[]} keys An array of keys.
     * @return {number} The graph depth offset.
     */
    getDepthOffsetGraph(depth: number, keys: any[]): Promise<number>;
    /**
     * Check if the given subject is of a valid type.
     * This should be called when applying @reverse'd properties.
     * @param {Term} subject A subject.
     */
    validateReverseSubject(subject: RDF.Term): void;
    /**
     * Get the default graph.
     * @return {Term} An RDF term.
     */
    getDefaultGraph(): RDF.Term;
}
