"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Util_1 = require("../Util");
/**
 * Handles values that are part of an array.
 */
class EntryHandlerArrayValue {
    isPropertyHandler() {
        return false;
    }
    async validate(parsingContext, util, keys, depth, inProperty) {
        return this.test(parsingContext, util, null, keys, depth);
    }
    async test(parsingContext, util, key, keys, depth) {
        return typeof keys[depth] === 'number';
    }
    async handle(parsingContext, util, key, keys, value, depth) {
        const parentKey = await util.unaliasKeywordParent(keys, depth);
        // Check if we have an anonymous list
        if (parentKey === '@list') {
            // Our value is part of an array
            // Determine the list root key
            let listRootKey = null;
            let listRootDepth;
            for (let i = depth - 2; i > 0; i--) {
                const keyOption = keys[i];
                if (typeof keyOption === 'string') {
                    listRootDepth = i;
                    listRootKey = keyOption;
                    break;
                }
            }
            // Throw an error if we encounter a nested list
            if (listRootKey === '@list' ||
                (listRootDepth !== depth - 2 && typeof keys[depth - 2] === 'number'
                    && Util_1.Util.getContextValueContainer(await parsingContext
                        .getContext(keys, listRootDepth - depth), listRootKey) === '@list')) {
                throw new Error(`Lists of lists are not supported: '${listRootKey}'`);
            }
            const object = await util.valueToTerm(await parsingContext.getContext(keys), listRootKey, value, depth, keys);
            if (listRootKey !== null) {
                await this.handleListElement(parsingContext, util, object, depth, keys.slice(0, listRootDepth), listRootDepth, listRootKey, keys);
            }
        }
        else if (parentKey === '@set') {
            // Our value is part of a set, so we just add it to the parent-parent
            await parsingContext.newOnValueJob(keys.slice(0, -2), value, depth - 2);
        }
        else if (parentKey !== undefined && parentKey !== '@type') {
            // Buffer our value using the parent key as predicate
            // Check if the predicate is marked as an @list in the context
            const parentContext = await parsingContext.getContext(keys.slice(0, -1));
            if (Util_1.Util.getContextValueContainer(parentContext, parentKey) === '@list') {
                // Our value is part of an array
                const object = await util.valueToTerm(await parsingContext.getContext(keys), parentKey, value, depth, keys);
                await this.handleListElement(parsingContext, util, object, depth, keys.slice(0, -1), depth - 1, parentKey, keys);
            }
            else {
                // Copy the id stack value up one level so that the next job can access the id.
                if (parsingContext.idStack[depth + 1]) {
                    parsingContext.idStack[depth] = parsingContext.idStack[depth + 1];
                    parsingContext.emittedStack[depth] = true;
                }
                // Execute the job one level higher
                await parsingContext.newOnValueJob(keys.slice(0, -1), value, depth - 1);
            }
        }
    }
    async handleListElement(parsingContext, util, value, depth, listRootKeys, listRootDepth, listRootKey, keys) {
        // Buffer our value as an RDF list using the listRootKey as predicate
        let listPointer = parsingContext.listPointerStack[depth];
        if (value) {
            if (!listPointer || !listPointer.term) {
                const linkTerm = util.dataFactory.blankNode();
                const listRootContext = await parsingContext.getContext(listRootKeys);
                const predicate = await util.predicateToTerm(listRootContext, listRootKey);
                const reverse = Util_1.Util.isPropertyReverse(listRootContext, listRootKey, keys[listRootDepth - 1]);
                // Lists are not allowed in @reverse'd properties
                if (reverse && !parsingContext.allowSubjectList) {
                    throw new Error(`Found illegal list value in subject position at ${listRootKey}`);
                }
                parsingContext.getUnidentifiedValueBufferSafe(listRootDepth)
                    .push({ predicate, object: linkTerm, reverse });
                listPointer = { term: linkTerm, initialPredicate: null, listRootDepth };
            }
            else {
                // rdf:rest links are always emitted before the next element,
                // as the blank node identifier is only created at that point.
                // Because of this reason, the final rdf:nil is emitted when the stack depth is decreased.
                const newLinkTerm = util.dataFactory.blankNode();
                parsingContext.emitQuad(depth, util.dataFactory.quad(listPointer.term, util.rdfRest, newLinkTerm, util.getDefaultGraph()));
                // Update the list pointer for the next element
                listPointer.term = newLinkTerm;
            }
            // Emit a list element for the current value
            parsingContext.emitQuad(depth, util.dataFactory.quad(listPointer.term, util.rdfFirst, value, util.getDefaultGraph()));
        }
        else {
            // A falsy list element if found.
            // Just enable the list flag for this depth if it has not been set before.
            if (!listPointer) {
                const predicate = await util.predicateToTerm(await parsingContext.getContext(listRootKeys), listRootKey);
                listPointer = { term: null, initialPredicate: predicate, listRootDepth };
            }
        }
        parsingContext.listPointerStack[depth] = listPointer;
    }
}
exports.EntryHandlerArrayValue = EntryHandlerArrayValue;
//# sourceMappingURL=EntryHandlerArrayValue.js.map