"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Util_1 = require("../Util");
/**
 * Interprets keys as predicates.
 * The most common case in JSON-LD processing.
 */
class EntryHandlerPredicate {
    /**
     * Handle the given predicate-object by either emitting it,
     * or by placing it in the appropriate stack for later emission when no @graph and/or @id has been defined.
     * @param {ParsingContext} parsingContext A parsing context.
     * @param {Util} util A utility instance.
     * @param {any[]} keys A stack of keys.
     * @param {number} depth The current depth.
     * @param parentKey The parent key.
     * @param {Term} predicate The predicate.
     * @param {Term} object The object.
     * @param {boolean} reverse If the property is reversed.
     * @return {Promise<void>} A promise resolving when handling is done.
     */
    static async handlePredicateObject(parsingContext, util, keys, depth, parentKey, predicate, object, reverse) {
        const depthProperties = depth - (parentKey === '@reverse' ? 1 : 0);
        const depthOffsetGraph = await util.getDepthOffsetGraph(depth, keys);
        const depthPropertiesGraph = depth - depthOffsetGraph;
        if (parsingContext.idStack[depthProperties]) {
            // Emit directly if the @id was already defined
            const subject = parsingContext.idStack[depthProperties];
            // Check if we're in a @graph context
            const atGraph = depthOffsetGraph >= 0;
            if (atGraph) {
                const graph = parsingContext.idStack[depthPropertiesGraph - 1];
                if (graph) {
                    // Emit our quad if graph @id is known
                    if (reverse) {
                        util.validateReverseSubject(object);
                        parsingContext.emitQuad(depth, util.dataFactory.quad(object, predicate, subject, graph));
                    }
                    else {
                        parsingContext.emitQuad(depth, util.dataFactory.quad(subject, predicate, object, graph));
                    }
                }
                else {
                    // Buffer our triple if graph @id is not known yet.
                    if (reverse) {
                        util.validateReverseSubject(object);
                        parsingContext.getUnidentifiedGraphBufferSafe(depthPropertiesGraph - 1).push({ subject: object, predicate, object: subject });
                    }
                    else {
                        parsingContext.getUnidentifiedGraphBufferSafe(depthPropertiesGraph - 1)
                            .push({ subject, predicate, object });
                    }
                }
            }
            else {
                // Emit if no @graph was applicable
                if (reverse) {
                    util.validateReverseSubject(object);
                    parsingContext.emitQuad(depth, util.dataFactory.quad(object, predicate, subject, util.getDefaultGraph()));
                }
                else {
                    parsingContext.emitQuad(depth, util.dataFactory.quad(subject, predicate, object, util.getDefaultGraph()));
                }
            }
        }
        else {
            // Buffer until our @id becomes known, or we go up the stack
            if (reverse) {
                util.validateReverseSubject(object);
            }
            parsingContext.getUnidentifiedValueBufferSafe(depthProperties).push({ predicate, object, reverse });
        }
    }
    isPropertyHandler() {
        return true;
    }
    async validate(parsingContext, util, keys, depth, inProperty) {
        return keys[depth] && !!await util.predicateToTerm(await parsingContext.getContext(keys), keys[depth]);
    }
    async test(parsingContext, util, key, keys, depth) {
        return keys[depth];
    }
    async handle(parsingContext, util, key, keys, value, depth, testResult) {
        const keyOriginal = keys[depth];
        const parentKey = await util.unaliasKeywordParent(keys, depth);
        const context = await parsingContext.getContext(keys);
        const predicate = await util.predicateToTerm(context, key);
        if (predicate) {
            const objectContext = await parsingContext.getContext(keys, 0);
            let object = await util.valueToTerm(objectContext, key, value, depth, keys);
            if (object) {
                const reverse = Util_1.Util.isPropertyReverse(context, keyOriginal, parentKey);
                // Special case if our term was defined as an @list, but does not occur in an array,
                // In that case we just emit it as an RDF list with a single element.
                const listValueContainer = Util_1.Util.getContextValueContainer(context, key) === '@list';
                if (listValueContainer || value['@list']) {
                    if ((listValueContainer || (value['@list'] && !Array.isArray(value['@list']))) && object !== util.rdfNil) {
                        const listPointer = util.dataFactory.blankNode();
                        parsingContext.emitQuad(depth, util.dataFactory.quad(listPointer, util.rdfRest, util.rdfNil, util.getDefaultGraph()));
                        parsingContext.emitQuad(depth, util.dataFactory.quad(listPointer, util.rdfFirst, object, util.getDefaultGraph()));
                        object = listPointer;
                    }
                    // Lists are not allowed in @reverse'd properties
                    if (reverse && !parsingContext.allowSubjectList) {
                        throw new Error(`Found illegal list value in subject position at ${key}`);
                    }
                }
                await EntryHandlerPredicate.handlePredicateObject(parsingContext, util, keys, depth, parentKey, predicate, object, reverse);
            }
            else {
                // An invalid value was encountered, so we ignore it higher in the stack.
                parsingContext.emittedStack[depth] = false;
            }
        }
    }
}
exports.EntryHandlerPredicate = EntryHandlerPredicate;
//# sourceMappingURL=EntryHandlerPredicate.js.map