"use strict";

var JsSIP_C = require('./Constants');

var SIPMessage = require('./SIPMessage');

var Utils = require('./Utils');

var debug = require('debug')('JsSIP:sanityCheck'); // Checks for requests and responses.


var all = [minimumHeaders]; // Checks for requests.

var requests = [rfc3261_8_2_2_1, rfc3261_16_3_4, rfc3261_18_3_request, rfc3261_8_2_2_2]; // Checks for responses.

var responses = [rfc3261_8_1_3_3, rfc3261_18_3_response]; // local variables.

var message;
var ua;
var transport;

module.exports = function (m, u, t) {
  message = m;
  ua = u;
  transport = t;

  for (var _i = 0; _i < all.length; _i++) {
    var _check2 = all[_i];

    if (_check2() === false) {
      return false;
    }
  }

  if (message instanceof SIPMessage.IncomingRequest) {
    for (var _i2 = 0; _i2 < requests.length; _i2++) {
      var check = requests[_i2];

      if (check() === false) {
        return false;
      }
    }
  } else if (message instanceof SIPMessage.IncomingResponse) {
    for (var _i3 = 0; _i3 < responses.length; _i3++) {
      var _check = responses[_i3];

      if (_check() === false) {
        return false;
      }
    }
  } // Everything is OK.


  return true;
};
/*
 * Sanity Check for incoming Messages
 *
 * Requests:
 *  - _rfc3261_8_2_2_1_ Receive a Request with a non supported URI scheme
 *  - _rfc3261_16_3_4_ Receive a Request already sent by us
 *   Does not look at via sent-by but at jssip_id, which is inserted as
 *   a prefix in all initial requests generated by the ua
 *  - _rfc3261_18_3_request_ Body Content-Length
 *  - _rfc3261_8_2_2_2_ Merged Requests
 *
 * Responses:
 *  - _rfc3261_8_1_3_3_ Multiple Via headers
 *  - _rfc3261_18_3_response_ Body Content-Length
 *
 * All:
 *  - Minimum headers in a SIP message
 */
// Sanity Check functions for requests.


function rfc3261_8_2_2_1() {
  if (message.s('to').uri.scheme !== 'sip') {
    reply(416);
    return false;
  }
}

function rfc3261_16_3_4() {
  if (!message.to_tag) {
    if (message.call_id.substr(0, 5) === ua.configuration.jssip_id) {
      reply(482);
      return false;
    }
  }
}

function rfc3261_18_3_request() {
  var len = Utils.str_utf8_length(message.body);
  var contentLength = message.getHeader('content-length');

  if (len < contentLength) {
    reply(400);
    return false;
  }
}

function rfc3261_8_2_2_2() {
  var fromTag = message.from_tag;
  var call_id = message.call_id;
  var cseq = message.cseq;
  var tr; // Accept any in-dialog request.

  if (message.to_tag) {
    return;
  } // INVITE request.


  if (message.method === JsSIP_C.INVITE) {
    // If the branch matches the key of any IST then assume it is a retransmission
    // and ignore the INVITE.
    // TODO: we should reply the last response.
    if (ua._transactions.ist[message.via_branch]) {
      return false;
    } // Otherwise check whether it is a merged request.
    else {
        for (var transaction in ua._transactions.ist) {
          if (Object.prototype.hasOwnProperty.call(ua._transactions.ist, transaction)) {
            tr = ua._transactions.ist[transaction];

            if (tr.request.from_tag === fromTag && tr.request.call_id === call_id && tr.request.cseq === cseq) {
              reply(482);
              return false;
            }
          }
        }
      }
  } // Non INVITE request.
  // If the branch matches the key of any NIST then assume it is a retransmission
  // and ignore the request.
  // TODO: we should reply the last response.
  else if (ua._transactions.nist[message.via_branch]) {
      return false;
    } // Otherwise check whether it is a merged request.
    else {
        for (var _transaction in ua._transactions.nist) {
          if (Object.prototype.hasOwnProperty.call(ua._transactions.nist, _transaction)) {
            tr = ua._transactions.nist[_transaction];

            if (tr.request.from_tag === fromTag && tr.request.call_id === call_id && tr.request.cseq === cseq) {
              reply(482);
              return false;
            }
          }
        }
      }
} // Sanity Check functions for responses.


function rfc3261_8_1_3_3() {
  if (message.getHeaders('via').length > 1) {
    debug('more than one Via header field present in the response, dropping the response');
    return false;
  }
}

function rfc3261_18_3_response() {
  var len = Utils.str_utf8_length(message.body),
      contentLength = message.getHeader('content-length');

  if (len < contentLength) {
    debug('message body length is lower than the value in Content-Length header field, dropping the response');
    return false;
  }
} // Sanity Check functions for requests and responses.


function minimumHeaders() {
  var mandatoryHeaders = ['from', 'to', 'call_id', 'cseq', 'via'];

  for (var _i4 = 0; _i4 < mandatoryHeaders.length; _i4++) {
    var header = mandatoryHeaders[_i4];

    if (!message.hasHeader(header)) {
      debug("missing mandatory header field : ".concat(header, ", dropping the response"));
      return false;
    }
  }
} // Reply.


function reply(status_code) {
  var vias = message.getHeaders('via');
  var to;
  var response = "SIP/2.0 ".concat(status_code, " ").concat(JsSIP_C.REASON_PHRASE[status_code], "\r\n");
  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = vias[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var via = _step.value;
      response += "Via: ".concat(via, "\r\n");
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return != null) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }

  to = message.getHeader('To');

  if (!message.to_tag) {
    to += ";tag=".concat(Utils.newTag());
  }

  response += "To: ".concat(to, "\r\n");
  response += "From: ".concat(message.getHeader('From'), "\r\n");
  response += "Call-ID: ".concat(message.call_id, "\r\n");
  response += "CSeq: ".concat(message.cseq, " ").concat(message.method, "\r\n");
  response += '\r\n';
  transport.send(response);
}