// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { nullTranslator } from '@jupyterlab/translation';
import { circleEmptyIcon, circleIcon, LabIcon, offlineBoltIcon, ReactWidget, refreshIcon, stopIcon, ToolbarButton, ToolbarButtonComponent, UseSignal } from '@jupyterlab/ui-components';
import { Widget } from '@lumino/widgets';
import * as React from 'react';
import { sessionContextDialogs } from '../sessioncontext';
import { translateKernelStatuses } from '../kernelstatuses';
/**
 * The class name added to toolbar kernel name text.
 */
const TOOLBAR_KERNEL_NAME_CLASS = 'jp-Toolbar-kernelName';
/**
 * The class name added to toolbar kernel status icon.
 */
const TOOLBAR_KERNEL_STATUS_CLASS = 'jp-Toolbar-kernelStatus';
/**
 * The namespace for Toolbar class statics.
 */
export var Toolbar;
(function (Toolbar) {
    /**
     * Create an interrupt toolbar item.
     *
     * @deprecated since version v3.2
     * This is dead code now.
     */
    function createInterruptButton(sessionContext, translator) {
        translator = translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        return new ToolbarButton({
            icon: stopIcon,
            onClick: () => {
                var _a, _b;
                void ((_b = (_a = sessionContext.session) === null || _a === void 0 ? void 0 : _a.kernel) === null || _b === void 0 ? void 0 : _b.interrupt());
            },
            tooltip: trans.__('Interrupt the kernel')
        });
    }
    Toolbar.createInterruptButton = createInterruptButton;
    /**
     * Create a restart toolbar item.
     *
     * @deprecated since v3.2
     * This is dead code now.
     */
    function createRestartButton(sessionContext, dialogs, translator) {
        translator = translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        return new ToolbarButton({
            icon: refreshIcon,
            onClick: () => {
                void (dialogs !== null && dialogs !== void 0 ? dialogs : sessionContextDialogs).restart(sessionContext, translator);
            },
            tooltip: trans.__('Restart the kernel')
        });
    }
    Toolbar.createRestartButton = createRestartButton;
    /**
     * Create a kernel name indicator item.
     *
     * #### Notes
     * It will display the `'display_name`' of the session context. It can
     * handle a change in context or kernel.
     */
    function createKernelNameItem(sessionContext, dialogs, translator) {
        const el = ReactWidget.create(React.createElement(Private.KernelNameComponent, { sessionContext: sessionContext, dialogs: dialogs !== null && dialogs !== void 0 ? dialogs : sessionContextDialogs, translator: translator }));
        el.addClass('jp-KernelName');
        return el;
    }
    Toolbar.createKernelNameItem = createKernelNameItem;
    /**
     * Create a kernel status indicator item.
     *
     * #### Notes
     * It will show a busy status if the kernel status is busy.
     * It will show the current status in the node title.
     * It can handle a change to the context or the kernel.
     */
    function createKernelStatusItem(sessionContext, translator) {
        return new Private.KernelStatus(sessionContext, translator);
    }
    Toolbar.createKernelStatusItem = createKernelStatusItem;
})(Toolbar || (Toolbar = {}));
/**
 * A namespace for private data.
 */
var Private;
(function (Private) {
    /**
     * React component for a kernel name button.
     *
     * This wraps the ToolbarButtonComponent and watches the kernel
     * session for changes.
     */
    function KernelNameComponent(props) {
        const translator = props.translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        const callback = () => {
            void props.dialogs.selectKernel(props.sessionContext, translator);
        };
        return (React.createElement(UseSignal, { signal: props.sessionContext.kernelChanged, initialSender: props.sessionContext }, sessionContext => (React.createElement(ToolbarButtonComponent, { className: TOOLBAR_KERNEL_NAME_CLASS, onClick: callback, tooltip: trans.__('Switch kernel'), label: sessionContext === null || sessionContext === void 0 ? void 0 : sessionContext.kernelDisplayName }))));
    }
    Private.KernelNameComponent = KernelNameComponent;
    /**
     * A toolbar item that displays kernel status.
     */
    class KernelStatus extends Widget {
        /**
         * Construct a new kernel status widget.
         */
        constructor(sessionContext, translator) {
            super();
            this.translator = translator || nullTranslator;
            this._trans = this.translator.load('jupyterlab');
            this.addClass(TOOLBAR_KERNEL_STATUS_CLASS);
            this._statusNames = translateKernelStatuses(this.translator);
            this._onStatusChanged(sessionContext);
            sessionContext.statusChanged.connect(this._onStatusChanged, this);
            sessionContext.connectionStatusChanged.connect(this._onStatusChanged, this);
        }
        /**
         * Handle a status on a kernel.
         */
        _onStatusChanged(sessionContext) {
            if (this.isDisposed) {
                return;
            }
            const status = sessionContext.kernelDisplayStatus;
            const circleIconProps = {
                container: this.node,
                title: this._trans.__('Kernel %1', this._statusNames[status] || status),
                stylesheet: 'toolbarButton',
                alignSelf: 'normal',
                height: '24px'
            };
            // set the icon
            LabIcon.remove(this.node);
            if (status === 'busy' ||
                status === 'starting' ||
                status === 'terminating' ||
                status === 'restarting' ||
                status === 'initializing') {
                circleIcon.element(circleIconProps);
            }
            else if (status === 'connecting' ||
                status === 'disconnected' ||
                status === 'unknown') {
                offlineBoltIcon.element(circleIconProps);
            }
            else {
                circleEmptyIcon.element(circleIconProps);
            }
        }
    }
    Private.KernelStatus = KernelStatus;
})(Private || (Private = {}));
//# sourceMappingURL=widget.js.map