// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { Signal } from '@lumino/signaling';
export class StatusMessage {
    constructor() {
        this._changed = new Signal(this);
        this._isDisposed = false;
        this._message = '';
        this._timer = null;
    }
    /**
     * Signal emitted on status changed event.
     */
    get changed() {
        return this._changed;
    }
    /**
     * Test whether the object is disposed.
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Dispose the object.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        this._isDisposed = true;
        if (this._timer) {
            window.clearTimeout(this._timer);
        }
        Signal.clearData(this);
    }
    /**
     * The text message to be shown on the statusbar.
     */
    get message() {
        return this._message;
    }
    /**
     * Set the text message and (optionally) the timeout to remove it.
     * @param message
     * @param timeout - number of ms to until the message is cleaned;
     *        -1 if the message should stay up indefinitely;
     *        defaults to 3000ms (3 seconds)
     */
    set(message, timeout = 1000 * 3) {
        this._expireTimer();
        this._message = message;
        this._changed.emit();
        if (timeout !== -1) {
            this._timer = window.setTimeout(this.clear.bind(this), timeout);
        }
    }
    /**
     * Clear the status message.
     */
    clear() {
        this._message = '';
        this._changed.emit();
    }
    /**
     * Clear the previous `setTimeout` call.
     */
    _expireTimer() {
        if (this._timer !== null) {
            window.clearTimeout(this._timer);
            this._timer = null;
        }
    }
}
//# sourceMappingURL=statusmessage.js.map