import { ISignal } from '@lumino/signaling';
import { ClientNotifications, ClientRequests, IDocumentInfo, ILSPConnection, ILSPOptions, Method, ServerNotifications, ServerRequests } from './tokens';
import { LspWsConnection } from './ws-connection/ws-connection';
import type * as lsp from 'vscode-languageserver-protocol';
export declare const Provider: {
    [key: string]: keyof lsp.ServerCapabilities;
};
declare type AnyMethod = Method.ServerNotification | Method.ClientNotification | Method.ClientRequest | Method.ServerRequest;
declare enum MessageKind {
    clientNotifiedServer = 0,
    serverNotifiedClient = 1,
    serverRequested = 2,
    clientRequested = 3,
    resultForClient = 4,
    responseForServer = 5
}
interface IMessageLog<T extends AnyMethod = AnyMethod> {
    method: T;
    message: any;
}
export declare class LSPConnection extends LspWsConnection implements ILSPConnection {
    constructor(options: ILSPOptions);
    /**
     * Identifier of the language server
     */
    readonly serverIdentifier?: string;
    /**
     * Language of the language server
     */
    readonly serverLanguage?: string;
    /**
     * Notifications comes from the client.
     */
    readonly clientNotifications: ClientNotifications;
    /**
     * Notifications comes from the server.
     */
    readonly serverNotifications: ServerNotifications;
    /**
     * Requests comes from the client.
     */
    clientRequests: ClientRequests;
    /**
     * Responses comes from the server.
     */
    serverRequests: ServerRequests;
    /**
     * Should log all communication?
     */
    logAllCommunication: boolean;
    /**
     * Signal emitted when the connection is closed.
     */
    get closeSignal(): ISignal<ILSPConnection, boolean>;
    /**
     * Signal emitted when the connection receives an error
     * message..
     */
    get errorSignal(): ISignal<ILSPConnection, any>;
    /**
     * Signal emitted when the connection is initialized.
     */
    get serverInitialized(): ISignal<ILSPConnection, lsp.ServerCapabilities<any>>;
    /**
     * Dispose the connection.
     */
    dispose(): void;
    /**
     * Helper to print the logs to logger, for now we are using
     * directly the browser's console.
     */
    log(kind: MessageKind, message: IMessageLog): void;
    /**
     * Send the open request to the backend when the server is
     * ready.
     */
    sendOpenWhenReady(documentInfo: IDocumentInfo): void;
    /**
     * Send the document changes to the server.
     */
    sendSelectiveChange(changeEvent: lsp.TextDocumentContentChangeEvent, documentInfo: IDocumentInfo): void;
    /**
     * Send all changes to the server.
     */
    sendFullTextChange(text: string, documentInfo: IDocumentInfo): void;
    /**
     * Check if a provider is available in the registered capabilities.
     */
    provides(provider: keyof lsp.ServerCapabilities): boolean;
    /**
     * Close the connection to the server.
     */
    close(): void;
    /**
     * initialize a connection over a web socket that speaks the LSP
     */
    connect(socket: WebSocket): void;
    /**
     * Get send request to the server to get completion results
     * from a completion item
     */
    getCompletionResolve(completionItem: lsp.CompletionItem): Promise<lsp.CompletionItem | undefined>;
    /**
     * List of documents waiting to be opened once the connection
     * is ready.
     */
    protected documentsToOpen: IDocumentInfo[];
    /**
     * Generate the notification handlers
     */
    protected constructNotificationHandlers<T extends ServerNotifications | ClientNotifications>(methods: typeof Method.ServerNotification | typeof Method.ClientNotification): T;
    /**
     * Generate the client request handler
     */
    protected constructClientRequestHandler<T extends ClientRequests, U extends keyof T = keyof T>(methods: typeof Method.ClientRequest): T;
    /**
     * Generate the server response handler
     */
    protected constructServerRequestHandler<T extends ServerRequests, U extends keyof T = keyof T>(methods: typeof Method.ServerRequest): T;
    /**
     * Initialization parameters to be sent to the language server.
     * Subclasses can overload this when adding more features.
     */
    protected initializeParams(): lsp.InitializeParams;
    /**
     * Callback called when the server is initialized.
     */
    protected onServerInitialized(params: lsp.InitializeResult): void;
    /**
     * Once the server is initialized, this method generates the
     * client and server handlers
     */
    protected afterInitialized(): void;
    /**
     * Is the connection is closed manually?
     */
    private _closingManually;
    private _options;
    private _closeSignal;
    private _errorSignal;
    private _serverInitialized;
    /**
     * Send the document changed data to the server.
     */
    private _sendChange;
}
export {};
