import { MessageConnection } from 'vscode-ws-jsonrpc';
import { ISignal } from '@lumino/signaling';
import { IDocumentInfo, ILspConnection, ILspOptions } from './types';
import type * as protocol from 'vscode-languageserver-protocol';
export declare class LspWsConnection implements ILspConnection {
    constructor(options: ILspOptions);
    /**
     * Is the language server is connected?
     */
    get isConnected(): boolean;
    /**
     * Is the language server is initialized?
     */
    get isInitialized(): boolean;
    /**
     * Is the language server is connected and initialized?
     */
    get isReady(): boolean;
    /**
     * A signal emitted when the connection is disposed.
     */
    get disposed(): ISignal<this, void>;
    /**
     * Check if the connection is disposed
     */
    get isDisposed(): boolean;
    /**
     * Initialize a connection over a web socket that speaks the LSP protocol
     */
    connect(socket: WebSocket): void;
    /**
     * Close the connection
     */
    close(): void;
    /**
     * The initialize request telling the server which options the client supports
     */
    sendInitialize(): void;
    /**
     * Inform the server that the document was opened
     */
    sendOpen(documentInfo: IDocumentInfo): void;
    /**
     * Sends the full text of the document to the server
     */
    sendChange(documentInfo: IDocumentInfo): void;
    /**
     * Send save notification to the server.
     */
    sendSaved(documentInfo: IDocumentInfo): void;
    /**
     * Send configuration change to the server.
     */
    sendConfigurationChange(settings: protocol.DidChangeConfigurationParams): void;
    /**
     * Dispose the connection.
     */
    dispose(): void;
    /**
     * The internal websocket connection to the LSP handler
     */
    protected socket: WebSocket;
    /**
     * The json-rpc wrapper over the internal websocket connection.
     */
    protected connection: MessageConnection;
    /**
     * Map to track opened virtual documents..
     */
    protected openedUris: Map<string, boolean>;
    /**
     * Server capabilities provider.
     */
    protected serverCapabilities: protocol.ServerCapabilities;
    /**
     * The connection is connected?
     */
    protected _isConnected: boolean;
    /**
     * The connection is initialized?
     */
    protected _isInitialized: boolean;
    /**
     * Array of LSP callback disposables, it is used to
     * clear the callbacks when the connection is disposed.
     */
    protected _disposables: Array<protocol.Disposable>;
    /**
     * Callback called when the server is initialized.
     */
    protected onServerInitialized(params: protocol.InitializeResult): void;
    /**
     * Initialization parameters to be sent to the language server.
     * Subclasses should override this when adding more features.
     */
    protected initializeParams(): protocol.InitializeParams;
    /**
     * URI of the LSP handler enpoint.
     */
    private _rootUri;
    private _disposed;
    private _isDisposed;
}
