// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { CodeCellModel, MarkdownCellModel, RawCellModel } from '@jupyterlab/cells';
import { ObservableList, ObservableMap } from '@jupyterlab/observables';
import { Signal } from '@lumino/signaling';
/**
 * A cell list object that supports undo/redo.
 */
export class CellList {
    /**
     * Construct the cell list.
     */
    constructor(model) {
        this._isDisposed = false;
        this._changed = new Signal(this);
        this._cellOrder = new ObservableList();
        this._cellMap = new ObservableMap();
        this.nbmodel = model;
        this._cellOrder.changed.connect(this._onOrderChanged, this);
        this.nbmodel.changed.connect(this.onSharedModelChanged, this);
        this._insertCells(0, this.nbmodel.cells);
    }
    _insertCells(index, cells) {
        const cellModels = cells.map(nbcell => {
            switch (nbcell.cell_type) {
                case 'code': {
                    return new CodeCellModel({
                        sharedModel: nbcell
                    });
                }
                case 'markdown': {
                    return new MarkdownCellModel({
                        sharedModel: nbcell
                    });
                }
                default: {
                    return new RawCellModel({
                        sharedModel: nbcell
                    });
                }
            }
        });
        for (const cell of cellModels) {
            this._cellMap.set(cell.id, cell);
            this._cellOrder.insert(index++, cell.id);
        }
        return this.length;
    }
    onSharedModelChanged(self, change) {
        var _a;
        let currpos = 0;
        (_a = change.cellsChange) === null || _a === void 0 ? void 0 : _a.forEach(delta => {
            if (delta.insert != null) {
                this._insertCells(currpos, delta.insert);
                currpos += delta.insert.length;
            }
            else if (delta.delete != null) {
                this._cellOrder.removeRange(currpos, currpos + delta.delete);
            }
            else if (delta.retain != null) {
                currpos += delta.retain;
            }
        });
    }
    /**
     * A signal emitted when the cell list has changed.
     */
    get changed() {
        return this._changed;
    }
    /**
     * Test whether the cell list has been disposed.
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Test whether the list is empty.
     *
     * @returns `true` if the cell list is empty, `false` otherwise.
     *
     * #### Notes
     * This is a read-only property.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * No changes.
     */
    get isEmpty() {
        return this._cellOrder.length === 0;
    }
    /**
     * Get the length of the cell list.
     *
     * @returns The number of cells in the cell list.
     *
     * #### Notes
     * This is a read-only property.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * No changes.
     */
    get length() {
        return this._cellOrder.length;
    }
    /**
     * Create an iterator over the cells in the cell list.
     *
     * @returns A new iterator starting at the front of the cell list.
     */
    *[Symbol.iterator]() {
        for (const id of this._cellOrder) {
            yield this._cellMap.get(id);
        }
    }
    /**
     * Dispose of the resources held by the cell list.
     */
    dispose() {
        if (this._isDisposed) {
            return;
        }
        this._isDisposed = true;
        Signal.clearData(this);
        // Clean up the cell map and cell order objects.
        for (const cell of this._cellMap.values()) {
            cell.dispose();
        }
        this._cellMap.dispose();
        this._cellOrder.dispose();
    }
    /**
     * Get the cell at the specified index.
     *
     * @param index - The positive integer index of interest.
     *
     * @returns The cell at the specified index.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * No changes.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral or out of range.
     */
    get(index) {
        return this._cellMap.get(this._cellOrder.get(index));
    }
    _onOrderChanged(order, change) {
        const newValues = [];
        const oldValues = [];
        for (const id of change.newValues) {
            newValues.push(this._cellMap.get(id));
        }
        for (const id of change.oldValues) {
            oldValues.push(this._cellMap.get(id));
        }
        this._changed.emit({
            type: change.type,
            oldIndex: change.oldIndex,
            newIndex: change.newIndex,
            oldValues,
            newValues
        });
    }
}
//# sourceMappingURL=celllist.js.map