import { IChangedArgs } from '@jupyterlab/coreutils';
import { DocumentRegistry } from '@jupyterlab/docregistry';
import * as nbformat from '@jupyterlab/nbformat';
import { IObservableJSON } from '@jupyterlab/observables';
import * as sharedModels from '@jupyterlab/shared-models';
import { ITranslator } from '@jupyterlab/translation';
import { ISignal } from '@lumino/signaling';
import { CellList } from './celllist';
/**
 * The definition of a model object for a notebook widget.
 */
export interface INotebookModel extends DocumentRegistry.IModel {
    /**
     * The list of cells in the notebook.
     */
    readonly cells: CellList;
    /**
     * The major version number of the nbformat.
     */
    readonly nbformat: number;
    /**
     * The minor version number of the nbformat.
     */
    readonly nbformatMinor: number;
    /**
     * The metadata associated with the notebook.
     */
    readonly metadata: IObservableJSON;
    /**
     * The array of deleted cells since the notebook was last run.
     */
    readonly deletedCells: string[];
    /**
     * Shared model
     */
    readonly sharedModel: sharedModels.ISharedNotebook;
}
/**
 * An implementation of a notebook Model.
 */
export declare class NotebookModel implements INotebookModel {
    /**
     * Construct a new notebook model.
     */
    constructor(options?: NotebookModel.IOptions);
    /**
     * A signal emitted when the document content changes.
     */
    get contentChanged(): ISignal<this, void>;
    /**
     * A signal emitted when the document state changes.
     */
    get stateChanged(): ISignal<this, IChangedArgs<any>>;
    /**
     * The dirty state of the document.
     */
    get dirty(): boolean;
    set dirty(newValue: boolean);
    /**
     * The read only state of the document.
     */
    get readOnly(): boolean;
    set readOnly(newValue: boolean);
    /**
     * The metadata associated with the notebook.
     */
    get metadata(): IObservableJSON;
    /**
     * Get the observable list of notebook cells.
     */
    get cells(): CellList;
    /**
     * The major version number of the nbformat.
     */
    get nbformat(): number;
    /**
     * The minor version number of the nbformat.
     */
    get nbformatMinor(): number;
    /**
     * The default kernel name of the document.
     */
    get defaultKernelName(): string;
    /**
     * A list of deleted cells for the notebook..
     */
    get deletedCells(): string[];
    /**
     * The default kernel language of the document.
     */
    get defaultKernelLanguage(): string;
    /**
     * Dispose of the resources held by the model.
     */
    dispose(): void;
    /**
     * Serialize the model to a string.
     */
    toString(): string;
    /**
     * Deserialize the model from a string.
     *
     * #### Notes
     * Should emit a [contentChanged] signal.
     */
    fromString(value: string): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.INotebookContent;
    /**
     * Deserialize the model from JSON.
     *
     * #### Notes
     * Should emit a [contentChanged] signal.
     */
    fromJSON(value: nbformat.INotebookContent): void;
    /**
     * Handle a change in the cells list.
     */
    private _onCellsChanged;
    private _onStateChanged;
    private _onMetadataChanged;
    /**
     * Make sure we have the required metadata fields.
     */
    private _ensureMetadata;
    /**
     * Trigger a state change signal.
     */
    protected triggerStateChange(args: IChangedArgs<any>): void;
    /**
     * Trigger a content changed signal.
     */
    protected triggerContentChange(): void;
    /**
     * Whether the model is disposed.
     */
    get isDisposed(): boolean;
    /**
     * The shared notebook model.
     */
    readonly sharedModel: sharedModels.ISharedNotebook;
    /**
     * A mutex to update the shared model.
     */
    protected readonly _modelDBMutex: (f: () => void) => void;
    private _dirty;
    private _readOnly;
    private _contentChanged;
    private _stateChanged;
    private _trans;
    private _cells;
    private _metadata;
    private _nbformat;
    private _nbformatMinor;
    private _deletedCells;
    private _isDisposed;
}
/**
 * The namespace for the `NotebookModel` class statics.
 */
export declare namespace NotebookModel {
    /**
     * An options object for initializing a notebook model.
     */
    interface IOptions {
        /**
         * The language preference for the model.
         */
        languagePreference?: string;
        /**
         * Default cell type.
         */
        defaultCell?: 'code' | 'markdown' | 'raw';
        /**
         * Language translator.
         */
        translator?: ITranslator;
        /**
         * Defines if the document can be undo/redo.
         */
        disableDocumentWideUndoRedo?: boolean;
    }
}
