// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { InputPrompt } from '@jupyterlab/cells';
import { JSONEditor } from '@jupyterlab/codeeditor';
import { Mode } from '@jupyterlab/codemirror';
import { ObservableJSON } from '@jupyterlab/observables';
import { nullTranslator } from '@jupyterlab/translation';
import { Collapser, Styling } from '@jupyterlab/ui-components';
import { ArrayExt } from '@lumino/algorithm';
import { ConflatableMessage, MessageLoop } from '@lumino/messaging';
import { h, VirtualDOM } from '@lumino/virtualdom';
import { Debouncer } from '@lumino/polling';
import { PanelLayout, Widget } from '@lumino/widgets';
class RankedPanel extends Widget {
    constructor() {
        super();
        this._items = [];
        this.layout = new PanelLayout();
        this.addClass('jp-RankedPanel');
    }
    addWidget(widget, rank) {
        const rankItem = { widget, rank };
        const index = ArrayExt.upperBound(this._items, rankItem, Private.itemCmp);
        ArrayExt.insert(this._items, index, rankItem);
        const layout = this.layout;
        layout.insertWidget(index, widget);
    }
    /**
     * Handle the removal of a child
     *
     */
    onChildRemoved(msg) {
        const index = ArrayExt.findFirstIndex(this._items, item => item.widget === msg.child);
        if (index !== -1) {
            ArrayExt.removeAt(this._items, index);
        }
    }
}
/**
 * A widget that provides metadata tools.
 */
export class NotebookTools extends Widget {
    /**
     * Construct a new NotebookTools object.
     */
    constructor(options) {
        super();
        this.addClass('jp-NotebookTools');
        this.translator = options.translator || nullTranslator;
        this._trans = this.translator.load('jupyterlab');
        this._commonTools = new RankedPanel();
        this._advancedTools = new RankedPanel();
        this._advancedTools.title.label = this._trans.__('Advanced Tools');
        const layout = (this.layout = new PanelLayout());
        layout.addWidget(this._commonTools);
        layout.addWidget(new Collapser({ widget: this._advancedTools }));
        this._tracker = options.tracker;
        this._tracker.currentChanged.connect(this._onActiveNotebookPanelChanged, this);
        this._tracker.activeCellChanged.connect(this._onActiveCellChanged, this);
        this._tracker.selectionChanged.connect(this._onSelectionChanged, this);
        this._onActiveNotebookPanelChanged();
        this._onActiveCellChanged();
        this._onSelectionChanged();
    }
    /**
     * The active cell widget.
     */
    get activeCell() {
        return this._tracker.activeCell;
    }
    /**
     * The currently selected cells.
     */
    get selectedCells() {
        const panel = this._tracker.currentWidget;
        if (!panel) {
            return [];
        }
        const notebook = panel.content;
        return notebook.widgets.filter(cell => notebook.isSelectedOrActive(cell));
    }
    /**
     * The current notebook.
     */
    get activeNotebookPanel() {
        return this._tracker.currentWidget;
    }
    /**
     * Add a cell tool item.
     */
    addItem(options) {
        var _a;
        const tool = options.tool;
        const rank = (_a = options.rank) !== null && _a !== void 0 ? _a : 100;
        let section;
        if (options.section === 'advanced') {
            section = this._advancedTools;
        }
        else {
            section = this._commonTools;
        }
        tool.addClass('jp-NotebookTools-tool');
        section.addWidget(tool, rank);
        // TODO: perhaps the necessary notebookTools functionality should be
        // consolidated into a single object, rather than a broad reference to this.
        tool.notebookTools = this;
        // Trigger the tool to update its active notebook and cell.
        MessageLoop.sendMessage(tool, NotebookTools.ActiveNotebookPanelMessage);
        MessageLoop.sendMessage(tool, NotebookTools.ActiveCellMessage);
    }
    /**
     * Handle a change to the notebook panel.
     */
    _onActiveNotebookPanelChanged() {
        if (this._prevActiveNotebookModel &&
            !this._prevActiveNotebookModel.isDisposed) {
            this._prevActiveNotebookModel.metadata.changed.disconnect(this._onActiveNotebookPanelMetadataChanged, this);
        }
        const activeNBModel = this.activeNotebookPanel && this.activeNotebookPanel.content
            ? this.activeNotebookPanel.content.model
            : null;
        this._prevActiveNotebookModel = activeNBModel;
        if (activeNBModel) {
            activeNBModel.metadata.changed.connect(this._onActiveNotebookPanelMetadataChanged, this);
        }
        for (const widget of this._toolChildren()) {
            MessageLoop.sendMessage(widget, NotebookTools.ActiveNotebookPanelMessage);
        }
    }
    /**
     * Handle a change to the active cell.
     */
    _onActiveCellChanged() {
        if (this._prevActiveCell && !this._prevActiveCell.isDisposed) {
            this._prevActiveCell.metadata.changed.disconnect(this._onActiveCellMetadataChanged, this);
        }
        const activeCell = this.activeCell ? this.activeCell.model : null;
        this._prevActiveCell = activeCell;
        if (activeCell) {
            activeCell.metadata.changed.connect(this._onActiveCellMetadataChanged, this);
        }
        for (const widget of this._toolChildren()) {
            MessageLoop.sendMessage(widget, NotebookTools.ActiveCellMessage);
        }
    }
    /**
     * Handle a change in the selection.
     */
    _onSelectionChanged() {
        for (const widget of this._toolChildren()) {
            MessageLoop.sendMessage(widget, NotebookTools.SelectionMessage);
        }
    }
    /**
     * Handle a change in the active cell metadata.
     */
    _onActiveNotebookPanelMetadataChanged(sender, args) {
        const message = new ObservableJSON.ChangeMessage('activenotebookpanel-metadata-changed', args);
        for (const widget of this._toolChildren()) {
            MessageLoop.sendMessage(widget, message);
        }
    }
    /**
     * Handle a change in the notebook model metadata.
     */
    _onActiveCellMetadataChanged(sender, args) {
        const message = new ObservableJSON.ChangeMessage('activecell-metadata-changed', args);
        for (const widget of this._toolChildren()) {
            MessageLoop.sendMessage(widget, message);
        }
    }
    *_toolChildren() {
        yield* this._commonTools.children();
        yield* this._advancedTools.children();
    }
}
/**
 * The namespace for NotebookTools class statics.
 */
(function (NotebookTools) {
    /**
     * A singleton conflatable `'activenotebookpanel-changed'` message.
     */
    NotebookTools.ActiveNotebookPanelMessage = new ConflatableMessage('activenotebookpanel-changed');
    /**
     * A singleton conflatable `'activecell-changed'` message.
     */
    NotebookTools.ActiveCellMessage = new ConflatableMessage('activecell-changed');
    /**
     * A singleton conflatable `'selection-changed'` message.
     */
    NotebookTools.SelectionMessage = new ConflatableMessage('selection-changed');
    /**
     * The base notebook tool, meant to be subclassed.
     */
    class Tool extends Widget {
        dispose() {
            super.dispose();
            if (this.notebookTools) {
                this.notebookTools = null;
            }
        }
        /**
         * Process a message sent to the widget.
         *
         * @param msg - The message sent to the widget.
         */
        processMessage(msg) {
            super.processMessage(msg);
            switch (msg.type) {
                case 'activenotebookpanel-changed':
                    this.onActiveNotebookPanelChanged(msg);
                    break;
                case 'activecell-changed':
                    this.onActiveCellChanged(msg);
                    break;
                case 'selection-changed':
                    this.onSelectionChanged(msg);
                    break;
                case 'activecell-metadata-changed':
                    this.onActiveCellMetadataChanged(msg);
                    break;
                case 'activenotebookpanel-metadata-changed':
                    this.onActiveNotebookPanelMetadataChanged(msg);
                    break;
                default:
                    break;
            }
        }
        /**
         * Handle a change to the notebook panel.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        onActiveNotebookPanelChanged(msg) {
            /* no-op */
        }
        /**
         * Handle a change to the active cell.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        onActiveCellChanged(msg) {
            /* no-op */
        }
        /**
         * Handle a change to the selection.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        onSelectionChanged(msg) {
            /* no-op */
        }
        /**
         * Handle a change to the metadata of the active cell.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        onActiveCellMetadataChanged(msg) {
            /* no-op */
        }
        /**
         * Handle a change to the metadata of the active cell.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        onActiveNotebookPanelMetadataChanged(msg) {
            /* no-op */
        }
    }
    NotebookTools.Tool = Tool;
    /**
     * A cell tool displaying the active cell contents.
     */
    class ActiveCellTool extends Tool {
        /**
         * Construct a new active cell tool.
         */
        constructor() {
            super();
            this.addClass('jp-ActiveCellTool');
            this.layout = new PanelLayout();
            this._inputPrompt = new InputPrompt();
            this.layout.addWidget(this._inputPrompt);
            // First code line container
            const node = document.createElement('div');
            node.classList.add('jp-ActiveCell-Content');
            const container = node.appendChild(document.createElement('div'));
            const editor = container.appendChild(document.createElement('pre'));
            container.className = 'jp-Cell-Content';
            this._editorEl = editor;
            this.layout.addWidget(new Widget({ node }));
            const update = async () => {
                var _a, _b, _c;
                this._editorEl.innerHTML = '';
                if (((_a = this._cellModel) === null || _a === void 0 ? void 0 : _a.type) === 'code') {
                    this._inputPrompt.executionCount = `${(_b = this._cellModel.executionCount) !== null && _b !== void 0 ? _b : ''}`;
                    this._inputPrompt.show();
                }
                else {
                    this._inputPrompt.executionCount = null;
                    this._inputPrompt.hide();
                }
                if (this._cellModel) {
                    const spec = await Mode.ensure((_c = Mode.findByMIME(this._cellModel.mimeType)) !== null && _c !== void 0 ? _c : 'text/plain');
                    Mode.run(this._cellModel.sharedModel.getSource().split('\n')[0], spec, this._editorEl);
                }
            };
            this._refreshDebouncer = new Debouncer(update, 150);
        }
        /**
         * Handle a change to the active cell.
         */
        async onActiveCellChanged() {
            const activeCell = this.notebookTools.activeCell;
            if (this._cellModel && !this._cellModel.isDisposed) {
                this._cellModel.sharedModel.changed.disconnect(this.refresh, this);
                this._cellModel.mimeTypeChanged.disconnect(this.refresh, this);
            }
            if (!activeCell) {
                this._cellModel = null;
                return;
            }
            const cellModel = (this._cellModel = activeCell.model);
            cellModel.sharedModel.changed.connect(this.refresh, this);
            cellModel.mimeTypeChanged.connect(this.refresh, this);
            await this.refresh();
        }
        async refresh() {
            await this._refreshDebouncer.invoke();
        }
    }
    NotebookTools.ActiveCellTool = ActiveCellTool;
    /**
     * A raw metadata editor.
     */
    class MetadataEditorTool extends Tool {
        /**
         * Construct a new raw metadata tool.
         */
        constructor(options) {
            super();
            const { editorFactory } = options;
            this.addClass('jp-MetadataEditorTool');
            const layout = (this.layout = new PanelLayout());
            this.editor = new JSONEditor({
                editorFactory
            });
            this.editor.title.label = options.label || 'Edit Metadata';
            const titleNode = new Widget({ node: document.createElement('label') });
            titleNode.node.textContent = options.label || 'Edit Metadata';
            layout.addWidget(titleNode);
            layout.addWidget(this.editor);
        }
    }
    NotebookTools.MetadataEditorTool = MetadataEditorTool;
    /**
     * A notebook metadata editor
     */
    class NotebookMetadataEditorTool extends MetadataEditorTool {
        constructor(options) {
            const translator = options.translator || nullTranslator;
            const trans = translator.load('jupyterlab');
            options.label = options.label || trans.__('Notebook Metadata');
            super(options);
        }
        /**
         * Handle a change to the notebook.
         */
        onActiveNotebookPanelChanged(msg) {
            this._update();
        }
        /**
         * Handle a change to the notebook metadata.
         */
        onActiveNotebookPanelMetadataChanged(msg) {
            this._update();
        }
        _update() {
            var _a, _b;
            const nb = this.notebookTools.activeNotebookPanel &&
                this.notebookTools.activeNotebookPanel.content;
            this.editor.source = (_b = (_a = nb === null || nb === void 0 ? void 0 : nb.model) === null || _a === void 0 ? void 0 : _a.metadata) !== null && _b !== void 0 ? _b : null;
        }
    }
    NotebookTools.NotebookMetadataEditorTool = NotebookMetadataEditorTool;
    /**
     * A cell metadata editor
     */
    class CellMetadataEditorTool extends MetadataEditorTool {
        constructor(options) {
            const translator = options.translator || nullTranslator;
            const trans = translator.load('jupyterlab');
            options.label = options.label || trans.__('Cell Metadata');
            super(options);
        }
        /**
         * Handle a change to the active cell.
         */
        onActiveCellChanged(msg) {
            this._update();
        }
        /**
         * Handle a change to the active cell metadata.
         */
        onActiveCellMetadataChanged(msg) {
            this._update();
        }
        _update() {
            const cell = this.notebookTools.activeCell;
            this.editor.source = cell ? cell.model.metadata : null;
        }
    }
    NotebookTools.CellMetadataEditorTool = CellMetadataEditorTool;
    /**
     * A cell tool that provides a selection for a given metadata key.
     */
    class KeySelector extends Tool {
        /**
         * Construct a new KeySelector.
         */
        constructor(options) {
            // TODO: use react
            super({ node: Private.createSelectorNode(options) });
            /**
             * Get the value for the data.
             */
            this._getValue = (cell) => {
                let value = cell.model.metadata.get(this.key);
                if (value === undefined) {
                    value = this._default;
                }
                return value;
            };
            /**
             * Set the value for the data.
             */
            this._setValue = (cell, value) => {
                if (value === this._default) {
                    cell.model.metadata.delete(this.key);
                }
                else {
                    cell.model.metadata.set(this.key, value);
                }
            };
            this._changeGuard = false;
            this.addClass('jp-KeySelector');
            this.key = options.key;
            this._default = options.default;
            this._validCellTypes = options.validCellTypes || [];
            this._getter = options.getter || this._getValue;
            this._setter = options.setter || this._setValue;
        }
        /**
         * The select node for the widget.
         */
        get selectNode() {
            return this.node.getElementsByTagName('select')[0];
        }
        /**
         * Handle the DOM events for the widget.
         *
         * @param event - The DOM event sent to the widget.
         *
         * #### Notes
         * This method implements the DOM `EventListener` interface and is
         * called in response to events on the notebook panel's node. It should
         * not be called directly by user code.
         */
        handleEvent(event) {
            switch (event.type) {
                case 'change':
                    this.onValueChanged();
                    break;
                default:
                    break;
            }
        }
        /**
         * Handle `after-attach` messages for the widget.
         */
        onAfterAttach(msg) {
            const node = this.selectNode;
            node.addEventListener('change', this);
        }
        /**
         * Handle `before-detach` messages for the widget.
         */
        onBeforeDetach(msg) {
            const node = this.selectNode;
            node.removeEventListener('change', this);
        }
        /**
         * Handle a change to the active cell.
         */
        onActiveCellChanged(msg) {
            const select = this.selectNode;
            const activeCell = this.notebookTools.activeCell;
            if (!activeCell) {
                select.disabled = true;
                select.value = '';
                return;
            }
            const cellType = activeCell.model.type;
            if (this._validCellTypes.length &&
                this._validCellTypes.indexOf(cellType) === -1) {
                select.value = '';
                select.disabled = true;
                return;
            }
            select.disabled = false;
            this._changeGuard = true;
            const getter = this._getter;
            select.value = JSON.stringify(getter(activeCell));
            this._changeGuard = false;
        }
        /**
         * Handle a change to the metadata of the active cell.
         */
        onActiveCellMetadataChanged(msg) {
            if (this._changeGuard) {
                return;
            }
            const select = this.selectNode;
            const cell = this.notebookTools.activeCell;
            if (msg.args.key === this.key && cell) {
                this._changeGuard = true;
                const getter = this._getter;
                select.value = JSON.stringify(getter(cell));
                this._changeGuard = false;
            }
        }
        /**
         * Handle a change to the value.
         */
        onValueChanged() {
            const activeCell = this.notebookTools.activeCell;
            if (!activeCell || this._changeGuard) {
                return;
            }
            this._changeGuard = true;
            const select = this.selectNode;
            const setter = this._setter;
            setter(activeCell, JSON.parse(select.value));
            this._changeGuard = false;
        }
    }
    NotebookTools.KeySelector = KeySelector;
    /**
     * A notebook metadata number editor
     */
    class NotebookMetadataNumberTool extends Tool {
        constructor(options) {
            super({
                node: Private.createInputNode({
                    label: options.label,
                    value: '1'
                })
            });
            this.addClass('jp-NumberSetter');
            this._key = options.key;
        }
        /**
         * The select node for the widget.
         */
        get inputNode() {
            return this.node.getElementsByTagName('input')[0];
        }
        /**
         * Handle the DOM events for the widget.
         *
         * @param event - The DOM event sent to the widget.
         *
         * #### Notes
         * This method implements the DOM `EventListener` interface and is
         * called in response to events on the notebook panel's node. It should
         * not be called directly by user code.
         */
        handleEvent(event) {
            switch (event.type) {
                case 'change':
                case 'input':
                    this.onValueChanged();
                    break;
                default:
                    break;
            }
        }
        /**
         * Handle `after-attach` messages for the widget.
         */
        onAfterAttach(msg) {
            const node = this.inputNode;
            node.addEventListener('change', this);
            node.addEventListener('input', this);
        }
        /**
         * Handle `before-detach` messages for the widget.
         */
        onBeforeDetach(msg) {
            const node = this.inputNode;
            node.removeEventListener('change', this);
            node.removeEventListener('input', this);
        }
        /**
         * Handle a change to the notebook.
         */
        onActiveNotebookPanelChanged(msg) {
            this._update();
        }
        /**
         * Handle a change to the notebook metadata.
         */
        onActiveNotebookPanelMetadataChanged(msg) {
            this._update();
        }
        /**
         * Handle a change to the value.
         */
        onValueChanged() {
            var _a, _b, _c, _d;
            const nb = this.notebookTools.activeNotebookPanel &&
                this.notebookTools.activeNotebookPanel.content;
            const metadata = (_b = (_a = nb === null || nb === void 0 ? void 0 : nb.model) === null || _a === void 0 ? void 0 : _a.metadata) !== null && _b !== void 0 ? _b : null;
            if (metadata) {
                const keyPath = this._key.split('/');
                const value = { ...((_c = metadata.get(keyPath[0])) !== null && _c !== void 0 ? _c : {}) };
                let lastObj = value;
                for (let p = 1; p < keyPath.length - 1; p++) {
                    if (lastObj[keyPath[p]] === undefined) {
                        lastObj[keyPath[p]] = {};
                    }
                    lastObj = lastObj[keyPath[p]];
                }
                lastObj[keyPath[keyPath.length - 1]] =
                    (_d = this.inputNode.valueAsNumber) !== null && _d !== void 0 ? _d : 1;
                metadata.set(keyPath[0], value);
            }
        }
        _update() {
            var _a, _b;
            const nb = this.notebookTools.activeNotebookPanel &&
                this.notebookTools.activeNotebookPanel.content;
            const metadata = (_b = (_a = nb === null || nb === void 0 ? void 0 : nb.model) === null || _a === void 0 ? void 0 : _a.metadata) !== null && _b !== void 0 ? _b : null;
            if (metadata) {
                const keyPath = this._key.split('/');
                let value = metadata.get(keyPath[0]);
                for (let p = 1; p < keyPath.length; p++) {
                    value = (value !== null && value !== void 0 ? value : {})[keyPath[p]];
                }
                if (value !== undefined) {
                    this.inputNode.valueAsNumber = value;
                }
                else {
                    this.inputNode.valueAsNumber = 1;
                }
            }
        }
    }
    NotebookTools.NotebookMetadataNumberTool = NotebookMetadataNumberTool;
    /**
     * Create a slideshow selector.
     */
    function createSlideShowSelector(translator) {
        translator = translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        trans.__('');
        const options = {
            key: 'slideshow',
            title: trans.__('Slide Type'),
            optionValueArray: [
                ['-', null],
                [trans.__('Slide'), 'slide'],
                [trans.__('Sub-Slide'), 'subslide'],
                [trans.__('Fragment'), 'fragment'],
                [trans.__('Skip'), 'skip'],
                [trans.__('Notes'), 'notes']
            ],
            getter: cell => {
                const value = cell.model.metadata.get('slideshow');
                return value && value['slide_type'];
            },
            setter: (cell, value) => {
                let data = cell.model.metadata.get('slideshow') || Object.create(null);
                if (value === null) {
                    // Make a shallow copy so we aren't modifying the original metadata.
                    data = { ...data };
                    delete data.slide_type;
                }
                else {
                    data = { ...data, slide_type: value };
                }
                if (Object.keys(data).length > 0) {
                    cell.model.metadata.set('slideshow', data);
                }
                else {
                    cell.model.metadata.delete('slideshow');
                }
            }
        };
        return new KeySelector(options);
    }
    NotebookTools.createSlideShowSelector = createSlideShowSelector;
    /**
     * Create an nbconvert selector.
     */
    function createNBConvertSelector(optionValueArray, translator) {
        translator = translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        return new KeySelector({
            key: 'raw_mimetype',
            title: trans.__('Raw NBConvert Format'),
            optionValueArray: optionValueArray,
            validCellTypes: ['raw']
        });
    }
    NotebookTools.createNBConvertSelector = createNBConvertSelector;
    /**
     * Create read-only toggle.
     */
    function createEditableToggle(translator) {
        translator = translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        return new KeySelector({
            key: 'editable',
            title: trans.__('Editable'),
            optionValueArray: [
                [trans.__('Editable'), true],
                [trans.__('Read-Only'), false]
            ],
            default: true
        });
    }
    NotebookTools.createEditableToggle = createEditableToggle;
    /**
     * Create base table of content numbering
     */
    function createToCBaseNumbering(translator) {
        translator = translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        return new NotebookMetadataNumberTool({
            key: 'toc/base_numbering',
            label: trans.__('Table of content - Base number')
        });
    }
    NotebookTools.createToCBaseNumbering = createToCBaseNumbering;
})(NotebookTools || (NotebookTools = {}));
/**
 * A namespace for private data.
 */
var Private;
(function (Private) {
    /**
     * A comparator function for widget rank items.
     */
    function itemCmp(first, second) {
        return first.rank - second.rank;
    }
    Private.itemCmp = itemCmp;
    /**
     * Create the node for a KeySelector.
     */
    function createSelectorNode(options) {
        const name = options.key;
        const title = options.title || name[0].toLocaleUpperCase() + name.slice(1);
        const optionNodes = [];
        let value;
        let option;
        for (const item of options.optionValueArray) {
            option = item[0];
            value = JSON.stringify(item[1]);
            const attrs = options.default == item[1]
                ? { value, selected: 'selected' }
                : { value };
            optionNodes.push(h.option(attrs, option));
        }
        const node = VirtualDOM.realize(h.div({}, h.label(title, h.select({}, optionNodes))));
        Styling.styleNode(node);
        return node;
    }
    Private.createSelectorNode = createSelectorNode;
    /**
     * Create the node for a number input.
     */
    function createInputNode(options) {
        const title = options.label;
        const node = VirtualDOM.realize(h.div({}, h.label(title, h.input({ value: options.value, type: 'number' }))));
        Styling.styleNode(node);
        return node;
    }
    Private.createInputNode = createInputNode;
})(Private || (Private = {}));
//# sourceMappingURL=notebooktools.js.map