import { IRenderMime } from '@jupyterlab/rendermime-interfaces';
import { Signal } from '@lumino/signaling';
import { ElementAttrs, VirtualElement, VirtualNode } from '@lumino/virtualdom';
import React from 'react';
import { LabIconStyle } from '../style';
export declare class LabIcon implements LabIcon.ILabIcon, VirtualElement.IRenderer {
    /** *********
     * statics *
     ***********/
    /**
     * Remove any rendered icon from the element that contains it
     *
     * @param container - a DOM node into which an icon was
     * previously rendered
     *
     * @returns the cleaned container
     */
    static remove(container: HTMLElement): HTMLElement;
    /**
     * Resolve an icon name or a {name, svgstr} pair into an
     * actual LabIcon.
     *
     * @param icon - either a string with the name of an existing icon
     * or an object with {name: string, svgstr: string} fields.
     *
     * @returns a LabIcon instance
     */
    static resolve({ icon }: {
        icon: LabIcon.IResolvable;
    }): LabIcon;
    /**
     * Resolve an icon name or a {name, svgstr} pair into a DOM element.
     * If icon arg is undefined, the function will fall back to trying to render
     * the icon as a CSS background image, via the iconClass arg.
     * If both icon and iconClass are undefined, this function will return
     * an empty div.
     *
     * @param icon - optional, either a string with the name of an existing icon
     * or an object with {name: string, svgstr: string} fields
     *
     * @param iconClass - optional, if the icon arg is not set, the iconClass arg
     * should be a CSS class associated with an existing CSS background-image
     *
     * @deprecated fallback - don't use, optional, a LabIcon instance that will
     * be used if neither icon nor iconClass are defined
     *
     * @param props - any additional args are passed though to the element method
     * of the resolved icon on render
     *
     * @returns a DOM node with the resolved icon rendered into it
     */
    static resolveElement({ icon, iconClass, fallback, ...props }: Partial<LabIcon.IResolverProps> & LabIcon.IProps): HTMLElement;
    /**
     * Resolve an icon name or a {name, svgstr} pair into a React component.
     * If icon arg is undefined, the function will fall back to trying to render
     * the icon as a CSS background image, via the iconClass arg.
     * If both icon and iconClass are undefined, the returned component
     * will simply render an empty div.
     *
     * @param icon - optional, either a string with the name of an existing icon
     * or an object with {name: string, svgstr: string} fields
     *
     * @param iconClass - optional, if the icon arg is not set, the iconClass arg
     * should be a CSS class associated with an existing CSS background-image
     *
     * @deprecated fallback - don't use, optional, a LabIcon instance that will
     * be used if neither icon nor iconClass are defined
     *
     * @param props - any additional args are passed though to the React component
     * of the resolved icon on render
     *
     * @returns a React component that will render the resolved icon
     */
    static resolveReact({ icon, iconClass, fallback, ...props }: Partial<LabIcon.IResolverProps> & LabIcon.IReactProps): JSX.Element;
    /**
     * Resolve a {name, svgstr} pair into an actual svg node.
     */
    static resolveSvg({ name, svgstr }: LabIcon.IIcon): HTMLElement | null;
    /**
     * Toggle icon debug from off-to-on, or vice-versa.
     *
     * @param debug - optional boolean to force debug on or off
     */
    static toggleDebug(debug?: boolean): void;
    private static _debug;
    private static _instances;
    /** *********
     * members *
     ***********/
    constructor({ name, svgstr, render, unrender, _loading }: LabIcon.IOptions & {
        _loading?: boolean;
    });
    /**
     * Get a view of this icon that is bound to the specified icon/style props
     *
     * @param optional icon/style props (same as args for .element
     * and .react methods). These will be bound to the resulting view
     *
     * @returns a view of this LabIcon instance
     */
    bindprops(props?: LabIcon.IProps): LabIcon;
    /**
     * Create an icon as a DOM element
     *
     * @param className - a string that will be used as the class
     * of the container element. Overrides any existing class
     *
     * @param container - a preexisting DOM element that
     * will be used as the container for the svg element
     *
     * @param label - text that will be displayed adjacent
     * to the icon
     *
     * @param title - a tooltip for the icon
     *
     * @param tag - if container is not explicitly
     * provided, this tag will be used when creating the container
     *
     * @param stylesheet - optional string naming a builtin icon
     * stylesheet, for example 'menuItem' or `statusBar`. Can also be an
     * object defining a custom icon stylesheet, or a list of builtin
     * stylesheet names and/or custom stylesheet objects. If array,
     * the given stylesheets will be merged.
     *
     *   See @jupyterlab/ui-components/src/style/icon.ts for details
     *
     * @param elementPosition - optional position for the inner svg element
     *
     * @param elementSize - optional size for the inner svg element.
     * Set to 'normal' to get a standard 16px x 16px icon
     *
     * @param ...elementCSS - all additional args are treated as
     * overrides for the CSS props applied to the inner svg element
     *
     * @returns A DOM element that contains an (inline) svg element
     * that displays an icon
     */
    element(props?: LabIcon.IProps): HTMLElement;
    render(container: HTMLElement, options?: LabIcon.IRendererOptions): void;
    protected get svgElement(): HTMLElement | null;
    protected get svgInnerHTML(): string | null;
    protected get svgReactAttrs(): any | null;
    get svgstr(): string;
    set svgstr(svgstr: string);
    unrender?(container: HTMLElement, options?: LabIcon.IRendererOptions): void;
    protected _initReact(displayName: string): React.ForwardRefExoticComponent<LabIcon.IProps & React.RefAttributes<SVGElement>>;
    protected _initRender({ render, unrender }: Partial<VirtualElement.IRenderer>): void;
    protected _initSvg({ title, uuid }?: {
        title?: string;
        uuid?: string;
    }): HTMLElement | null;
    readonly name: string;
    /**
     * A React component that will create the icon.
     *
     * @param className - a string that will be used as the class
     * of the container element. Overrides any existing class
     *
     * @param container - a preexisting DOM element that
     * will be used as the container for the svg element
     *
     * @param label - text that will be displayed adjacent
     * to the icon
     *
     * @param title - a tooltip for the icon
     *
     * @param tag - if container is not explicitly
     * provided, this tag will be used when creating the container
     *
     * @param stylesheet - optional string naming a builtin icon
     * stylesheet, for example 'menuItem' or `statusBar`. Can also be an
     * object defining a custom icon stylesheet, or a list of builtin
     * stylesheet names and/or custom stylesheet objects. If array,
     * the given stylesheets will be merged.
     *
     *   See @jupyterlab/ui-components/src/style/icon.ts for details
     *
     * @param elementPosition - optional position for the inner svg element
     *
     * @param elementSize - optional size for the inner svg element.
     * Set to 'normal' to get a standard 16px x 16px icon
     *
     * @param ...elementCSS - all additional args are treated as
     * overrides for the CSS props applied to the inner svg element
     *
     * @param ref - forwarded to the ref prop of the icon's svg element
     */
    readonly react: LabIcon.IReact;
    protected _className: string;
    protected _loading: boolean;
    protected _props: LabIcon.IProps;
    protected _svgReplaced: Signal<this, void>;
    protected _svgstr: string;
    protected _uuid: string;
    /**
     * Cache for svg parsing intermediates
     *   - undefined: the cache has not yet been populated
     *   - null: a valid, but empty, value
     */
    protected _svgElement: HTMLElement | null | undefined;
    protected _svgInnerHTML: string | null | undefined;
    protected _svgReactAttrs: any | null | undefined;
}
/**
 * A namespace for LabIcon statics.
 */
export declare namespace LabIcon {
    /** ***********
     * interfaces *
     *************/
    /**
     * The simplest possible interface for defining a generic icon.
     */
    interface IIcon extends IRenderMime.LabIcon.IIcon {
    }
    interface IRendererOptions {
        attrs?: ElementAttrs;
        children?: ReadonlyArray<VirtualNode>;
        props?: IProps;
    }
    /**
     * The ILabIcon interface. Outside of this interface the actual
     * implementation of LabIcon may vary
     */
    interface ILabIcon extends IIcon, VirtualElement.IRenderer {
    }
    /**
     * Interface defining the parameters to be passed to the LabIcon
     * constructor
     */
    interface IOptions extends IIcon, Partial<VirtualElement.IRenderer> {
    }
    /**
     * The input props for creating a new LabIcon
     */
    interface IProps extends LabIconStyle.IProps {
        /**
         * Extra classNames. Used in addition to the typestyle className to
         * set the className of the icon's outermost container node
         */
        className?: string;
        /**
         * The icon's outermost node, which acts as a container for the actual
         * svg node. If container is not supplied, it will be created
         */
        container?: HTMLElement;
        /**
         * Optional text label that will be added as a sibling to the icon's
         * svg node
         */
        label?: string;
        /**
         * HTML element tag used to create the icon's outermost container node,
         * if no container is passed in
         */
        tag?: 'div' | 'span';
        /**
         * Optional title that will be set on the icon's outermost container node
         */
        title?: string;
    }
    interface IResolverProps {
        icon?: IMaybeResolvable;
        iconClass?: string;
        fallback?: LabIcon;
    }
    /** ******
     * types *
     *********/
    /**
     * A type that can be resolved to a LabIcon instance.
     */
    type IResolvable = IRenderMime.LabIcon.IResolvable;
    /**
     * A type that maybe can be resolved to a LabIcon instance.
     */
    type IMaybeResolvable = IResolvable | VirtualElement.IRenderer | undefined;
    /**
     * The type of the svg node ref that can be passed into icon React components
     */
    type IReactRef = React.RefObject<SVGElement>;
    /**
     * The properties that can be passed into the React component stored in
     * the .react field of a LabIcon.
     */
    type IReactProps = IProps & React.RefAttributes<SVGElement>;
    /**
     * The complete type of the React component stored in the .react
     * field of a LabIcon.
     */
    type IReact = React.ForwardRefExoticComponent<IReactProps>;
}
export declare const badIcon: LabIcon;
export declare const blankIcon: LabIcon;
