/*!
 * ----------------------
 * Google Charts Adapter
 * ----------------------
 */

/*

  TODO:

  [ ] Build a more robust DataTable transformer
  [ ] ^Expose date parser for google charts tooltips (#70)
  [ ] ^Allow custom tooltips (#147)

*/

var Dataviz = require("../dataviz"),
    each = require("../../core/utils/each"),
    extend = require("../../core/utils/extend"),
    Keen = require("../../core");

module.exports = function(){

  Keen.loaded = false;

  var errorMapping = {
    "Data column(s) for axis #0 cannot be of type string": "No results to visualize"
  };

  var chartTypes = ['AreaChart', 'BarChart', 'ColumnChart', 'LineChart', 'PieChart', 'Table'];
  var chartMap = {};

  var dataTypes = {
    // dataType           // chartTypes (namespace)
    // 'singular':        null,
    'categorical':        ['piechart', 'barchart', 'columnchart', 'table'],
    'cat-interval':       ['columnchart', 'barchart', 'table'],
    'cat-ordinal':        ['barchart', 'columnchart', 'areachart', 'linechart', 'table'],
    'chronological':      ['areachart', 'linechart', 'table'],
    'cat-chronological':  ['linechart', 'columnchart', 'barchart', 'areachart'],
    'nominal':            ['table'],
    'extraction':         ['table']
  };

  // Create chart types
  // -------------------------------
  each(chartTypes, function (type) {
    var name = type.toLowerCase();
    chartMap[name] = {
      initialize: function(){
        // Nothing to do here
      },
      render: function(){
        if(typeof google === "undefined") {
          this.error("The Google Charts library could not be loaded.");
          return;
        }
        var self = this;
        if (self.view._artifacts['googlechart']) {
          this.destroy();
        }
        self.view._artifacts['googlechart'] = self.view._artifacts['googlechart'] || new google.visualization[type](self.el());
        google.visualization.events.addListener(self.view._artifacts['googlechart'], 'error', function(stack){
          _handleErrors.call(self, stack);
        });
        this.update();
      },
      update: function(){
        var options = _getDefaultAttributes.call(this, type);
        extend(options, this.chartOptions(), this.attributes());

        // Apply stacking if set by top-level option
        options['isStacked'] = (this.stacked() || options['isStacked']);

        this.view._artifacts['datatable'] = google.visualization.arrayToDataTable(this.data());

        // Apply date formatting (also applied to hAxis/vAxis as defaults)
        if (options.dateFormat) {
          if (typeof options.dateFormat === 'function') {
            options.dateFormat(this.view._artifacts['datatable']);
          }
          else if (typeof options.dateFormat === 'string') {
            new google.visualization.DateFormat({
              pattern: options.dateFormat
            }).format(this.view._artifacts['datatable'], 0);
          }
        }

        if (this.view._artifacts['googlechart']) {
          this.view._artifacts['googlechart'].draw(this.view._artifacts['datatable'], options);
        }
      },
      destroy: function(){
        if (this.view._artifacts['googlechart']) {
          google.visualization.events.removeAllListeners(this.view._artifacts['googlechart']);
          this.view._artifacts['googlechart'].clearChart();
          this.view._artifacts['googlechart'] = null;
          this.view._artifacts['datatable'] = null;
        }
      }
    };
  });


  // Register library + types
  // -------------------------------

  Dataviz.register('google', chartMap, {
    capabilities: dataTypes,
    dependencies: [{
      type: 'script',
      url: 'https://www.google.com/jsapi',
      cb: function(done) {
        if (typeof google === 'undefined'){
          this.trigger("error", "Problem loading Google Charts library. Please contact us!");
          done();
        }
        else {
          google.load('visualization', '1.1', {
              packages: ['corechart', 'table'],
              callback: function(){
                done();
              }
          });
        }
      }
    }]
  });

  function _handleErrors(stack){
    var message = errorMapping[stack['message']] || stack['message'] || 'An error occurred';
    this.error(message);
  }

  function _getDefaultAttributes(type){
    var output = {};
    switch (type.toLowerCase()) {

      case "areachart":
        output.lineWidth = 2;
        output.hAxis = {
          baselineColor: 'transparent',
          gridlines: { color: 'transparent' }
        };
        output.vAxis = {
          viewWindow: { min: 0 }
        };
        if (this.dataType() === "chronological" || this.dataType() === "cat-ordinal") {
          output.legend = "none";
          output.chartArea = {
            width: "85%"
          };
        }
        if (this.dateFormat() && typeof this.dateFormat() === 'string') {
          output.hAxis.format = this.dateFormat();
        }
        break;

      case "barchart":
        output.hAxis = {
          viewWindow: { min: 0 }
        };
        output.vAxis = {
          baselineColor: 'transparent',
          gridlines: { color: 'transparent' }
        };
        if (this.dataType() === "chronological" || this.dataType() === "cat-ordinal") {
          output.legend = "none";
        }
        if (this.dateFormat() && typeof this.dateFormat() === 'string') {
          output.vAxis.format = this.dateFormat();
        }
        break;

      case "columnchart":
        output.hAxis = {
          baselineColor: 'transparent',
          gridlines: { color: 'transparent' }
        };
        output.vAxis = {
          viewWindow: { min: 0 }
        };
        if (this.dataType() === "chronological" || this.dataType() === "cat-ordinal") {
          output.legend = "none";
          output.chartArea = {
            width: "85%"
          };
        }
        if (this.dateFormat() && typeof this.dateFormat() === 'string') {
          output.hAxis.format = this.dateFormat();
        }
        break;

      case "linechart":
        output.lineWidth = 2;
        output.hAxis = {
          baselineColor: 'transparent',
          gridlines: { color: 'transparent' }
        };
        output.vAxis = {
          viewWindow: { min: 0 }
        };
        if (this.dataType() === "chronological" || this.dataType() === "cat-ordinal") {
          output.legend = "none";
          output.chartArea = {
            width: "85%"
          };
        }
        if (this.dateFormat() && typeof this.dateFormat() === 'string') {
          output.hAxis.format = this.dateFormat();
        }
        break;

      case "piechart":
        output.sliceVisibilityThreshold = 0.01;
        break;

      case "table":
        break;
    }
    return output;
  }

};
