'use strict';

var lezerFeel = require('lezer-feel');
var language = require('@codemirror/language');
var autocomplete = require('@codemirror/autocomplete');

// / A collection of FEEL-related
// / [snippets](#autocomplete.snippet).
const snippets = [
    autocomplete.snippetCompletion('function(${params}) ${body}', {
        label: 'function',
        detail: 'definition',
        type: 'keyword'
    }),
    autocomplete.snippetCompletion('for ${var} in ${collection} return ${value}', {
        label: 'for',
        detail: 'expression',
        type: 'keyword'
    }),
    autocomplete.snippetCompletion('every ${var} in ${collection} satisfies ${condition}', {
        label: 'every',
        detail: 'quantified expression',
        type: 'keyword'
    }),
    autocomplete.snippetCompletion('some ${var} in ${collection} satisfies ${condition}', {
        label: 'some',
        detail: 'quantified expression',
        type: 'keyword'
    }),
    autocomplete.snippetCompletion('if ${condition} then ${value} else ${other value}', {
        label: 'if',
        detail: 'block',
        type: 'keyword'
    })
];

// / A language provider based on the [Lezer FEEL
// / parser](https://github.com/nikku/lezer-feel), extended with
// / highlighting and indentation information.
const feelLanguage = language.LRLanguage.define({
    parser: lezerFeel.parser.configure({
        props: [
            language.indentNodeProp.add({
                'Context': language.delimitedIndent({
                    closing: '}'
                }),
                'List FilterExpression': language.delimitedIndent({
                    closing: ']'
                }),
                'ParenthesizedExpression FunctionInvocation': language.continuedIndent({
                    except: /^\s*\)/
                }),
                'ForExpression QuantifiedExpression IfExpression': language.continuedIndent({
                    except: /^\s*(then|else|return|satisfies)\b/
                }),
                'FunctionDefinition': language.continuedIndent({
                    except: /^\s*(\(|\))/
                })
            }),
            language.foldNodeProp.add({
                Context: language.foldInside,
                List: language.foldInside,
                ParenthesizedExpression: language.foldInside,
                FunctionDefinition(node) {
                    const last = node.getChild(')');
                    if (!last)
                        return null;
                    return {
                        from: last.to,
                        to: node.to
                    };
                }
            })
        ]
    }),
    languageData: {
        indentOnInput: /^\s*(\)|\}|\]|then|else|return|satisfies)$/,
        commentTokens: {
            line: '//',
            block: {
                open: '/*',
                close: '*/'
            }
        }
    }
});
// / A language provider for TypeScript.
const unaryTestsLanguage = feelLanguage.configure({ top: 'UnaryTests' });
// / Language provider for JSX.
const expressionLanguage = feelLanguage.configure({ top: 'Expression' });
const keywords = 'return satisfies then in'.split(' ').map(kw => ({ label: kw, type: 'keyword' }));
const dontComplete = [
    'StringLiteral', 'Name',
    'LineComment', 'BlockComment'
];
// / FEEL support. Includes [snippet](#lang-feel.snippets)
// / completion.
function feel(config = {}) {
    const lang = config.dialect === 'unaryTests' ? unaryTestsLanguage : expressionLanguage;
    const contextualLang = lang.configure({
        contextTracker: lezerFeel.trackVariables(config.context)
    });
    return new language.LanguageSupport(contextualLang, [
        feelLanguage.data.of({
            autocomplete: autocomplete.ifNotIn(dontComplete, autocomplete.completeFromList(snippets.concat(keywords)))
        })
    ]);
}

exports.expressionLanguage = expressionLanguage;
exports.feel = feel;
exports.feelLanguage = feelLanguage;
exports.snippets = snippets;
exports.unaryTestsLanguage = unaryTestsLanguage;
